# AGENTS.md - NeonFS Development Guide

## Project Overview

NeonFS is a single-file, zero-dependency PHP file manager with a cyberpunk theme. The entire application lives in `index.php` (~4000 lines). No Composer, no external dependencies, no build system.

**NeonTerm Extension**: Integrated multi-tab SSH client with VT100 terminal emulator using proc_open and sshpass for SSH connections.

## Build, Lint, and Test Commands

### PHP Syntax Check
```bash
/lamp/php/bin/php -l index.php
```

### Run PHP Locally (CLI)
```bash
/lamp/php/bin/php index.php
```

### Run PHP via Web Server
Access through web browser at `/file_manager/` or use curl:
```bash
curl -s http://localhost/file_manager/
```

### Single Test Execution
There is no automated test suite. Test new features manually:
1. Create a test directory with sample files
2. Open `index.php` in browser with `?current_folder=/path/to/test`
3. Verify all operations (upload, delete, rename, etc.)
4. Check browser console for JavaScript errors

## Code Style Guidelines

### General Principles
- Keep `index.php` self-contained - no external files
- Zero dependencies - pure vanilla PHP and JavaScript
- Security-first approach - validate all inputs, block directory traversal
- Responsive design with CSS variables for theming

### PHP Conventions

#### Constants
```php
define('NEONFS_PREFIX_NAME', 'value');  // UPPERCASE with underscores
```

#### Functions
```php
function function_name($param1, $param2 = default) {
    // snake_case for all function names
}
```

#### Return Values
Use associative arrays for all function returns:
```php
return [
    'success' => true,
    'data' => $result,
    'message' => 'Operation completed'
];
```

#### Arrays
```php
$array = [
    'key1' => 'value1',
    'key2' => 'value2',
];
```

#### Error Handling
```php
if (!@some_operation()) {
    return ['success' => false, 'message' => 'Human-readable error'];
}
```
- Use `@` operator to suppress warnings when checking result
- Always return `['success' => false, 'message' => '...']` on failure
- Never expose sensitive system information in error messages

### Security Requirements

#### Path Validation (Critical)
Every file operation MUST call `isPathSafe()` first:
```php
if (!isPathSafe($path)) {
    return ['success' => false, 'message' => 'Access denied'];
}
```

#### Blacklist Checking
The `NEONFS_BLACKLIST` array contains forbidden paths. Never allow access to:
- `/etc/shadow`, `/etc/passwd`
- `/root/.ssh`
- Other sensitive system directories

#### Authentication
- Use `NEONFS_REQUIRE_AUTH` to enable password protection
- Store authenticated state in `$_SESSION['neonfs_authenticated']`
- Always verify session before allowing operations

### JavaScript Conventions

#### Naming
```javascript
// camelCase for variables and functions
let fileList = [];
function handleUpload() { }
```

#### Event Handling
```javascript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize after DOM ready
});
```

#### AJAX/Fetch
```javascript
fetch('index.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: new URLSearchParams({ action: 'upload', path: currentPath })
})
```

### CSS Guidelines

#### CSS Variables
Define all colors and spacing at `:root` level:
```css
:root {
    --bg-primary: #0a0e27;
    --accent-cyan: #00eaff;
    --accent-magenta: #ff00ea;
}
```

#### Naming
- Use kebab-case for class names
- Prefix cyberpunk-themed classes appropriately

### File Structure (index.php)

1. **Header comment** - Version, features, philosophy
2. **Configuration section** - All `define()` calls
3. **Session setup**
4. **Authentication handling**
5. **Utility functions** (getFileIcon, formatSize, timeAgo)
6. **Core functions** (scanDirectory, deleteItem, copyItem)
7. **Request handling** - POST/GET logic
8. **HTML output** - Complete page structure
9. **Inline CSS** - Cyberpunk theme
10. **Inline JavaScript** - All client-side logic

### Naming Conventions

| Element | Style | Example |
|---------|-------|---------|
| Constants | UPPER_CASE_WITH_UNDERSCORES | `NEONFS_ROOT_PATH` |
| Functions | snake_case | `scan_directory()` |
| Variables | snake_case | `$current_path` |
| JavaScript vars | camelCase | `selectedItems` |
| CSS classes | kebab-case | `.cyber-button` |
| Session keys | neonfs_prefixed | `$_SESSION['neonfs_clipboard']` |

### Session Management

```php
// Initialize session with defaults
if (!isset($_SESSION['neonfs_clipboard'])) {
    $_SESSION['neonfs_clipboard'] = [];
}
```

### File Operations

- Always check `isPathSafe()` before any file operation
- Use `@` to suppress errors from filesystem functions
- Return consistent response format: `['success' => bool, 'message' => string, ...]`
- For directory operations, handle `.` and `..` explicitly

### HTML/JavaScript Integration

- No external JavaScript libraries
- CSS and JS embedded in HTML (no separate files)
- Use `data-*` attributes for dynamic behavior
- Event delegation for dynamic elements

### Testing New Features

1. Test on both CLI and web:
   ```bash
   /lamp/php/bin/php index.php  # CLI
   curl -s http://localhost/file_manager/  # Web
   ```

2. Test path traversal attempts are blocked
3. Test file operations on various file types
4. Test with SAFE_MODE enabled/disabled
5. Test with authentication enabled/disabled

### Deployment

- File permissions: `chmod 755` for directory, `chmod 644` for `index.php`
- No server restart required after changes
- Clear browser cache after updates (Ctrl+F5)
- See DEPLOYMENT.md for full details

## NeonTerm (SSH Terminal) Extension

### Architecture
- Uses `proc_open` with `sshpass` for SSH connections (fallback: expect scripts if sshpass unavailable)
- Session state stored in PHP `$_SESSION` with encrypted credentials
- Output buffered via temporary files in `/tmp/neonfs_ssh/`

### API Endpoints
| Action | Purpose |
|--------|---------|
| `ssh_connect` | Establish SSH connection |
| `ssh_command` | Execute command in shell |
| `ssh_read` | Poll for terminal output |
| `ssh_disconnect` | Close session |
| `ssh_list` | List saved connections |
| `exec_local` | Local shell commands |

### Security
- Credentials encrypted with AES-256-CBC using `NEONFS_PASSWORD` as key
- Host validation prevents localhost connections in safe mode
- Session isolation per user

### Frontend
- `NeonTerm` class manages tabs and sessions
- `AnsiParser` converts VT100 codes to HTML spans
- Supports: colors (30-37, 90-97), bold (1), italic (3), underline (4), background (40-47)

### Keyboard Shortcuts
| Key | Action |
|-----|--------|
| `Ctrl+`` | Toggle terminal panel |
| `Enter` | Send command |
| `Up/Down` | Command history |
| `Tab` | Auto-complete |
| `Ctrl+C` | Send interrupt |
