<?php
/**
 * 🌌 NeonFS v1.1 - Cyberpunk File Manager LEGENDARY EDITION
 *
 * "Where filesystem mastery meets neon glory"
 *
 * NOW WITH:
 * • Batch operations with multi-select
 * • Drag & drop upload with progress bars
 * • Copy/paste clipboard
 * • Right-click context menu
 * • File previews (images, text, code)
 * • Chmod/chown editor
 * • Full keyboard shortcuts
 * • File sorting (name, size, date, type)
 * • Toast notifications
 * • Path copying
 * • And more cyberpunk magic...
 *
 * Zero dependencies • Pure vanilla • Single file • LEGENDARY
 *
 * The blueprint was never lost. It was waiting... for you.
 */

// ═══════════════════════════════════════════════════════════════════════════
// 🔮 SECTION 1: CONFIGURATION - The Foundation
// ═══════════════════════════════════════════════════════════════════════════

define('NEONFS_VERSION', '1.1.0');
define('NEONFS_REQUIRE_AUTH', false);           // Set true to require password
define('NEONFS_PASSWORD', 'neon2026');          // Change this if auth enabled
define('NEONFS_ROOT_PATH', '/');                // Base path - can restrict to specific dir
define('NEONFS_MAX_UPLOAD_SIZE', 100);          // MB
define('NEONFS_SAFE_MODE', false);              // Disable delete/chmod operations
define('NEONFS_BLACKLIST', [                    // Forbidden paths
    '/etc/shadow',
    '/etc/passwd',
    '/root/.ssh'
]);

// Session setup
session_start();

// Initialize clipboard in session
if (!isset($_SESSION['neonfs_clipboard'])) {
    $_SESSION['neonfs_clipboard'] = [];
}

// Initialize SSH sessions
if (!isset($_SESSION['neonfs_ssh_sessions'])) {
    $_SESSION['neonfs_ssh_sessions'] = [];
}

// Initialize saved connections (encrypted)
if (!isset($_SESSION['neonfs_ssh_connections'])) {
    $_SESSION['neonfs_ssh_connections'] = [];
}

// Initialize authentication check
if (NEONFS_REQUIRE_AUTH && !isset($_SESSION['neonfs_authenticated'])) {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['password'])) {
        if ($_POST['password'] === NEONFS_PASSWORD) {
            $_SESSION['neonfs_authenticated'] = true;
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        } else {
            $authError = 'Invalid password';
        }
    }

    // Show login screen
    if (!isset($_SESSION['neonfs_authenticated'])) {
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title>🌌 NeonFS - Authentication</title>
            <style>
                body {
                    margin: 0;
                    padding: 0;
                    background: linear-gradient(135deg, #0a0e27 0%, #1a1f3a 100%);
                    font-family: 'Courier New', monospace;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    min-height: 100vh;
                    color: #e0e0ff;
                }
                .login-box {
                    background: rgba(26, 31, 58, 0.95);
                    border: 2px solid #00eaff;
                    border-radius: 8px;
                    padding: 40px;
                    box-shadow: 0 0 30px rgba(0, 234, 255, 0.5);
                    text-align: center;
                    min-width: 300px;
                }
                .login-box h1 {
                    margin: 0 0 30px 0;
                    font-size: 32px;
                    text-shadow: 0 0 20px rgba(0, 234, 255, 0.8);
                }
                .login-box input {
                    width: 100%;
                    padding: 12px;
                    margin: 10px 0;
                    background: rgba(10, 14, 39, 0.8);
                    border: 1px solid #00eaff;
                    border-radius: 4px;
                    color: #e0e0ff;
                    font-family: 'Courier New', monospace;
                    font-size: 14px;
                    box-sizing: border-box;
                }
                .login-box button {
                    width: 100%;
                    padding: 12px;
                    margin: 20px 0 0 0;
                    background: linear-gradient(135deg, #00eaff, #00a8cc);
                    border: none;
                    border-radius: 4px;
                    color: #0a0e27;
                    font-family: 'Courier New', monospace;
                    font-size: 14px;
                    font-weight: bold;
                    cursor: pointer;
                    transition: all 0.3s;
                }
                .login-box button:hover {
                    box-shadow: 0 0 20px rgba(0, 234, 255, 0.8);
                    transform: translateY(-2px);
                }
                .error {
                    color: #ff00ea;
                    margin: 10px 0;
                    font-size: 12px;
                }
            </style>
        </head>
        <body>
            <div class="login-box">
                <h1>🌌 NeonFS</h1>
                <form method="POST">
                    <input type="password" name="password" placeholder="Enter password..." autofocus>
                    <?php if (isset($authError)): ?>
                        <div class="error"><?= htmlspecialchars($authError) ?></div>
                    <?php endif; ?>
                    <button type="submit">ENTER THE GRID</button>
                </form>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// 🛡️ SECTION 2: SECURITY - The Guardian
// ═══════════════════════════════════════════════════════════════════════════

function isPathSafe($path) {
    $realPath = @realpath($path);
    if ($realPath === false) {
        $realPath = $path;
    }

    $rootPath = realpath(NEONFS_ROOT_PATH);

    if (strpos($realPath, $rootPath) !== 0) {
        return false;
    }

    foreach (NEONFS_BLACKLIST as $blocked) {
        if (strpos($realPath, $blocked) === 0) {
            return false;
        }
    }

    return true;
}

function sanitizeFilename($filename) {
    $filename = str_replace(['/', '\\'], '', $filename);
    $filename = str_replace("\0", '', $filename);
    $filename = ltrim($filename, '.');
    return $filename;
}

function checkSafeMode($operation) {
    if (NEONFS_SAFE_MODE && in_array($operation, ['delete', 'chmod'])) {
        return ['success' => false, 'message' => 'Operation disabled in safe mode'];
    }
    return null;
}

// ═══════════════════════════════════════════════════════════════════════════
// 🔌 SECTION 2.5: SSH/TERMINAL OPERATIONS - NeonTerm Core
// ═══════════════════════════════════════════════════════════════════════════

define('NEONFS_SSH_BUFFER_DIR', sys_get_temp_dir() . '/neonfs_ssh');
if (!is_dir(NEONFS_SSH_BUFFER_DIR)) {
    @mkdir(NEONFS_SSH_BUFFER_DIR, 0755, true);
}

function ssh_encrypt_data($data, $key) {
    $iv = random_bytes(16);
    $encrypted = openssl_encrypt(json_encode($data), 'aes-256-cbc', $key, 0, $iv);
    return base64_encode($iv . $encrypted);
}

function ssh_decrypt_data($data, $key) {
    if (empty($data)) return null;
    try {
        $raw = base64_decode($data, true);
        if ($raw === false) return null;
        $iv = substr($raw, 0, 16);
        $encrypted = substr($raw, 16);
        $decrypted = openssl_decrypt($encrypted, 'aes-256-cbc', $key, 0, $iv);
        return json_decode($decrypted, true);
    } catch (Exception $e) {
        return null;
    }
}

function ssh_validate_host($host) {
    if (NEONFS_SAFE_MODE) {
        $blocked = ['localhost', '127.0.0.1', '::1', '0.0.0.0'];
        if (in_array(strtolower($host), $blocked)) {
            return ['success' => false, 'message' => 'Local connections disabled in safe mode'];
        }
    }
    if (!filter_var($host, FILTER_VALIDATE_DOMAIN) && !filter_var($host, FILTER_VALIDATE_IP)) {
        return ['success' => false, 'message' => 'Invalid host format'];
    }
    return ['success' => true];
}

function ssh_create_connection($host, $port, $username, $password, $connectionName) {
    $validation = ssh_validate_host($host);
    if (!$validation['success']) {
        return $validation;
    }

    $port = intval($port);
    if ($port < 1 || $port > 65535) {
        return ['success' => false, 'message' => 'Invalid port number'];
    }

    $sessionId = bin2hex(random_bytes(16));
    $stdinFile = NEONFS_SSH_BUFFER_DIR . "/sess_{$sessionId}_stdin";
    $stdoutFile = NEONFS_SSH_BUFFER_DIR . "/sess_{$sessionId}_stdout";
    $stderrFile = NEONFS_SSH_BUFFER_DIR . "/sess_{$sessionId}_stderr";
    $metaFile = NEONFS_SSH_BUFFER_DIR . "/sess_{$sessionId}_meta";

    $cmd = sprintf(
        'sshpass -p %s ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o LogLevel=ERROR ' .
        '-p %d %s@%s "export TERM=xterm-256color; exec bash" 2>&1',
        escapeshellarg($password),
        $port,
        escapeshellarg($username),
        escapeshellarg($host)
    );

    $descriptors = [
        0 => ['file', $stdinFile, 'w'],
        1 => ['file', $stdoutFile, 'w'],
        2 => ['file', $stderrFile, 'w'],
    ];

    $env = ['TERM' => 'xterm-256color', 'COLUMNS' => '80', 'LINES' => '24'];

    $process = @proc_open($cmd, $descriptors, $pipes, null, $env);

    if (!is_resource($process)) {
        @unlink($stdinFile);
        @unlink($stdoutFile);
        @unlink($stderrFile);
        return ['success' => false, 'message' => 'Failed to start SSH process. Is sshpass installed?'];
    }

    stream_set_blocking($pipes[0], false);
    stream_set_blocking($pipes[1], false);
    stream_set_blocking($pipes[2], false);

    $meta = [
        'host' => $host,
        'port' => $port,
        'username' => $username,
        'name' => $connectionName ?: $host,
        'created' => time(),
        'cwd' => '~',
        'stdin' => $stdinFile,
        'stdout' => $stdoutFile,
        'stderr' => $stderrFile,
    ];
    file_put_contents($metaFile, json_encode($meta));

    $sessionData = [
        'process' => $process,
        'pipes' => $pipes,
        'host' => $host,
        'port' => $port,
        'username' => $username,
        'name' => $connectionName ?: $host,
        'created' => time(),
        'cwd' => '~',
        'stdin' => $stdinFile,
        'stdout' => $stdoutFile,
        'stderr' => $stderrFile,
        'meta' => $metaFile,
    ];

    $_SESSION['neonfs_ssh_sessions'][$sessionId] = $sessionData;

    $sessionKey = NEONFS_PASSWORD;
    $encryptedConnections = $_SESSION['neonfs_ssh_connections'] ?? [];
    $encryptedConnections[$sessionId] = ssh_encrypt_data([
        'host' => $host,
        'port' => $port,
        'username' => $username,
        'password' => $password,
        'name' => $connectionName ?: $host,
    ], $sessionKey);
    $_SESSION['neonfs_ssh_connections'] = $encryptedConnections;

    return [
        'success' => true,
        'sessionId' => $sessionId,
        'name' => $sessionData['name'],
        'host' => $host,
        'username' => $username,
    ];
}

function ssh_execute_command($sessionId, $command) {
    $sessions = $_SESSION['neonfs_ssh_sessions'] ?? [];

    if (!isset($sessions[$sessionId])) {
        return ['success' => false, 'message' => 'Session not found'];
    }

    $sess = $sessions[$sessionId];
    $stdin = $sess['stdin'];

    $command = rtrim($command, "\r\n") . "\n";
    $written = @file_put_contents($stdin, $command, FILE_APPEND);

    if ($written === false) {
        return ['success' => false, 'message' => 'Failed to write to stdin'];
    }

    usleep(75000);

    $output = '';
    if (file_exists($sess['stdout'])) {
        $output = @file_get_contents($sess['stdout']);
    }

    if ($output === false) {
        $output = '';
    }

    $processStatus = proc_get_status($sess['process']);
    if (!$processStatus['running']) {
        $stderr = file_exists($sess['stderr']) ? @file_get_contents($sess['stderr']) : '';
        return [
            'success' => false,
            'message' => 'Process died: ' . trim($stderr),
            'dead' => true,
        ];
    }

    if (preg_match('/^\s*cd\s+(["\']?)([^"\']+)\1?\s*$/', $command, $matches)) {
        $newCwd = trim($matches[2]);
        if (strpos($newCwd, '/') === 0 || strpos($newCwd, '~') === 0) {
            $sess['cwd'] = $newCwd;
            $sess['meta'] = $sess['meta'] ?? '';
            if ($sess['meta'] && file_exists($sess['meta'])) {
                $meta = json_decode(file_get_contents($sess['meta']), true);
                $meta['cwd'] = $newCwd;
                file_put_contents($sess['meta'], json_encode($meta));
            }
            $_SESSION['neonfs_ssh_sessions'][$sessionId] = $sess;
        }
    } elseif (preg_match('/^cd\s+(\.\.)\s*$/', $command, $matches)) {
        $cwd = $sess['cwd'];
        if ($cwd !== '~' && $cwd !== '/') {
            $parent = dirname($cwd);
            if ($parent === $cwd || $parent === '.' || $parent === '/') {
                $sess['cwd'] = '/';
            } else {
                $sess['cwd'] = $parent;
            }
            if ($sess['meta'] && file_exists($sess['meta'])) {
                $meta = json_decode(file_get_contents($sess['meta']), true);
                $meta['cwd'] = $sess['cwd'];
                file_put_contents($sess['meta'], json_encode($meta));
            }
            $_SESSION['neonfs_ssh_sessions'][$sessionId] = $sess;
        }
    }

    return [
        'success' => true,
        'output' => $output,
        'cwd' => $sess['cwd'],
    ];
}

function ssh_read_output($sessionId) {
    $sessions = $_SESSION['neonfs_ssh_sessions'] ?? [];

    if (!isset($sessions[$sessionId])) {
        return ['success' => false, 'message' => 'Session not found'];
    }

    $sess = $sessions[$sessionId];
    $process = $sess['process'] ?? null;

    $alive = false;
    if ($process && is_resource($process)) {
        $status = proc_get_status($process);
        $alive = $status['running'];
    }

    $output = '';
    if (file_exists($sess['stdout'])) {
        $output = @file_get_contents($sess['stdout']);
    }
    if ($output === false) {
        $output = '';
    }

    if (!$alive) {
        if (file_exists($sess['stderr'])) {
            $stderr = @file_get_contents($sess['stderr']);
            if ($stderr && strlen(trim($stderr)) > 0) {
                $output .= "\r\n" . trim($stderr);
            }
        }
    }

    return [
        'success' => true,
        'output' => $output,
        'cwd' => $sess['cwd'],
        'alive' => $alive,
    ];
}

function ssh_disconnect_session($sessionId) {
    $sessions = $_SESSION['neonfs_ssh_sessions'] ?? [];

    if (isset($sessions[$sessionId])) {
        $sess = $sessions[$sessionId];

        if (isset($sess['process']) && is_resource($sess['process'])) {
            @file_put_contents($sess['stdin'], "exit\n", FILE_APPEND);
            usleep(100000);
            proc_terminate($sess['process']);
            proc_close($sess['process']);
        }

        @unlink($sess['stdin'] ?? '');
        @unlink($sess['stdout'] ?? '');
        @unlink($sess['stderr'] ?? '');
        @unlink($sess['meta'] ?? '');

        unset($_SESSION['neonfs_ssh_sessions'][$sessionId]);
        unset($_SESSION['neonfs_ssh_connections'][$sessionId]);
    }

    return ['success' => true];
}

function ssh_get_connections() {
    $connections = [];
    $sessions = $_SESSION['neonfs_ssh_sessions'] ?? [];
    $saved = $_SESSION['neonfs_ssh_connections'] ?? [];
    $key = NEONFS_PASSWORD;

    foreach ($sessions as $id => $sess) {
        $connections[] = [
            'sessionId' => $id,
            'name' => $sess['name'],
            'host' => $sess['host'],
            'port' => $sess['port'],
            'username' => $sess['username'],
            'connected' => true,
            'saved' => isset($saved[$id]),
        ];
    }

    foreach ($saved as $id => $encrypted) {
        if (!isset($sessions[$id])) {
            $decrypted = ssh_decrypt_data($encrypted, $key);
            if ($decrypted) {
                $connections[] = [
                    'sessionId' => $id,
                    'name' => $decrypted['name'],
                    'host' => $decrypted['host'],
                    'port' => $decrypted['port'],
                    'username' => $decrypted['username'],
                    'connected' => false,
                    'saved' => true,
                ];
            }
        }
    }

    return ['success' => true, 'connections' => $connections];
}

function ssh_cleanup_dead_sessions() {
    $sessions = $_SESSION['neonfs_ssh_sessions'] ?? [];
    $maxAge = 3600;
    $cleaned = 0;

    foreach ($sessions as $id => $sess) {
        $process = $sess['process'] ?? null;
        $isDead = false;

        if (!$process || !is_resource($process)) {
            $isDead = true;
        } else {
            $status = proc_get_status($process);
            if (!$status['running']) {
                $isDead = true;
            }
        }

        if ($isDead || (time() - $sess['created'] > $maxAge)) {
            ssh_disconnect_session($id);
            $cleaned++;
        }
    }

    return $cleaned;
}

ssh_cleanup_dead_sessions();

function exec_local_command($command, $cwd = '~') {
    if (NEONFS_SAFE_MODE) {
        return [
            'success' => false,
            'output' => 'Local command execution disabled in safe mode',
            'cwd' => $cwd
        ];
    }

    $allowedCommands = ['ls', 'pwd', 'whoami', 'date', 'echo', 'hostname', 'uptime', 'clear'];

    $cmd = trim($command);
    $firstWord = explode(' ', trim($cmd))[0];

    if (!in_array($firstWord, $allowedCommands)) {
        return [
            'success' => false,
            'output' => 'Command not allowed: ' . $firstWord . '\nTry: ' . implode(', ', $allowedCommands),
            'cwd' => $cwd
        ];
    }

    if (strpos($cmd, 'cd ') === 0) {
        $newCwd = trim(substr($cmd, 3));
        if (!empty($newCwd)) {
            $validCwd = $newCwd;
            if (strpos($newCwd, '/') !== 0) {
                if ($cwd === '~' || $cwd === '/') {
                    $validCwd = '/' . $newCwd;
                } else {
                    $validCwd = $cwd . '/' . $newCwd;
                }
            }
            return [
                'success' => true,
                'output' => '',
                'cwd' => $validCwd
            ];
        }
        return ['success' => true, 'output' => '', 'cwd' => $cwd];
    }

    if ($cmd === 'clear') {
        return ['success' => true, 'output' => "\033[2J\033[H", 'cwd' => $cwd];
    }

    $descriptors = [
        0 => ['pipe', 'r'],
        1 => ['pipe', 'w'],
        2 => ['pipe', 'w']
    ];

    $process = proc_open($cmd, $descriptors, $pipes, null, ['HOME' => '/tmp']);

    if (!is_resource($process)) {
        return ['success' => false, 'output' => 'Failed to execute command', 'cwd' => $cwd];
    }

    fclose($pipes[0]);

    $output = stream_get_contents($pipes[1]);
    fclose($pipes[1]);

    $error = stream_get_contents($pipes[2]);
    fclose($pipes[2]);

    proc_close($process);

    return [
        'success' => true,
        'output' => $output . $error,
        'cwd' => $cwd
    ];
}

// ═══════════════════════════════════════════════════════════════════════════
// 🔧 SECTION 3: FILE OPERATIONS - The Manipulator
// ═══════════════════════════════════════════════════════════════════════════

function getFileIcon($filename, $isDir) {
    if ($isDir) return '📁';

    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

    $iconMap = [
        'jpg' => '🖼️', 'jpeg' => '🖼️', 'png' => '🖼️', 'gif' => '🖼️', 'svg' => '🖼️', 'webp' => '🖼️', 'bmp' => '🖼️',
        'php' => '💻', 'js' => '💻', 'py' => '💻', 'java' => '💻', 'cpp' => '💻', 'c' => '💻', 'h' => '💻',
        'html' => '💻', 'css' => '💻', 'ts' => '💻', 'jsx' => '💻', 'vue' => '💻', 'go' => '💻', 'rs' => '💻',
        'txt' => '📄', 'md' => '📄', 'log' => '📄',
        'pdf' => '📝', 'doc' => '📝', 'docx' => '📝', 'odt' => '📝',
        'zip' => '📦', 'tar' => '📦', 'gz' => '📦', 'rar' => '📦', '7z' => '📦', 'bz2' => '📦',
        'mp4' => '🎬', 'avi' => '🎬', 'mkv' => '🎬', 'mov' => '🎬', 'wmv' => '🎬', 'flv' => '🎬',
        'mp3' => '🎵', 'wav' => '🎵', 'flac' => '🎵', 'ogg' => '🎵', 'm4a' => '🎵',
        'db' => '🗄️', 'sql' => '🗄️', 'sqlite' => '🗄️', 'mdb' => '🗄️',
        'conf' => '⚙️', 'ini' => '⚙️', 'yaml' => '⚙️', 'yml' => '⚙️', 'json' => '⚙️', 'xml' => '⚙️',
        'sh' => '🔧', 'bash' => '🔧', 'bat' => '🔧',
        'exe' => '⚡', 'app' => '⚡', 'dmg' => '⚡',
    ];

    return $iconMap[$ext] ?? '📄';
}

function formatSize($bytes) {
    if ($bytes === 0) return '0 B';
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $factor = floor(log($bytes) / log(1024));
    return sprintf("%.2f %s", $bytes / pow(1024, $factor), $units[$factor]);
}

function timeAgo($timestamp) {
    $diff = time() - $timestamp;
    if ($diff < 60) return 'just now';
    if ($diff < 3600) return floor($diff / 60) . 'm ago';
    if ($diff < 86400) return floor($diff / 3600) . 'h ago';
    if ($diff < 604800) return floor($diff / 86400) . 'd ago';
    if ($diff < 2592000) return floor($diff / 604800) . 'w ago';
    return date('Y-m-d', $timestamp);
}

function scanDirectory($path) {
    if (!isPathSafe($path)) {
        return ['success' => false, 'message' => 'Access denied'];
    }

    if (!is_dir($path)) {
        return ['success' => false, 'message' => 'Not a directory'];
    }

    $items = [];
    $files = @scandir($path);

    if ($files === false) {
        return ['success' => false, 'message' => 'Permission denied'];
    }

    foreach ($files as $file) {
        if ($file === '.') continue;

        $fullPath = $path . DIRECTORY_SEPARATOR . $file;
        $isDir = is_dir($fullPath);
        $stat = @stat($fullPath);

        $items[] = [
            'name' => $file,
            'path' => $fullPath,
            'isDir' => $isDir,
            'icon' => getFileIcon($file, $isDir),
            'size' => $isDir ? null : ($stat['size'] ?? 0),
            'sizeFormatted' => $isDir ? '-' : formatSize($stat['size'] ?? 0),
            'modified' => $stat['mtime'] ?? 0,
            'modifiedFormatted' => timeAgo($stat['mtime'] ?? 0),
            'permissions' => substr(sprintf('%o', $stat['mode'] ?? 0), -4),
            'permissionsOctal' => substr(sprintf('%o', $stat['mode'] ?? 0), -3),
            'owner' => function_exists('posix_getpwuid') ? (posix_getpwuid($stat['uid'] ?? 0)['name'] ?? 'unknown') : 'unknown',
            'group' => function_exists('posix_getgrgid') ? (posix_getgrgid($stat['gid'] ?? 0)['name'] ?? 'unknown') : 'unknown',
            'isParent' => $file === '..',
            'isReadable' => is_readable($fullPath),
            'isWritable' => is_writable($fullPath),
            'isExecutable' => is_executable($fullPath),
        ];
    }

    usort($items, function($a, $b) {
        if ($a['isParent']) return -1;
        if ($b['isParent']) return 1;
        if ($a['isDir'] !== $b['isDir']) return $b['isDir'] - $a['isDir'];
        return strcasecmp($a['name'], $b['name']);
    });

    $freeSpace = @disk_free_space($path);
    $totalSpace = @disk_total_space($path);

    return [
        'success' => true,
        'items' => $items,
        'path' => $path,
        'freeSpace' => $freeSpace ? formatSize($freeSpace) : 'unknown',
        'totalSpace' => $totalSpace ? formatSize($totalSpace) : 'unknown',
        'itemCount' => count($items) - 1
    ];
}

function deleteItem($path) {
    if ($safeCheck = checkSafeMode('delete')) return $safeCheck;
    if (!isPathSafe($path)) return ['success' => false, 'message' => 'Access denied'];

    if (!file_exists($path)) {
        return ['success' => false, 'message' => 'File not found'];
    }

    if (is_dir($path)) {
        $files = @scandir($path);
        if ($files === false) {
            return ['success' => false, 'message' => 'Permission denied'];
        }

        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;
            $fullPath = $path . DIRECTORY_SEPARATOR . $file;
            $result = deleteItem($fullPath);
            if (!$result['success']) {
                return $result;
            }
        }

        if (!@rmdir($path)) {
            return ['success' => false, 'message' => 'Failed to delete directory'];
        }
    } else {
        if (!@unlink($path)) {
            return ['success' => false, 'message' => 'Failed to delete file'];
        }
    }

    return ['success' => true];
}

function copyItem($source, $destination) {
    if (!isPathSafe($source) || !isPathSafe($destination)) {
        return ['success' => false, 'message' => 'Access denied'];
    }

    if (!file_exists($source)) {
        return ['success' => false, 'message' => 'Source not found'];
    }

    if (is_dir($source)) {
        if (!@mkdir($destination, 0755, true)) {
            return ['success' => false, 'message' => 'Failed to create directory'];
        }

        $files = @scandir($source);
        if ($files === false) {
            return ['success' => false, 'message' => 'Permission denied'];
        }

        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;
            $srcPath = $source . DIRECTORY_SEPARATOR . $file;
            $dstPath = $destination . DIRECTORY_SEPARATOR . $file;
            $result = copyItem($srcPath, $dstPath);
            if (!$result['success']) {
                return $result;
            }
        }
    } else {
        if (!@copy($source, $destination)) {
            return ['success' => false, 'message' => 'Failed to copy file'];
        }
    }

    return ['success' => true];
}

function getFilePreview($path) {
    if (!isPathSafe($path) || !is_file($path)) {
        return ['success' => false, 'message' => 'Access denied'];
    }

    $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
    $size = filesize($path);

    // Image preview
    if (in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp'])) {
        $data = base64_encode(file_get_contents($path));
        $mime = mime_content_type($path);
        return [
            'success' => true,
            'type' => 'image',
            'data' => "data:$mime;base64,$data"
        ];
    }

    // Text/Code preview
    if (in_array($ext, ['txt', 'md', 'log', 'php', 'js', 'html', 'css', 'json', 'xml', 'yaml', 'yml', 'py', 'java', 'cpp', 'c', 'h', 'sh', 'conf', 'ini']) && $size < 1048576) {
        $content = file_get_contents($path);
        return [
            'success' => true,
            'type' => 'text',
            'extension' => $ext,
            'content' => $content,
            'lines' => substr_count($content, "\n") + 1
        ];
    }

    return ['success' => false, 'message' => 'Preview not available for this file type'];
}

// ═══════════════════════════════════════════════════════════════════════════
// 🌐 SECTION 4: API ENDPOINTS - The Gateway
// ═══════════════════════════════════════════════════════════════════════════

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action'])) {
    header('Content-Type: application/json');

    $action = $_GET['action'];
    $response = ['success' => false, 'message' => 'Unknown action'];

    try {
        switch ($action) {
            case 'list':
                $path = $_POST['path'] ?? NEONFS_ROOT_PATH;
                $response = scanDirectory($path);
                break;

            case 'rename':
                $path = $_POST['path'] ?? '';
                $newName = sanitizeFilename($_POST['newName'] ?? '');

                if (!isPathSafe($path)) {
                    $response = ['success' => false, 'message' => 'Access denied'];
                } elseif (empty($newName)) {
                    $response = ['success' => false, 'message' => 'Invalid name'];
                } else {
                    $dir = dirname($path);
                    $newPath = $dir . DIRECTORY_SEPARATOR . $newName;

                    if (file_exists($newPath)) {
                        $response = ['success' => false, 'message' => 'File already exists'];
                    } elseif (@rename($path, $newPath)) {
                        $response = ['success' => true, 'message' => 'Renamed successfully'];
                    } else {
                        $response = ['success' => false, 'message' => 'Rename failed'];
                    }
                }
                break;

            case 'delete':
                $paths = json_decode($_POST['paths'] ?? '[]', true);
                $deleted = 0;

                foreach ($paths as $path) {
                    $result = deleteItem($path);
                    if ($result['success']) {
                        $deleted++;
                    } else {
                        $response = $result;
                        break;
                    }
                }

                if ($deleted > 0) {
                    $response = ['success' => true, 'deleted' => $deleted];
                }
                break;

            case 'copy':
                $paths = json_decode($_POST['paths'] ?? '[]', true);
                $destination = $_POST['destination'] ?? '';
                $copied = 0;

                foreach ($paths as $path) {
                    $filename = basename($path);
                    $destPath = $destination . DIRECTORY_SEPARATOR . $filename;

                    $counter = 1;
                    $origDestPath = $destPath;
                    while (file_exists($destPath)) {
                        $info = pathinfo($origDestPath);
                        $destPath = $info['dirname'] . DIRECTORY_SEPARATOR .
                                   $info['filename'] . " ($counter)" .
                                   (isset($info['extension']) ? '.' . $info['extension'] : '');
                        $counter++;
                    }

                    $result = copyItem($path, $destPath);
                    if ($result['success']) {
                        $copied++;
                    } else {
                        $response = $result;
                        break;
                    }
                }

                if ($copied > 0) {
                    $response = ['success' => true, 'copied' => $copied];
                }
                break;

            case 'clipboard':
                $operation = $_POST['operation'] ?? '';

                if ($operation === 'copy') {
                    $_SESSION['neonfs_clipboard'] = [
                        'paths' => json_decode($_POST['paths'] ?? '[]', true),
                        'operation' => 'copy'
                    ];
                    $response = ['success' => true, 'message' => 'Copied to clipboard'];
                } elseif ($operation === 'cut') {
                    $_SESSION['neonfs_clipboard'] = [
                        'paths' => json_decode($_POST['paths'] ?? '[]', true),
                        'operation' => 'cut'
                    ];
                    $response = ['success' => true, 'message' => 'Cut to clipboard'];
                } elseif ($operation === 'paste') {
                    $destination = $_POST['destination'] ?? '';
                    $clipboard = $_SESSION['neonfs_clipboard'];

                    if (empty($clipboard['paths'])) {
                        $response = ['success' => false, 'message' => 'Clipboard is empty'];
                    } else {
                        $pasted = 0;

                        foreach ($clipboard['paths'] as $path) {
                            $filename = basename($path);
                            $destPath = $destination . DIRECTORY_SEPARATOR . $filename;

                            if ($clipboard['operation'] === 'cut') {
                                if (@rename($path, $destPath)) {
                                    $pasted++;
                                }
                            } else {
                                $counter = 1;
                                $origDestPath = $destPath;
                                while (file_exists($destPath)) {
                                    $info = pathinfo($origDestPath);
                                    $destPath = $info['dirname'] . DIRECTORY_SEPARATOR .
                                               $info['filename'] . " ($counter)" .
                                               (isset($info['extension']) ? '.' . $info['extension'] : '');
                                    $counter++;
                                }

                                $result = copyItem($path, $destPath);
                                if ($result['success']) {
                                    $pasted++;
                                }
                            }
                        }

                        if ($clipboard['operation'] === 'cut') {
                            $_SESSION['neonfs_clipboard'] = [];
                        }

                        $response = ['success' => true, 'pasted' => $pasted];
                    }
                } elseif ($operation === 'get') {
                    $response = [
                        'success' => true,
                        'clipboard' => $_SESSION['neonfs_clipboard']
                    ];
                }
                break;

            case 'newFolder':
                $path = $_POST['path'] ?? '';
                $name = sanitizeFilename($_POST['name'] ?? '');

                if (!isPathSafe($path)) {
                    $response = ['success' => false, 'message' => 'Access denied'];
                } elseif (empty($name)) {
                    $response = ['success' => false, 'message' => 'Invalid name'];
                } else {
                    $newPath = $path . DIRECTORY_SEPARATOR . $name;

                    if (file_exists($newPath)) {
                        $response = ['success' => false, 'message' => 'Folder already exists'];
                    } elseif (@mkdir($newPath, 0755)) {
                        $response = ['success' => true, 'message' => 'Folder created'];
                    } else {
                        $response = ['success' => false, 'message' => 'Failed to create folder'];
                    }
                }
                break;

            case 'upload':
                $path = $_POST['path'] ?? '';

                if (!isPathSafe($path)) {
                    $response = ['success' => false, 'message' => 'Access denied'];
                } elseif (!isset($_FILES['files'])) {
                    $response = ['success' => false, 'message' => 'No files uploaded'];
                } else {
                    $uploaded = 0;
                    $files = $_FILES['files'];

                    $fileCount = is_array($files['name']) ? count($files['name']) : 1;

                    for ($i = 0; $i < $fileCount; $i++) {
                        $filename = is_array($files['name']) ? $files['name'][$i] : $files['name'];
                        $tmpName = is_array($files['tmp_name']) ? $files['tmp_name'][$i] : $files['tmp_name'];
                        $error = is_array($files['error']) ? $files['error'][$i] : $files['error'];

                        if ($error === UPLOAD_ERR_OK) {
                            $filename = sanitizeFilename($filename);
                            $destPath = $path . DIRECTORY_SEPARATOR . $filename;

                            if (@move_uploaded_file($tmpName, $destPath)) {
                                $uploaded++;
                            }
                        }
                    }

                    $response = ['success' => true, 'uploaded' => $uploaded];
                }
                break;

            case 'chmod':
                if ($safeCheck = checkSafeMode('chmod')) {
                    $response = $safeCheck;
                    break;
                }

                $path = $_POST['path'] ?? '';
                $mode = $_POST['mode'] ?? '';

                if (!isPathSafe($path)) {
                    $response = ['success' => false, 'message' => 'Access denied'];
                } elseif (!preg_match('/^[0-7]{3}$/', $mode)) {
                    $response = ['success' => false, 'message' => 'Invalid mode'];
                } else {
                    if (@chmod($path, octdec($mode))) {
                        $response = ['success' => true, 'message' => 'Permissions changed'];
                    } else {
                        $response = ['success' => false, 'message' => 'Failed to change permissions'];
                    }
                }
                break;

            case 'preview':
                $path = $_POST['path'] ?? '';
                $response = getFilePreview($path);
                break;

            case 'download':
                $paths = json_decode($_POST['paths'] ?? '[]', true);

                if (count($paths) === 1 && is_file($paths[0])) {
                    $file = $paths[0];
                    if (isPathSafe($file) && file_exists($file)) {
                        header('Content-Type: application/octet-stream');
                        header('Content-Disposition: attachment; filename="' . basename($file) . '"');
                        header('Content-Length: ' . filesize($file));
                        readfile($file);
                        exit;
                    }
                } else {
                    $zipFile = sys_get_temp_dir() . '/neonfs_' . uniqid() . '.zip';
                    $zip = new ZipArchive();

                    if ($zip->open($zipFile, ZipArchive::CREATE) === true) {
                        foreach ($paths as $path) {
                            if (isPathSafe($path) && file_exists($path)) {
                                if (is_dir($path)) {
                                    addDirToZip($zip, $path, basename($path));
                                } else {
                                    $zip->addFile($path, basename($path));
                                }
                            }
                        }
                        $zip->close();

                        header('Content-Type: application/zip');
                        header('Content-Disposition: attachment; filename="neonfs_download.zip"');
                        header('Content-Length: ' . filesize($zipFile));
                        readfile($zipFile);
                        unlink($zipFile);
                        exit;
                    }
                }

                $response = ['success' => false, 'message' => 'Download failed'];
                break;

            case 'ssh_connect':
                if (NEONFS_REQUIRE_AUTH && !isset($_SESSION['neonfs_authenticated'])) {
                    $response = ['success' => false, 'message' => 'Authentication required'];
                    break;
                }
                $host = $_POST['host'] ?? '';
                $port = $_POST['port'] ?? '22';
                $username = $_POST['username'] ?? '';
                $password = $_POST['password'] ?? '';
                $name = $_POST['name'] ?? '';
                if (empty($host) || empty($username) || empty($password)) {
                    $response = ['success' => false, 'message' => 'Missing credentials'];
                } else {
                    $response = ssh_create_connection($host, $port, $username, $password, $name);
                }
                break;

            case 'ssh_command':
                $sessionId = $_POST['sessionId'] ?? '';
                $command = $_POST['command'] ?? '';
                if (empty($sessionId) || empty($command)) {
                    $response = ['success' => false, 'message' => 'Missing parameters'];
                } else {
                    $response = ssh_execute_command($sessionId, $command);
                }
                break;

            case 'ssh_read':
                $sessionId = $_POST['sessionId'] ?? '';
                if (empty($sessionId)) {
                    $response = ['success' => false, 'message' => 'Missing session ID'];
                } else {
                    $response = ssh_read_output($sessionId);
                }
                break;

            case 'ssh_disconnect':
                $sessionId = $_POST['sessionId'] ?? '';
                if (empty($sessionId)) {
                    $response = ['success' => false, 'message' => 'Missing session ID'];
                } else {
                    $response = ssh_disconnect_session($sessionId);
                }
                break;

            case 'ssh_list':
                $response = ssh_get_connections();
                break;

            case 'exec_local':
                $command = $_POST['command'] ?? '';
                $cwd = $_POST['cwd'] ?? '~';
                $response = exec_local_command($command, $cwd);
                break;
        }
    } catch (Exception $e) {
        $response = ['success' => false, 'message' => $e->getMessage()];
    }

    echo json_encode($response);
    exit;
}

function addDirToZip($zip, $dir, $zipPath) {
    $files = @scandir($dir);
    if ($files === false) return;

    foreach ($files as $file) {
        if ($file === '.' || $file === '..') continue;

        $fullPath = $dir . DIRECTORY_SEPARATOR . $file;
        $zipFilePath = $zipPath . '/' . $file;

        if (is_dir($fullPath)) {
            $zip->addEmptyDir($zipFilePath);
            addDirToZip($zip, $fullPath, $zipFilePath);
        } else {
            $zip->addFile($fullPath, $zipFilePath);
        }
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// 🎨 SECTION 5: HTML TEMPLATE - The Interface (LEGENDARY EDITION)
// ═══════════════════════════════════════════════════════════════════════════
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🌌 NeonFS v1.1 - LEGENDARY</title>
    <style>
        /* CSS continues in next part due to size... */
        :root {
            --bg-primary: #0a0e27;
            --bg-secondary: #1a1f3a;
            --bg-elevated: #252b4a;
            --text-primary: #e0e0ff;
            --text-secondary: #a0a0c0;
            --accent-cyan: #00eaff;
            --accent-magenta: #ff00ea;
            --border-color: #00eaff33;
            --hover-bg: #00eaff1a;
            --success: #00ff88;
            --error: #ff0055;
            --warning: #ffaa00;
        }

        [data-theme="light"] {
            --bg-primary: #f5f5ff;
            --bg-secondary: #ffffff;
            --bg-elevated: #fafafe;
            --text-primary: #1a1f3a;
            --text-secondary: #505070;
            --accent-cyan: #0099cc;
            --accent-magenta: #cc0099;
            --border-color: #0099cc33;
            --hover-bg: #0099cc1a;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Courier New', 'Consolas', monospace;
            background: var(--bg-primary);
            color: var(--text-primary);
            line-height: 1.6;
            font-size: var(--font-size, 14px);
            transition: background 0.3s, color 0.3s;
            overflow-x: hidden;
        }

        /* Header */
        .header {
            background: var(--bg-secondary);
            border-bottom: 2px solid var(--accent-cyan);
            padding: 16px 24px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 16px;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 0 20px rgba(0, 234, 255, 0.3);
        }

        .logo {
            font-size: 24px;
            font-weight: bold;
            color: var(--accent-cyan);
            text-shadow: 0 0 10px rgba(0, 234, 255, 0.8);
            white-space: nowrap;
        }

        .search-box {
            flex: 1;
            max-width: 400px;
        }

        .search-box input {
            width: 100%;
            padding: 8px 12px;
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            color: var(--text-primary);
            font-family: inherit;
            font-size: inherit;
            transition: all 0.2s;
        }

        .search-box input:focus {
            outline: none;
            border-color: var(--accent-cyan);
            box-shadow: 0 0 10px rgba(0, 234, 255, 0.3);
        }

        .toolbar {
            display: flex;
            gap: 8px;
            align-items: center;
            flex-wrap: wrap;
        }

        .btn {
            padding: 8px 16px;
            background: var(--bg-elevated);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            color: var(--text-primary);
            cursor: pointer;
            font-family: inherit;
            font-size: inherit;
            transition: all 0.2s;
            white-space: nowrap;
        }

        .btn:hover:not(:disabled) {
            background: var(--hover-bg);
            border-color: var(--accent-cyan);
            box-shadow: 0 0 10px rgba(0, 234, 255, 0.3);
        }

        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--accent-cyan), #00a8cc);
            border: none;
            color: var(--bg-primary);
            font-weight: bold;
        }

        .btn-primary:hover:not(:disabled) {
            box-shadow: 0 0 15px rgba(0, 234, 255, 0.6);
            transform: translateY(-1px);
        }

        /* Breadcrumbs */
        .breadcrumbs {
            padding: 16px 24px;
            background: var(--bg-secondary);
            border-bottom: 1px solid var(--border-color);
            display: flex;
            align-items: center;
            gap: 8px;
            overflow-x: auto;
        }

        .breadcrumb {
            color: var(--accent-cyan);
            cursor: pointer;
            transition: all 0.2s;
            white-space: nowrap;
        }

        .breadcrumb:hover {
            text-shadow: 0 0 10px rgba(0, 234, 255, 0.8);
        }

        .breadcrumb-separator {
            color: var(--text-secondary);
        }

        /* File List */
        .container {
            padding: 24px;
        }

        .file-list {
            background: var(--bg-secondary);
            border: 1px solid var(--border-color);
            border-radius: 8px;
            overflow: hidden;
        }

        .file-list-header {
            display: grid;
            grid-template-columns: 40px 40px 1fr 120px 120px 100px;
            gap: 16px;
            padding: 12px 16px;
            background: var(--bg-elevated);
            border-bottom: 1px solid var(--border-color);
            font-weight: bold;
            color: var(--accent-cyan);
            cursor: pointer;
            user-select: none;
        }

        .file-list-header > div {
            display: flex;
            align-items: center;
            gap: 4px;
        }

        .sort-indicator {
            font-size: 10px;
            opacity: 0.5;
        }

        .file-item {
            display: grid;
            grid-template-columns: 40px 40px 1fr 120px 120px 100px;
            gap: 16px;
            padding: 12px 16px;
            border-bottom: 1px solid var(--border-color);
            cursor: pointer;
            transition: all 0.2s;
            position: relative;
        }

        .file-item:hover {
            background: var(--hover-bg);
        }

        .file-item.selected {
            background: rgba(0, 234, 255, 0.2);
            border-left: 3px solid var(--accent-cyan);
        }

        .file-icon {
            font-size: 20px;
            text-align: center;
        }

        .file-name {
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .file-actions {
            display: flex;
            gap: 4px;
            opacity: 0;
            transition: opacity 0.2s;
        }

        .file-item:hover .file-actions {
            opacity: 1;
        }

        .action-btn {
            padding: 4px 8px;
            font-size: 12px;
        }

        /* Batch Actions Toolbar */
        .batch-toolbar {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            background: var(--bg-secondary);
            border: 2px solid var(--accent-cyan);
            border-radius: 8px;
            padding: 12px 24px;
            display: none;
            gap: 12px;
            align-items: center;
            box-shadow: 0 0 30px rgba(0, 234, 255, 0.5);
            z-index: 200;
        }

        .batch-toolbar.active {
            display: flex;
        }

        .batch-count {
            color: var(--accent-cyan);
            font-weight: bold;
        }

        /* Context Menu */
        .context-menu {
            position: fixed;
            background: var(--bg-secondary);
            border: 2px solid var(--accent-cyan);
            border-radius: 8px;
            box-shadow: 0 0 30px rgba(0, 234, 255, 0.5);
            z-index: 1000;
            display: none;
            min-width: 200px;
        }

        .context-menu.active {
            display: block;
        }

        .context-menu-item {
            padding: 12px 16px;
            cursor: pointer;
            transition: all 0.2s;
            border-bottom: 1px solid var(--border-color);
        }

        .context-menu-item:last-child {
            border-bottom: none;
        }

        .context-menu-item:hover {
            background: var(--hover-bg);
        }

        .context-menu-separator {
            height: 1px;
            background: var(--border-color);
            margin: 4px 0;
        }

        /* Modal */
        .modal-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(10, 14, 39, 0.95);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }

        .modal-overlay.active {
            display: flex;
        }

        .modal {
            background: var(--bg-secondary);
            border: 2px solid var(--accent-cyan);
            border-radius: 8px;
            padding: 24px;
            max-width: 90vw;
            max-height: 90vh;
            overflow: auto;
            box-shadow: 0 0 30px rgba(0, 234, 255, 0.5);
        }

        .modal-large {
            max-width: 1200px;
            width: 90%;
        }

        .modal-header {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 16px;
            color: var(--accent-cyan);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-close {
            cursor: pointer;
            font-size: 24px;
            line-height: 1;
        }

        .modal-body {
            margin-bottom: 16px;
        }

        .modal-footer {
            display: flex;
            gap: 8px;
            justify-content: flex-end;
        }

        .form-group {
            margin-bottom: 16px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: var(--text-secondary);
        }

        .form-group input,
        .form-group textarea {
            width: 100%;
            padding: 8px 12px;
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            color: var(--text-primary);
            font-family: inherit;
            font-size: inherit;
        }

        /* Preview */
        .preview-image {
            max-width: 100%;
            max-height: 70vh;
            display: block;
            margin: 0 auto;
        }

        .preview-code {
            background: var(--bg-primary);
            padding: 16px;
            border-radius: 4px;
            overflow: auto;
            max-height: 70vh;
            font-family: 'Courier New', monospace;
            white-space: pre;
            line-height: 1.5;
        }

        /* Permission Editor */
        .permission-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 16px;
            margin: 16px 0;
        }

        .permission-section {
            background: var(--bg-primary);
            padding: 12px;
            border-radius: 4px;
            border: 1px solid var(--border-color);
        }

        .permission-section h4 {
            color: var(--accent-cyan);
            margin-bottom: 8px;
        }

        .permission-checkbox {
            display: flex;
            align-items: center;
            gap: 8px;
            margin: 4px 0;
        }

        /* Toast Notifications */
        .toast-container {
            position: fixed;
            top: 80px;
            right: 20px;
            z-index: 2000;
            display: flex;
            flex-direction: column;
            gap: 12px;
        }

        .toast {
            background: var(--bg-secondary);
            border: 2px solid var(--accent-cyan);
            border-radius: 8px;
            padding: 16px;
            box-shadow: 0 0 20px rgba(0, 234, 255, 0.5);
            min-width: 300px;
            max-width: 400px;
            animation: slideIn 0.3s ease-out;
        }

        .toast.success {
            border-color: var(--success);
            box-shadow: 0 0 20px rgba(0, 255, 136, 0.5);
        }

        .toast.error {
            border-color: var(--error);
            box-shadow: 0 0 20px rgba(255, 0, 85, 0.5);
        }

        .toast.warning {
            border-color: var(--warning);
            box-shadow: 0 0 20px rgba(255, 170, 0, 0.5);
        }

        @keyframes slideIn {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        /* Drag & Drop Overlay */
        .drop-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 234, 255, 0.1);
            border: 4px dashed var(--accent-cyan);
            z-index: 999;
            align-items: center;
            justify-content: center;
        }

        .drop-overlay.active {
            display: flex;
        }

        .drop-message {
            font-size: 48px;
            color: var(--accent-cyan);
            text-shadow: 0 0 20px rgba(0, 234, 255, 0.8);
        }

        /* Loading */
        .loading {
            display: none;
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            z-index: 2000;
        }

        .loading.active {
            display: block;
        }

        .spinner {
            width: 60px;
            height: 60px;
            border: 4px solid var(--border-color);
            border-top-color: var(--accent-cyan);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* Responsive */
        @media (max-width: 768px) {
            .file-list-header,
            .file-item {
                grid-template-columns: 40px 40px 1fr 80px;
            }

            .file-list-header > :nth-child(4),
            .file-list-header > :nth-child(5),
            .file-item > :nth-child(4),
            .file-item > :nth-child(5) {
                display: none;
            }

            .toolbar {
                flex-wrap: wrap;
            }
        }

        /* Utilities */
        .hidden {
            display: none !important;
        }

        .text-success {
            color: var(--success);
        }

        .text-error {
            color: var(--error);
        }

        .text-warning {
            color: var(--warning);
        }

        /* ═══════════════════════════════════════════════════════════════════════
           🔌 NEONTERM - Terminal Styles
           ═══════════════════════════════════════════════════════════════════════ */

        .terminal-panel {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            height: 320px;
            background: var(--bg-primary);
            border-top: 2px solid var(--accent-cyan);
            box-shadow: 0 -5px 30px rgba(0, 234, 255, 0.3);
            display: flex;
            flex-direction: column;
            z-index: 500;
            transform: translateY(100%);
            transition: transform 0.3s ease;
        }

        .terminal-panel.active {
            transform: translateY(0);
        }

        .terminal-header {
            display: flex;
            align-items: center;
            background: var(--bg-secondary);
            border-bottom: 1px solid var(--border-color);
            padding: 8px 12px;
            gap: 8px;
        }

        .terminal-tabs {
            display: flex;
            gap: 4px;
            flex: 1;
            overflow-x: auto;
        }

        .terminal-tab {
            padding: 6px 12px;
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 4px 4px 0 0;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 12px;
            white-space: nowrap;
            transition: all 0.2s;
        }

        .terminal-tab:hover {
            background: var(--hover-bg);
        }

        .terminal-tab.active {
            background: var(--bg-elevated);
            border-bottom-color: var(--accent-cyan);
            color: var(--accent-cyan);
        }

        .terminal-tab-icon {
            font-size: 14px;
        }

        .terminal-tab-name {
            max-width: 120px;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .terminal-tab-close {
            opacity: 0;
            cursor: pointer;
            font-size: 14px;
            line-height: 1;
            transition: opacity 0.2s;
        }

        .terminal-tab:hover .terminal-tab-close {
            opacity: 1;
        }

        .terminal-tab-close:hover {
            color: var(--error);
        }

        .terminal-actions {
            display: flex;
            gap: 8px;
        }

        .terminal-btn {
            padding: 4px 8px;
            font-size: 12px;
        }

        .terminal-body {
            flex: 1;
            display: flex;
            overflow: hidden;
        }

        .terminal-output {
            flex: 1;
            background: #0a0e27;
            color: #e0e0ff;
            font-family: 'Courier New', Consolas, monospace;
            font-size: 13px;
            line-height: 1.4;
            padding: 12px;
            overflow-y: auto;
            white-space: pre-wrap;
            word-wrap: break-word;
            cursor: text;
        }

        .terminal-output::-webkit-scrollbar {
            width: 8px;
        }

        .terminal-output::-webkit-scrollbar-track {
            background: var(--bg-primary);
        }

        .terminal-output::-webkit-scrollbar-thumb {
            background: var(--accent-cyan);
            border-radius: 4px;
        }

        .terminal-line {
            min-height: 1.4em;
        }

        .terminal-prompt {
            color: var(--accent-cyan);
        }

        .terminal-input-line {
            display: flex;
            align-items: center;
            background: var(--bg-secondary);
            padding: 8px 12px;
            border-top: 1px solid var(--border-color);
        }

        .terminal-prompt-label {
            color: var(--accent-cyan);
            font-family: 'Courier New', Consolas, monospace;
            font-size: 13px;
            margin-right: 8px;
            white-space: nowrap;
        }

        .terminal-input {
            flex: 1;
            background: transparent;
            border: none;
            color: var(--text-primary);
            font-family: 'Courier New', Consolas, monospace;
            font-size: 13px;
            outline: none;
        }

        /* ANSI Colors */
        .ansi-0 { color: #e0e0ff; }
        .ansi-1 { font-weight: bold; }
        .ansi-3 { font-style: italic; }
        .ansi-4 { text-decoration: underline; }
        .ansi-7 { background: #e0e0ff; color: #0a0e27; }
        .ansi-9 { text-decoration: line-through; }
        .ansi-22 { font-weight: normal; }
        .ansi-23 { font-style: normal; }
        .ansi-24 { text-decoration: none; }
        .ansi-27 { background: transparent; color: #e0e0ff; }
        .ansi-29 { text-decoration: none; }

        .ansi-30 { color: #808080; }
        .ansi-31 { color: #ff5555; }
        .ansi-32 { color: #50fa7b; }
        .ansi-33 { color: #f1fa8c; }
        .ansi-34 { color: #bd93f9; }
        .ansi-35 { color: #ff79c6; }
        .ansi-36 { color: #8be9fd; }
        .ansi-37 { color: #ffffff; }

        .ansi-40 { background: #6272a4; }
        .ansi-41 { background: #ff5555; }
        .ansi-42 { background: #50fa7b; }
        .ansi-43 { background: #f1fa8c; }
        .ansi-44 { background: #bd93f9; }
        .ansi-45 { background: #ff79c6; }
        .ansi-46 { background: #8be9fd; }
        .ansi-47 { background: #ffffff; }

        .ansi-90 { color: #6272a4; }
        .ansi-91 { color: #ff6e6e; }
        .ansi-92 { color: #69ff94; }
        .ansi-93 { color: #ffffa5; }
        .ansi-94 { color: #d6acff; }
        .ansi-95 { color: #ff92df; }
        .ansi-96 { color: #a4ffff; }
        .ansi-97 { color: #ffffff; }

        .ansi-100 { background: #6272a4; color: #e0e0ff; }
        .ansi-101 { background: #ff5555; color: #e0e0ff; }
        .ansi-102 { background: #50fa7b; color: #0a0e27; }
        .ansi-103 { background: #f1fa8c; color: #0a0e27; }
        .ansi-104 { background: #bd93f9; color: #e0e0ff; }
        .ansi-105 { background: #ff79c6; color: #e0e0ff; }
        .ansi-106 { background: #8be9fd; color: #0a0e27; }
        .ansi-107 { background: #ffffff; color: #0a0e27; }

        /* Terminal toggle button in header */
        #btnTerminal.active {
            background: linear-gradient(135deg, var(--accent-cyan), #00a8cc);
            color: var(--bg-primary);
            font-weight: bold;
        }

        /* Quick Command Bar */
        .quick-bar {
            background: var(--bg-secondary);
            border-top: 1px solid var(--border-color);
            padding: 8px 12px;
            display: flex;
            gap: 8px;
            align-items: center;
        }

        .quick-bar.hidden {
            display: none;
        }

        .quick-input {
            flex: 1;
            padding: 8px 12px;
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            color: var(--text-primary);
            font-family: inherit;
        }

        .quick-input:focus {
            outline: none;
            border-color: var(--accent-cyan);
        }

        .quick-suggestions {
            position: absolute;
            bottom: 100%;
            left: 0;
            right: 0;
            background: var(--bg-secondary);
            border: 1px solid var(--border-color);
            border-bottom: none;
            max-height: 200px;
            overflow-y: auto;
            display: none;
        }

        .quick-suggestions.active {
            display: block;
        }

        .quick-suggestion {
            padding: 8px 12px;
            cursor: pointer;
            border-bottom: 1px solid var(--border-color);
        }

        .quick-suggestion:hover {
            background: var(--hover-bg);
        }

        .quick-suggestion-name {
            color: var(--accent-cyan);
            font-weight: bold;
        }

        .quick-suggestion-command {
            color: var(--text-secondary);
            font-size: 12px;
        }

        /* SSH Connection Modal Styles */
        .ssh-connections-list {
            max-height: 300px;
            overflow-y: auto;
            margin-bottom: 16px;
        }

        .ssh-connection-item {
            display: flex;
            align-items: center;
            padding: 12px;
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            margin-bottom: 8px;
            cursor: pointer;
            transition: all 0.2s;
        }

        .ssh-connection-item:hover {
            border-color: var(--accent-cyan);
        }

        .ssh-connection-item.connected {
            border-left: 3px solid var(--success);
        }

        .ssh-connection-info {
            flex: 1;
        }

        .ssh-connection-name {
            color: var(--accent-cyan);
            font-weight: bold;
        }

        .ssh-connection-host {
            color: var(--text-secondary);
            font-size: 12px;
        }

        .ssh-connection-status {
            font-size: 12px;
        }

        .ssh-connection-status.connected {
            color: var(--success);
        }

        .ssh-connection-status.disconnected {
            color: var(--text-secondary);
        }

        .form-group {
            margin-bottom: 16px;
        }

        .form-group label {
            display: block;
            margin-bottom: 4px;
            color: var(--text-secondary);
            font-size: 12px;
        }

        .form-group input {
            width: 100%;
            padding: 10px 12px;
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            color: var(--text-primary);
            font-family: inherit;
        }

        .form-group input:focus {
            outline: none;
            border-color: var(--accent-cyan);
        }
    </style>
<?php
$initialFolder = '';
if (isset($_GET['current_folder'])) {
    $folder = $_GET['current_folder'];
    if (strpos($folder, '/') === 0 && $folder !== '/') {
        $initialFolder = $folder;
    }
}
?>
<script>
    window.__initialFolder = <?= json_encode($initialFolder) ?>;
</script>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="logo">🌌 NeonFS v1.1</div>
        <div class="search-box">
            <input type="text" id="searchInput" placeholder="🔍 Search files..." autocomplete="off">
        </div>
        <div class="toolbar">
            <button class="btn" id="btnSort" title="Sort files">⇅ Sort</button>
            <button class="btn" id="btnFontDecrease" title="Decrease font size (Ctrl+-)" onclick="window.adjustFontSize(-2)">A-</button>
            <button class="btn" id="btnFontIncrease" title="Increase font size (Ctrl++)" onclick="window.adjustFontSize(2)">A+</button>
            <button class="btn" id="btnThemeToggle" title="Toggle theme (Ctrl+T)">☀️</button>
            <button class="btn btn-primary" id="btnNewFolder" title="New folder (N)">+ Folder</button>
            <button class="btn btn-primary" id="btnUpload" title="Upload files (U)">⬆️ Upload</button>
            <button class="btn" id="btnTerminal" title="Toggle Terminal (Ctrl+`)">⬛ Terminal</button>
        </div>
    </div>

    <!-- Breadcrumbs -->
    <div class="breadcrumbs" id="breadcrumbs">
        <span class="breadcrumb" data-path="/">/</span>
    </div>

    <!-- File List -->
    <div class="container">
        <div class="file-list">
            <div class="file-list-header">
                <div><input type="checkbox" id="selectAll" title="Select all (Ctrl+A)"></div>
                <div></div>
                <div data-sort="name">Name <span class="sort-indicator">▼</span></div>
                <div data-sort="size">Size <span class="sort-indicator"></span></div>
                <div data-sort="modified">Modified <span class="sort-indicator"></span></div>
                <div>Actions</div>
            </div>
            <div id="fileListBody"></div>
        </div>
    </div>

    <!-- Batch Toolbar -->
    <div class="batch-toolbar" id="batchToolbar">
        <span class="batch-count"><span id="batchCount">0</span> selected</span>
        <button class="btn" onclick="window.copySelected()">📋 Copy</button>
        <button class="btn" onclick="window.cutSelected()">✂️ Cut</button>
        <button class="btn" onclick="window.deleteSelected()">🗑️ Delete</button>
        <button class="btn" onclick="window.downloadSelected()">⬇️ Download</button>
        <button class="btn" onclick="window.deselectAll()">✖ Clear</button>
    </div>

    <!-- NeonTerm Panel -->
    <div class="terminal-panel" id="terminalPanel">
        <div class="terminal-header">
            <div class="terminal-tabs" id="terminalTabs">
                <div class="terminal-tab active" data-session-id="">
                    <span class="terminal-tab-icon">💻</span>
                    <span class="terminal-tab-name">Local</span>
                    <span class="terminal-tab-close" onclick="event.stopPropagation(); window.neonTerm.closeTab('')">✖</span>
                </div>
            </div>
            <div class="terminal-actions">
                <button class="btn terminal-btn" onclick="window.neonTerm.showConnectionManager()" title="Connections">🔌</button>
                <button class="btn terminal-btn" onclick="window.neonTerm.toggleQuickBar()" title="Quick Bar (Ctrl+/)">⚡</button>
                <button class="btn terminal-btn" onclick="window.neonTerm.togglePanel()" title="Hide Terminal (Ctrl+`)">▼</button>
            </div>
        </div>
        <div class="terminal-body">
            <div class="terminal-output" id="terminalOutput"></div>
        </div>
        <div class="terminal-input-line">
            <span class="terminal-prompt-label" id="terminalPromptLabel">user@localhost:~$</span>
            <input type="text" class="terminal-input" id="terminalInput" autocomplete="off" spellcheck="false" placeholder="Type command...">
        </div>
    </div>

    <!-- Context Menu -->
    <div class="context-menu" id="contextMenu">
        <div class="context-menu-item" onclick="window.contextAction('open')">📂 Open</div>
        <div class="context-menu-item" onclick="window.contextAction('preview')">👁️ Preview</div>
        <div class="context-menu-separator"></div>
        <div class="context-menu-item" onclick="window.contextAction('rename')">✏️ Rename</div>
        <div class="context-menu-item" onclick="window.contextAction('copy')">📋 Copy</div>
        <div class="context-menu-item" onclick="window.contextAction('cut')">✂️ Cut</div>
        <div class="context-menu-item" onclick="window.contextAction('paste')">📥 Paste</div>
        <div class="context-menu-separator"></div>
        <div class="context-menu-item" onclick="window.contextAction('download')">⬇️ Download</div>
        <div class="context-menu-item" onclick="window.contextAction('chmod')">🔒 Permissions</div>
        <div class="context-menu-item" onclick="window.contextAction('copyPath')">📍 Copy Path</div>
        <div class="context-menu-separator"></div>
        <div class="context-menu-item" onclick="window.contextAction('terminal')">📡 Open Terminal Here</div>
        <div class="context-menu-separator"></div>
        <div class="context-menu-item" onclick="window.contextAction('delete')">🗑️ Delete</div>
    </div>

    <!-- Modals -->
    <div class="modal-overlay" id="modalOverlay">
        <div class="modal" id="modal">
            <div class="modal-header">
                <span id="modalHeader">Modal Title</span>
                <span class="modal-close" onclick="window.closeModal()">✖</span>
            </div>
            <div class="modal-body" id="modalBody"></div>
            <div class="modal-footer" id="modalFooter">
                <button class="btn" onclick="window.closeModal()">Cancel</button>
                <button class="btn btn-primary" id="modalConfirm">Confirm</button>
            </div>
        </div>
    </div>

    <!-- Toast Container -->
    <div class="toast-container" id="toastContainer"></div>

    <!-- Drop Overlay -->
    <div class="drop-overlay" id="dropOverlay">
        <div class="drop-message">📦 Drop files here to upload</div>
    </div>

    <!-- Loading -->
    <div class="loading" id="loading">
        <div class="spinner"></div>
    </div>

    <!-- Hidden file input -->
    <input type="file" id="fileInput" multiple style="display: none">

    <script>
    // ═══════════════════════════════════════════════════════════════════════════
    // 🧠 JAVASCRIPT - THE MIND (LEGENDARY EDITION v1.1)
    // ═══════════════════════════════════════════════════════════════════════════

    (function() {
        'use strict';

        // State Management
const state = {
    // Navigation history arrays for proper back/forward functionality
    backStack: [],
    forwardStack: [],

    // Determine initial folder from URL ?current_folder=...
    currentPath: (function() {
        if (window.__initialFolder) {
            return window.__initialFolder;
        }
        const params = new URLSearchParams(window.location.search);
        const folder = params.get('current_folder');
        if (folder && folder.startsWith('/') && folder !== '/') {
            return folder;
        }
        return <?= json_encode(NEONFS_ROOT_PATH) ?>;
    })(),

    selectedItems: new Set(),
    theme: localStorage.getItem('neonfs_theme') || 'dark',
    fontSize: parseInt(localStorage.getItem('neonfs_font_size')) || 14,
    files: [],
    sortBy: localStorage.getItem('neonfs_sort_by') || 'name',
    sortDir: localStorage.getItem('neonfs_sort_dir') || 'asc',
    contextTarget: null,
    clipboard: []
};

        // Initialize
        document.documentElement.setAttribute('data-theme', state.theme);
        document.documentElement.style.setProperty('--font-size', state.fontSize + 'px');
        updateThemeIcon();
        loadDirectory(state.currentPath);
        loadClipboard();

        // ═══════════════════════════════════════════════════════════════
        // Core Functions
        // ═══════════════════════════════════════════════════════════════

function loadDirectory(path) {
    showLoading();

    fetch('?action=list', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: 'path=' + encodeURIComponent(path)
    })
    .then(r => r.json())
    .then(data => {
        hideLoading();

        if (!data.success) {
            showToast('Error: ' + data.message, 'error');
            return;
        }

        // Update navigation history
        if (state.currentPath && state.currentPath !== data.path) {
            state.backStack.push(state.currentPath);
            state.forwardStack = []; // clear forward history on new navigation
        }

        state.currentPath = data.path;
        state.files = data.items;
        state.selectedItems.clear();

        // Push new state to browser history
        history.pushState({ path: data.path }, '', `?current_folder=${encodeURIComponent(data.path)}`);
        
        // Debug logging
        console.log('Navigation:', {
            current: data.path,
            backStack: state.backStack,
            forwardStack: state.forwardStack
        });

        sortFiles();
        renderFileList();
        renderBreadcrumbs(data.path);
        updateBatchToolbar();
    })
    .catch(err => {
        hideLoading();
        showToast('Error loading directory: ' + err.message, 'error');
    });
}

        function sortFiles() {
            state.files.sort((a, b) => {
                if (a.isParent) return -1;
                if (b.isParent) return 1;

                let aVal, bVal;

                switch (state.sortBy) {
                    case 'name':
                        aVal = a.name.toLowerCase();
                        bVal = b.name.toLowerCase();
                        break;
                    case 'size':
                        aVal = a.size || 0;
                        bVal = b.size || 0;
                        break;
                    case 'modified':
                        aVal = a.modified || 0;
                        bVal = b.modified || 0;
                        break;
                    default:
                        aVal = a.name.toLowerCase();
                        bVal = b.name.toLowerCase();
                }

                if (a.isDir !== b.isDir) {
                    return b.isDir - a.isDir;
                }

                if (aVal < bVal) return state.sortDir === 'asc' ? -1 : 1;
                if (aVal > bVal) return state.sortDir === 'asc' ? 1 : -1;
                return 0;
            });

            updateSortIndicators();
        }

        function updateSortIndicators() {
            document.querySelectorAll('.file-list-header [data-sort]').forEach(el => {
                const indicator = el.querySelector('.sort-indicator');
                if (el.dataset.sort === state.sortBy) {
                    indicator.textContent = state.sortDir === 'asc' ? '▼' : '▲';
                    indicator.style.opacity = '1';
                } else {
                    indicator.textContent = '';
                    indicator.style.opacity = '0.5';
                }
            });
        }

        function renderFileList() {
            const container = document.getElementById('fileListBody');
            container.innerHTML = '';

            state.files.forEach(item => {
                if (item.isParent) return;

                const row = document.createElement('div');
                row.className = 'file-item';
                row.dataset.path = item.path;
                row.dataset.isDir = item.isDir;

                if (state.selectedItems.has(item.path)) {
                    row.classList.add('selected');
                }

                row.innerHTML = `
                    <div><input type="checkbox" class="item-checkbox" data-path="${item.path}" ${state.selectedItems.has(item.path) ? 'checked' : ''}></div>
                    <div class="file-icon">${item.icon}</div>
                    <div class="file-name">
                        <span>${escapeHtml(item.name)}</span>
                    </div>
                    <div>${item.sizeFormatted}</div>
                    <div>${item.modifiedFormatted}</div>
                    <div class="file-actions">
                        <button class="btn action-btn" data-action="preview" data-path="${item.path}" title="Preview">👁️</button>
                        <button class="btn action-btn" data-action="rename" data-path="${item.path}" title="Rename">✏️</button>
                        <button class="btn action-btn" data-action="delete" data-path="${item.path}" title="Delete">🗑️</button>
                        <button class="btn action-btn" data-action="download" data-path="${item.path}" title="Download">⬇️</button>
                    </div>
                `;

                // Double-click to open
row.addEventListener('click', (e) => {
    // Ignore clicks on checkboxes or action buttons
    if (e.target.classList.contains('item-checkbox') || e.target.closest('.file-actions')) {
        return;
    }

    // Toggle selection on single click
    const checkbox = row.querySelector('.item-checkbox');
    if (checkbox) {
        checkbox.checked = !checkbox.checked;
        const path = checkbox.dataset.path;
        if (checkbox.checked) {
            state.selectedItems.add(path);
            row.classList.add('selected');
        } else {
            state.selectedItems.delete(path);
            row.classList.remove('selected');
        }
        updateBatchToolbar();
    }
});

row.addEventListener('dblclick', (e) => {
    // Double‑click retains original behavior: open folder or preview file
    if (!e.target.classList.contains('item-checkbox') && !e.target.closest('.file-actions')) {
        if (item.isDir) {
            loadDirectory(item.path);
        } else {
            previewFile(item.path);
        }
    }
});

                // Right-click context menu
                row.addEventListener('contextmenu', (e) => {
                    e.preventDefault();
                    state.contextTarget = item.path;
                    showContextMenu(e.pageX, e.pageY);
                });

                container.appendChild(row);
            });
        }

        function renderBreadcrumbs(path) {
            const container = document.getElementById('breadcrumbs');
            container.innerHTML = '';

            const parts = path.split('/').filter(p => p);
            let currentPath = '';

            const root = document.createElement('span');
            root.className = 'breadcrumb';
            root.textContent = '/';
            root.dataset.path = '/';
            root.addEventListener('click', () => loadDirectory('/'));
            container.appendChild(root);

            parts.forEach((part, index) => {
                currentPath += '/' + part;

                const sep = document.createElement('span');
                sep.className = 'breadcrumb-separator';
                sep.textContent = ' > ';
                container.appendChild(sep);

                const crumb = document.createElement('span');
                crumb.className = 'breadcrumb';
                crumb.textContent = part;
                crumb.dataset.path = currentPath;
                crumb.addEventListener('click', function() {
                    loadDirectory(this.dataset.path);
                });
                container.appendChild(crumb);
            });
        }

        // ═══════════════════════════════════════════════════════════════
        // File Operations
        // ═══════════════════════════════════════════════════════════════

        function deleteItems(paths) {
            if (!paths || paths.length === 0) return;

            if (!confirm(`Delete ${paths.length} item(s)?`)) return;

            showLoading();

            fetch('?action=delete', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'paths=' + encodeURIComponent(JSON.stringify(paths))
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    showToast(`Deleted ${data.deleted} item(s)`, 'success');
                    loadDirectory(state.currentPath);
                } else {
                    showToast('Error: ' + data.message, 'error');
                }
            })
            .catch(err => {
                hideLoading();
                showToast('Error: ' + err.message, 'error');
            });
        }

        function renameItem(path) {
            const oldName = path.split('/').pop();
            const newName = prompt('Rename to:', oldName);

            if (!newName || newName === oldName) return;

            showLoading();

            fetch('?action=rename', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'path=' + encodeURIComponent(path) + '&newName=' + encodeURIComponent(newName)
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    showToast('Renamed successfully', 'success');
                    loadDirectory(state.currentPath);
                } else {
                    showToast('Error: ' + data.message, 'error');
                }
            })
            .catch(err => {
                hideLoading();
                showToast('Error: ' + err.message, 'error');
            });
        }

        function downloadItems(paths) {
            if (!paths || paths.length === 0) return;

            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '?action=download';

            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'paths';
            input.value = JSON.stringify(paths);

            form.appendChild(input);
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);

            showToast(`Downloading ${paths.length} item(s)...`, 'success');
        }

        function newFolder() {
            const name = prompt('Folder name:');
            if (!name) return;

            showLoading();

            fetch('?action=newFolder', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'path=' + encodeURIComponent(state.currentPath) + '&name=' + encodeURIComponent(name)
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    showToast('Folder created', 'success');
                    loadDirectory(state.currentPath);
                } else {
                    showToast('Error: ' + data.message, 'error');
                }
            })
            .catch(err => {
                hideLoading();
                showToast('Error: ' + err.message, 'error');
            });
        }

        function uploadFiles(files) {
            if (!files || files.length === 0) return;

            const formData = new FormData();
            formData.append('path', state.currentPath);

            for (let i = 0; i < files.length; i++) {
                formData.append('files[]', files[i]);
            }

            showLoading();

            fetch('?action=upload', {
                method: 'POST',
                body: formData
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    showToast(`Uploaded ${data.uploaded} file(s)`, 'success');
                    loadDirectory(state.currentPath);
                } else {
                    showToast('Error: ' + data.message, 'error');
                }
            })
            .catch(err => {
                hideLoading();
                showToast('Error: ' + err.message, 'error');
            });
        }

        function previewFile(path) {
            showLoading();

            fetch('?action=preview', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'path=' + encodeURIComponent(path)
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (!data.success) {
                    showToast('Preview not available', 'warning');
                    return;
                }

                const modal = document.getElementById('modal');
                modal.classList.add('modal-large');

                if (data.type === 'image') {
                    document.getElementById('modalHeader').textContent = '🖼️ Image Preview';
                    document.getElementById('modalBody').innerHTML = `<img src="${data.data}" class="preview-image">`;
                    document.getElementById('modalFooter').style.display = 'none';
                } else if (data.type === 'text') {
                    document.getElementById('modalHeader').textContent = `💻 ${data.extension.toUpperCase()} Preview (${data.lines} lines)`;
                    document.getElementById('modalBody').innerHTML = `<div class="preview-code">${escapeHtml(data.content)}</div>`;
                    document.getElementById('modalFooter').style.display = 'none';
                }

                document.getElementById('modalOverlay').classList.add('active');
            })
            .catch(err => {
                hideLoading();
                showToast('Error: ' + err.message, 'error');
            });
        }

        function showChmodEditor(path) {
            const filename = path.split('/').pop();

            document.getElementById('modalHeader').textContent = '🔒 Edit Permissions: ' + filename;
            document.getElementById('modalBody').innerHTML = `
                <div class="permission-grid">
                    <div class="permission-section">
                        <h4>Owner</h4>
                        <div class="permission-checkbox"><input type="checkbox" id="owner_r"> Read</div>
                        <div class="permission-checkbox"><input type="checkbox" id="owner_w"> Write</div>
                        <div class="permission-checkbox"><input type="checkbox" id="owner_x"> Execute</div>
                    </div>
                    <div class="permission-section">
                        <h4>Group</h4>
                        <div class="permission-checkbox"><input type="checkbox" id="group_r"> Read</div>
                        <div class="permission-checkbox"><input type="checkbox" id="group_w"> Write</div>
                        <div class="permission-checkbox"><input type="checkbox" id="group_x"> Execute</div>
                    </div>
                    <div class="permission-section">
                        <h4>Others</h4>
                        <div class="permission-checkbox"><input type="checkbox" id="other_r"> Read</div>
                        <div class="permission-checkbox"><input type="checkbox" id="other_w"> Write</div>
                        <div class="permission-checkbox"><input type="checkbox" id="other_x"> Execute</div>
                    </div>
                </div>
                <div class="form-group">
                    <label>Octal Mode:</label>
                    <input type="text" id="octalMode" placeholder="755" maxlength="3">
                </div>
            `;

            document.getElementById('modalFooter').style.display = 'flex';
            document.getElementById('modalConfirm').onclick = function() {
                const mode = document.getElementById('octalMode').value;
                if (!/^[0-7]{3}$/.test(mode)) {
                    showToast('Invalid mode format', 'error');
                    return;
                }

                showLoading();
                closeModal();

                fetch('?action=chmod', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: 'path=' + encodeURIComponent(path) + '&mode=' + mode
                })
                .then(r => r.json())
                .then(data => {
                    hideLoading();

                    if (data.success) {
                        showToast('Permissions changed', 'success');
                        loadDirectory(state.currentPath);
                    } else {
                        showToast('Error: ' + data.message, 'error');
                    }
                })
                .catch(err => {
                    hideLoading();
                    showToast('Error: ' + err.message, 'error');
                });
            };

            // Auto-calculate octal from checkboxes
            const checkboxes = ['owner_r', 'owner_w', 'owner_x', 'group_r', 'group_w', 'group_x', 'other_r', 'other_w', 'other_x'];
            checkboxes.forEach(id => {
                document.getElementById(id).addEventListener('change', () => {
                    let owner = (document.getElementById('owner_r').checked ? 4 : 0) +
                                (document.getElementById('owner_w').checked ? 2 : 0) +
                                (document.getElementById('owner_x').checked ? 1 : 0);
                    let group = (document.getElementById('group_r').checked ? 4 : 0) +
                                (document.getElementById('group_w').checked ? 2 : 0) +
                                (document.getElementById('group_x').checked ? 1 : 0);
                    let other = (document.getElementById('other_r').checked ? 4 : 0) +
                                (document.getElementById('other_w').checked ? 2 : 0) +
                                (document.getElementById('other_x').checked ? 1 : 0);
                    document.getElementById('octalMode').value = '' + owner + group + other;
                });
            });

            document.getElementById('modalOverlay').classList.add('active');
        }

        // ═══════════════════════════════════════════════════════════════
        // Clipboard Operations
        // ═══════════════════════════════════════════════════════════════

        function loadClipboard() {
            fetch('?action=clipboard', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'operation=get'
            })
            .then(r => r.json())
            .then(data => {
                if (data.success && data.clipboard) {
                    state.clipboard = data.clipboard;
                }
            });
        }

        function copyToClipboard(paths, operation) {
            fetch('?action=clipboard', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'operation=' + operation + '&paths=' + encodeURIComponent(JSON.stringify(paths))
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    showToast(data.message, 'success');
                    state.clipboard = { paths, operation };
                } else {
                    showToast('Error: ' + data.message, 'error');
                }
            });
        }

        function pasteFromClipboard() {
            if (!state.clipboard || !state.clipboard.paths || state.clipboard.paths.length === 0) {
                showToast('Clipboard is empty', 'warning');
                return;
            }

            showLoading();

            fetch('?action=clipboard', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'operation=paste&destination=' + encodeURIComponent(state.currentPath)
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    showToast(`Pasted ${data.pasted} item(s)`, 'success');
                    loadDirectory(state.currentPath);
                } else {
                    showToast('Error: ' + data.message, 'error');
                }
            })
            .catch(err => {
                hideLoading();
                showToast('Error: ' + err.message, 'error');
            });
        }

        // ═══════════════════════════════════════════════════════════════
        // UI Functions
        // ═══════════════════════════════════════════════════════════════

        function showLoading() {
            document.getElementById('loading').classList.add('active');
        }

        function hideLoading() {
            document.getElementById('loading').classList.remove('active');
        }

        function showToast(message, type = 'success') {
            const container = document.getElementById('toastContainer');
            const toast = document.createElement('div');
            toast.className = `toast ${type}`;
            toast.textContent = message;

            container.appendChild(toast);

            setTimeout(() => {
                toast.style.animation = 'slideIn 0.3s ease-out reverse';
                setTimeout(() => container.removeChild(toast), 300);
            }, 3000);
        }

        function updateThemeIcon() {
            const btn = document.getElementById('btnThemeToggle');
            btn.textContent = state.theme === 'dark' ? '☀️' : '🌙';
        }

        function updateBatchToolbar() {
            const toolbar = document.getElementById('batchToolbar');
            const count = state.selectedItems.size;

            document.getElementById('batchCount').textContent = count;

            if (count > 0) {
                toolbar.classList.add('active');
            } else {
                toolbar.classList.remove('active');
            }
        }

        function showContextMenu(x, y) {
            const menu = document.getElementById('contextMenu');
            menu.classList.add('active');
            menu.style.left = x + 'px';
            menu.style.top = y + 'px';
        }

        function hideContextMenu() {
            document.getElementById('contextMenu').classList.remove('active');
        }

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        // ═══════════════════════════════════════════════════════════════
        // Global Window Functions
        // ═══════════════════════════════════════════════════════════════

        window.adjustFontSize = function(delta) {
            state.fontSize = Math.max(12, Math.min(24, state.fontSize + delta));
            document.documentElement.style.setProperty('--font-size', state.fontSize + 'px');
            localStorage.setItem('neonfs_font_size', state.fontSize);
        };

        window.closeModal = function() {
            document.getElementById('modalOverlay').classList.remove('active');
            document.getElementById('modal').classList.remove('modal-large');
            document.getElementById('modalFooter').style.display = 'flex';
        };

        window.copySelected = function() {
            if (state.selectedItems.size === 0) return;
            copyToClipboard(Array.from(state.selectedItems), 'copy');
        };

        window.cutSelected = function() {
            if (state.selectedItems.size === 0) return;
            copyToClipboard(Array.from(state.selectedItems), 'cut');
        };

        window.deleteSelected = function() {
            if (state.selectedItems.size === 0) return;
            deleteItems(Array.from(state.selectedItems));
        };

        window.downloadSelected = function() {
            if (state.selectedItems.size === 0) return;
            downloadItems(Array.from(state.selectedItems));
        };

        window.deselectAll = function() {
            state.selectedItems.clear();
            document.querySelectorAll('.item-checkbox').forEach(cb => cb.checked = false);
            document.querySelectorAll('.file-item').forEach(item => item.classList.remove('selected'));
            updateBatchToolbar();
        };

        window.contextAction = function(action) {
            hideContextMenu();
            const path = state.contextTarget;

            switch (action) {
                case 'open':
                    const item = state.files.find(f => f.path === path);
                    if (item && item.isDir) {
                        loadDirectory(path);
                    }
                    break;
                case 'preview':
                    previewFile(path);
                    break;
                case 'rename':
                    renameItem(path);
                    break;
                case 'copy':
                    copyToClipboard([path], 'copy');
                    break;
                case 'cut':
                    copyToClipboard([path], 'cut');
                    break;
                case 'paste':
                    pasteFromClipboard();
                    break;
                case 'download':
                    downloadItems([path]);
                    break;
                case 'chmod':
                    showChmodEditor(path);
                    break;
                case 'copyPath':
                    navigator.clipboard.writeText(path);
                    showToast('Path copied to clipboard', 'success');
                    break;
                case 'terminal':
                    const termItem = state.files.find(f => f.path === path);
                    if (termItem) {
                        if (!window.neonTerm.isPanelVisible) {
                            window.neonTerm.togglePanel();
                        }
                        if (window.neonTerm.activeSession === '') {
                            window.neonTerm.execute('cd ' + path + '\n');
                        } else {
                            window.neonTerm.execute('cd "' + path + '"\n');
                        }
                    }
                    break;
                case 'delete':
                    deleteItems([path]);
                    break;
            }
        };

        // Add popstate event listener for browser navigation
        window.addEventListener('popstate', (event) => {
            if (event.state && event.state.path) {
                // When browser back/forward is used, update our stacks accordingly
                const previousPath = state.currentPath;
                const newPath = event.state.path;
                
                if (previousPath !== newPath) {
                    // If going back, add current to forward stack
                    if (state.backStack.length > 0 && state.backStack[state.backStack.length - 1] === newPath) {
                        state.forwardStack.push(previousPath);
                        state.backStack.pop();
                    }
                    // If going forward, add current to back stack
                    else if (state.forwardStack.length > 0 && state.forwardStack[state.forwardStack.length - 1] === newPath) {
                        state.backStack.push(previousPath);
                        state.forwardStack.pop();
                    }
                }
                
                loadDirectory(newPath);
            }
        });

        // ═══════════════════════════════════════════════════════════════
        // Event Listeners
        // ═══════════════════════════════════════════════════════════════

        // Theme toggle
        document.getElementById('btnThemeToggle').addEventListener('click', () => {
            state.theme = state.theme === 'dark' ? 'light' : 'dark';
            document.documentElement.setAttribute('data-theme', state.theme);
            localStorage.setItem('neonfs_theme', state.theme);
            updateThemeIcon();
        });

        // Font size controls
        document.getElementById('btnFontIncrease').addEventListener('click', () => window.adjustFontSize(2));
        document.getElementById('btnFontDecrease').addEventListener('click', () => window.adjustFontSize(-2));

        // New folder
        document.getElementById('btnNewFolder').addEventListener('click', newFolder);

        // Upload
        document.getElementById('btnUpload').addEventListener('click', () => {
            document.getElementById('fileInput').click();
        });

        document.getElementById('fileInput').addEventListener('change', function() {
            if (this.files.length > 0) {
                uploadFiles(this.files);
                this.value = '';
            }
        });

        // Search/filter
        document.getElementById('searchInput').addEventListener('input', function() {
            const query = this.value.toLowerCase();
            document.querySelectorAll('.file-item').forEach(item => {
                const name = item.querySelector('.file-name span').textContent.toLowerCase();
                item.style.display = name.includes(query) ? '' : 'none';
            });
        });

        // Select all
        document.getElementById('selectAll').addEventListener('change', function() {
            const checked = this.checked;
            state.files.forEach(item => {
                if (!item.isParent) {
                    if (checked) {
                        state.selectedItems.add(item.path);
                    } else {
                        state.selectedItems.delete(item.path);
                    }
                }
            });
            renderFileList();
            updateBatchToolbar();
        });

        // File item checkboxes (delegated)
        document.getElementById('fileListBody').addEventListener('change', (e) => {
            if (e.target.classList.contains('item-checkbox')) {
                const path = e.target.dataset.path;
                if (e.target.checked) {
                    state.selectedItems.add(path);
                    e.target.closest('.file-item').classList.add('selected');
                } else {
                    state.selectedItems.delete(path);
                    e.target.closest('.file-item').classList.remove('selected');
                }
                updateBatchToolbar();
            }
        });

        // File action buttons (delegated)
        document.getElementById('fileListBody').addEventListener('click', (e) => {
            const btn = e.target.closest('[data-action]');
            if (!btn) return;

            e.stopPropagation();

            const action = btn.dataset.action;
            const path = btn.dataset.path;

            switch (action) {
                case 'preview':
                    previewFile(path);
                    break;
                case 'rename':
                    renameItem(path);
                    break;
                case 'delete':
                    deleteItems([path]);
                    break;
                case 'download':
                    downloadItems([path]);
                    break;
            }
        });

        // Sort headers
        document.querySelectorAll('.file-list-header [data-sort]').forEach(header => {
            header.addEventListener('click', () => {
                const sortBy = header.dataset.sort;

                if (state.sortBy === sortBy) {
                    state.sortDir = state.sortDir === 'asc' ? 'desc' : 'asc';
                } else {
                    state.sortBy = sortBy;
                    state.sortDir = 'asc';
                }

                localStorage.setItem('neonfs_sort_by', state.sortBy);
                localStorage.setItem('neonfs_sort_dir', state.sortDir);

                sortFiles();
                renderFileList();
            });
        });

        // Hide context menu on click outside
        document.addEventListener('click', (e) => {
            if (!e.target.closest('.context-menu')) {
                hideContextMenu();
            }
        });

        // Drag & Drop
        let dragCounter = 0;

        document.addEventListener('dragenter', (e) => {
            e.preventDefault();
            dragCounter++;
            document.getElementById('dropOverlay').classList.add('active');
        });

        document.addEventListener('dragleave', () => {
            dragCounter--;
            if (dragCounter === 0) {
                document.getElementById('dropOverlay').classList.remove('active');
            }
        });

        document.addEventListener('dragover', (e) => {
            e.preventDefault();
        });

        document.addEventListener('drop', (e) => {
            e.preventDefault();
            dragCounter = 0;
            document.getElementById('dropOverlay').classList.remove('active');

            const files = e.dataTransfer.files;
            if (files.length > 0) {
                uploadFiles(files);
            }
        });

 // Keyboard Shortcuts
document.addEventListener('keydown', (e) => {
    // Delete key
    if (e.key === 'Delete' && state.selectedItems.size > 0) {
        e.preventDefault();
        deleteSelected();
    }

    // Backspace – navigate back in history
    if (e.key === 'Backspace' && !e.ctrlKey && !e.shiftKey) {
        e.preventDefault();
        if (state.backStack.length > 0) {
            const previous = state.backStack.pop();
            state.forwardStack.push(state.currentPath);
            console.log('Going back:', { from: state.currentPath, to: previous, forwardStack: state.forwardStack });
            loadDirectory(previous);
        } else {
            console.log('Cannot go back - backStack is empty');
        }
    }

    // Shift+Backspace – navigate forward in history
    if (e.key === 'Backspace' && e.shiftKey) {
        e.preventDefault();
        if (state.forwardStack.length > 0) {
            const next = state.forwardStack.pop();
            state.backStack.push(state.currentPath);
            console.log('Going forward:', { from: state.currentPath, to: next, backStack: state.backStack });
            loadDirectory(next);
        } else {
            console.log('Cannot go forward - forwardStack is empty');
        }
    }

    // Ctrl+A - Select all
    if (e.ctrlKey && e.key === 'a') {
        e.preventDefault();
        document.getElementById('selectAll').checked = true;
        document.getElementById('selectAll').dispatchEvent(new Event('change'));
    }

    // Ctrl+D - Deselect all
    if (e.ctrlKey && e.key === 'd') {
        e.preventDefault();
        deselectAll();
    }

    // Ctrl+C - Copy
    if (e.ctrlKey && e.key === 'c' && state.selectedItems.size > 0) {
        e.preventDefault();
        copySelected();
    }

    // Ctrl+X - Cut
    if (e.ctrlKey && e.key === 'x' && state.selectedItems.size > 0) {
        e.preventDefault();
        cutSelected();
    }

    // Ctrl+V - Paste
    if (e.ctrlKey && e.key === 'v') {
        e.preventDefault();
        pasteFromClipboard();
    }

    // N - New folder
    if (e.key === 'n' && !e.ctrlKey && !e.target.closest('input')) {
        e.preventDefault();
        newFolder();
    }

    // U - Upload
    if (e.key === 'u' && !e.ctrlKey && !e.target.closest('input')) {
        e.preventDefault();
        document.getElementById('btnUpload').click();
    }

    // F2 - Rename (first selected item)
    if (e.key === 'F2' && state.selectedItems.size > 0) {
        e.preventDefault();
        renameItem(Array.from(state.selectedItems)[0]);
    }

    // Escape - Close modal or deselect
    if (e.key === 'Escape') {
        if (document.getElementById('modalOverlay').classList.contains('active')) {
            closeModal();
        } else if (state.selectedItems.size > 0) {
            deselectAll();
        }
    }

    // Ctrl+T - Toggle theme
    if (e.ctrlKey && e.key === 't') {
        e.preventDefault();
        document.getElementById('btnThemeToggle').click();
    }

    // Ctrl++ - Increase font
    if (e.ctrlKey && (e.key === '+' || e.key === '=')) {
        e.preventDefault();
        window.adjustFontSize(2);
    }

    // Ctrl+- - Decrease font
    if (e.ctrlKey && e.key === '-') {
        e.preventDefault();
        window.adjustFontSize(-2);
    }

    // Ctrl+` - Toggle terminal
    if (e.ctrlKey && e.key === '`') {
        e.preventDefault();
        window.neonTerm.togglePanel();
    }
});

        // ═══════════════════════════════════════════════════════════════════════
        // 🔌 NEONTERM - Terminal Emulator Class
        // ═══════════════════════════════════════════════════════════════════════

        class AnsiParser {
            constructor() {
                this.resetSequence = '\x1b[0m';
            }

            parse(text) {
                if (!text) return '';

                let result = '';
                let currentText = '';
                let i = 0;

                while (i < text.length) {
                    if (text[i] === '\x1b' && text[i + 1] === '[') {
                        if (currentText) {
                            result += escapeHtml(currentText);
                            currentText = '';
                        }

                        let j = i + 2;
                        while (j < text.length && !/[a-zA-Z]/.test(text[j])) {
                            j++;
                        }

                        const codeStr = text.substring(i + 2, j);
                        const code = text[j];
                        const codes = codeStr.split(';').filter(Boolean).map(Number);

                        if (code === 'm') {
                            result += this.applySgrCodes(codes);
                        } else if (code === 'J') {
                            result += '';
                        } else if (code === 'H' || code === 'f') {
                            result += '';
                        }

                        i = j + 1;
                    } else {
                        currentText += text[i];
                        i++;
                    }
                }

                if (currentText) {
                    result += escapeHtml(currentText);
                }

                return result || escapeHtml(text);
            }

            applySgrCodes(codes) {
                if (codes.length === 0 || (codes.length === 1 && codes[0] === 0)) {
                    return '</span>';
                }

                let spans = [];

                for (let i = 0; i < codes.length; i++) {
                    const code = codes[i];

                    if (code === 0) {
                        spans = [];
                    } else if (code === 1) {
                        spans.push('font-weight: bold');
                    } else if (code === 3) {
                        spans.push('font-style: italic');
                    } else if (code === 4) {
                        spans.push('text-decoration: underline');
                    } else if (code === 7) {
                        spans.push('background: var(--text-primary)');
                        spans.push('color: var(--bg-primary)');
                    } else if (code === 9) {
                        spans.push('text-decoration: line-through');
                    } else if (code === 22) {
                        spans = spans.filter(s => s !== 'font-weight: bold');
                    } else if (code === 23) {
                        spans = spans.filter(s => s !== 'font-style: italic');
                    } else if (code === 24) {
                        spans = spans.filter(s => s !== 'text-decoration: underline');
                    } else if (code === 27) {
                        spans = spans.filter(s => s !== 'background: var(--text-primary)');
                        spans = spans.filter(s => s !== 'color: var(--bg-primary)');
                    } else if (code === 29) {
                        spans = spans.filter(s => s !== 'text-decoration: line-through');
                    } else if (code >= 30 && code <= 37) {
                        spans.push('color: ' + this.getColor(code - 30));
                    } else if (code >= 40 && code <= 47) {
                        spans.push('background: ' + this.getColor(code - 40));
                    } else if (code >= 90 && code <= 97) {
                        spans.push('color: ' + this.getBrightColor(code - 90));
                    } else if (code >= 100 && code <= 107) {
                        spans.push('background: ' + this.getBrightColor(code - 100));
                    }
                }

                if (spans.length > 0) {
                    return '<span style="' + spans.join(';') + '">';
                }
                return '';
            }

            getColor(code) {
                const colors = {
                    0: '#808080',
                    1: '#ff5555',
                    2: '#50fa7b',
                    3: '#f1fa8c',
                    4: '#bd93f9',
                    5: '#ff79c6',
                    6: '#8be9fd',
                    7: '#ffffff'
                };
                return colors[code] || '#e0e0ff';
            }

            getBrightColor(code) {
                const colors = {
                    0: '#6272a4',
                    1: '#ff6e6e',
                    2: '#69ff94',
                    3: '#ffffa5',
                    4: '#d6acff',
                    5: '#ff92df',
                    6: '#a4ffff',
                    7: '#ffffff'
                };
                return colors[code] || '#e0e0ff';
            }
        }

        class NeonTerm {
            constructor() {
                this.sessions = new Map();
                this.activeSession = null;
                this.history = [];
                this.historyIndex = -1;
                this.ansiParser = new AnsiParser();
                this.isPanelVisible = false;
                this.currentCwd = '~';
                this.pollingInterval = null;

                this.initLocalSession();
                this.bindEvents();
                this.loadSnippets();
            }

            initLocalSession() {
                this.sessions.set('', {
                    name: 'Local',
                    host: 'localhost',
                    username: 'user',
                    cwd: '~',
                    type: 'local'
                });
                this.activeSession = '';
            }

            bindEvents() {
                const input = document.getElementById('terminalInput');
                const output = document.getElementById('terminalOutput');

                input.addEventListener('keydown', (e) => this.handleInput(e));
                input.addEventListener('focus', () => this.startPolling());
                input.addEventListener('blur', () => this.stopPolling());

                output.addEventListener('click', () => {
                    input.focus();
                });

                document.getElementById('btnTerminal').addEventListener('click', () => {
                    this.togglePanel();
                });

                document.getElementById('terminalTabs').addEventListener('click', (e) => {
                    const tab = e.target.closest('.terminal-tab');
                    if (tab) {
                        const sessionId = tab.dataset.sessionId;
                        this.switchSession(sessionId);
                    }
                });
            }

            handleInput(e) {
                if (e.key === 'Enter') {
                    const command = e.target.value;
                    if (command.trim()) {
                        this.history.push(command);
                        this.historyIndex = this.history.length;
                        this.execute(command);
                    }
                    e.target.value = '';
                } else if (e.key === 'ArrowUp') {
                    e.preventDefault();
                    if (this.historyIndex > 0) {
                        this.historyIndex--;
                        e.target.value = this.history[this.historyIndex];
                    }
                } else if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    if (this.historyIndex < this.history.length - 1) {
                        this.historyIndex++;
                        e.target.value = this.history[this.historyIndex];
                    } else {
                        this.historyIndex = this.history.length;
                        e.target.value = '';
                    }
                } else if (e.key === 'Tab') {
                    e.preventDefault();
                    this.handleTabCompletion(e.target.value);
                } else if (e.key === 'c' && e.ctrlKey) {
                    e.preventDefault();
                    this.write('\n^C\n');
                    this.updatePrompt();
                } else if (e.key === 'l' && e.ctrlKey) {
                    e.preventDefault();
                    this.clear();
                }
            }

            handleTabCompletion(current) {
                const words = current.split(' ');
                const lastWord = words[words.length - 1];

                const completions = this.getCompletions(lastWord);

                if (completions.length === 1) {
                    words[words.length - 1] = completions[0];
                    document.getElementById('terminalInput').value = words.join(' ');
                } else if (completions.length > 1) {
                    this.write('\n' + completions.join('  ') + '\n');
                    this.updatePrompt();
                }
            }

            getCompletions(partial) {
                const commands = ['ls', 'cd', 'pwd', 'cat', 'tail', 'head', 'grep', 'find', 'mkdir', 'rm', 'cp', 'mv', 'chmod', 'chown', 'ps', 'top', 'df', 'du', 'free', 'uname', 'whoami', 'date', 'echo', 'export', 'source', 'alias', 'unalias', 'history', 'clear', 'exit'];

                if (!partial) return commands;

                return commands.filter(cmd => cmd.startsWith(partial));
            }

            execute(command) {
                const session = this.sessions.get(this.activeSession);

                this.write(this.getPrompt() + escapeHtml(command) + '\n');

                if (this.activeSession === '') {
                    this.executeLocal(command);
                } else {
                    this.executeRemote(command);
                }
            }

            executeLocal(command) {
                const trimmed = command.trim();

                if (trimmed === 'clear') {
                    this.clear();
                    return;
                }

                if (trimmed === 'exit') {
                    this.write('logout\n');
                    return;
                }

                if (trimmed.startsWith('cd ')) {
                    const path = trimmed.substring(3).trim();
                    if (path) {
                        this.currentCwd = path;
                        this.updatePrompt();
                    }
                    return;
                }

                fetch('?action=exec_local', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: 'command=' + encodeURIComponent(command) + '&cwd=' + encodeURIComponent(this.currentCwd)
                })
                .then(r => r.json())
                .then(data => {
                    if (data.output) {
                        this.write(this.ansiParser.parse(data.output));
                    }
                    if (data.cwd) {
                        this.currentCwd = data.cwd;
                    }
                    this.updatePrompt();
                })
                .catch(err => {
                    this.write('Error: ' + err.message + '\n');
                });
            }

            executeRemote(command) {
                fetch('?action=ssh_command', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: 'sessionId=' + encodeURIComponent(this.activeSession) + '&command=' + encodeURIComponent(command)
                })
                .then(r => r.json())
                .then(data => {
                    if (data.success) {
                        if (data.output) {
                            this.write(this.ansiParser.parse(data.output));
                        }
                        if (data.cwd) {
                            const session = this.sessions.get(this.activeSession);
                            session.cwd = data.cwd;
                        }
                    } else {
                        this.write('Error: ' + data.message + '\n');
                        if (data.dead) {
                            this.closeTab(this.activeSession);
                        }
                    }
                })
                .catch(err => {
                    this.write('Error: ' + err.message + '\n');
                });
            }

            getPrompt() {
                const session = this.sessions.get(this.activeSession);
                const username = session?.username || 'user';
                const host = session?.host || 'localhost';
                const cwd = session?.cwd || '~';
                return '<span class="terminal-prompt">' + escapeHtml(username) + '@' + escapeHtml(host) + ':' + escapeHtml(cwd) + '$</span> ';
            }

            updatePrompt() {
                document.getElementById('terminalPromptLabel').innerHTML = this.getPrompt();
            }

            write(text) {
                const output = document.getElementById('terminalOutput');
                const line = document.createElement('div');
                line.className = 'terminal-line';
                line.innerHTML = text;
                output.appendChild(line);
                this.scrollToBottom();
            }

            clear() {
                const output = document.getElementById('terminalOutput');
                output.innerHTML = '';
            }

            scrollToBottom() {
                const output = document.getElementById('terminalOutput');
                output.scrollTop = output.scrollHeight;
            }

            togglePanel() {
                const panel = document.getElementById('terminalPanel');
                this.isPanelVisible = !this.isPanelVisible;

                if (this.isPanelVisible) {
                    panel.classList.add('active');
                    document.getElementById('terminalInput').focus();
                    this.startPolling();
                } else {
                    panel.classList.remove('active');
                    this.stopPolling();
                }

                document.getElementById('btnTerminal').classList.toggle('active', this.isPanelVisible);
            }

            startPolling() {
                if (this.pollingInterval) return;

                this.pollingInterval = setInterval(() => {
                    if (this.activeSession && this.activeSession !== '') {
                        this.pollOutput();
                    }
                }, 500);
            }

            stopPolling() {
                if (this.pollingInterval) {
                    clearInterval(this.pollingInterval);
                    this.pollingInterval = null;
                }
            }

            pollOutput() {
                fetch('?action=ssh_read', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: 'sessionId=' + encodeURIComponent(this.activeSession)
                })
                .then(r => r.json())
                .then(data => {
                    if (data.success && data.output) {
                        this.write(this.ansiParser.parse(data.output));
                    }
                    if (!data.alive) {
                        this.closeTab(this.activeSession);
                    }
                })
                .catch(() => {});
            }

            switchSession(sessionId) {
                this.activeSession = sessionId;

                document.querySelectorAll('.terminal-tab').forEach(tab => {
                    tab.classList.toggle('active', tab.dataset.sessionId === sessionId);
                });

                this.updatePrompt();
            }

            addTab(sessionInfo) {
                let session = this.sessions.get(sessionInfo.sessionId);
                if (!session) {
                    session = {
                        name: sessionInfo.name,
                        host: sessionInfo.host,
                        username: sessionInfo.username,
                        cwd: '~',
                        type: 'ssh',
                        sessionId: sessionInfo.sessionId
                    };
                    this.sessions.set(sessionInfo.sessionId, session);
                }

                const tabs = document.getElementById('terminalTabs');
                const tab = document.createElement('div');
                tab.className = 'terminal-tab';
                tab.dataset.sessionId = sessionInfo.sessionId;
                tab.innerHTML = `
                    <span class="terminal-tab-icon">🖥️</span>
                    <span class="terminal-tab-name">${escapeHtml(sessionInfo.name)}</span>
                    <span class="terminal-tab-close" onclick="event.stopPropagation(); window.neonTerm.closeTab('${sessionInfo.sessionId}')">✖</span>
                `;
                tabs.appendChild(tab);

                this.switchSession(sessionInfo.sessionId);
            }

            closeTab(sessionId) {
                if (sessionId && sessionId !== '') {
                    fetch('?action=ssh_disconnect', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                        body: 'sessionId=' + encodeURIComponent(sessionId)
                    });

                    this.sessions.delete(sessionId);

                    const tab = document.querySelector('.terminal-tab[data-session-id="' + sessionId + '"]');
                    if (tab) tab.remove();
                }

                if (this.activeSession === sessionId) {
                    this.switchSession('');
                }
            }

            showConnectionManager() {
                fetch('?action=ssh_list')
                .then(r => r.json())
                .then(data => {
                    if (data.success) {
                        this.renderConnectionManager(data.connections);
                    }
                });
            }

            renderConnectionManager(connections) {
                const listHtml = connections.length > 0
                    ? connections.map(conn => `
                        <div class="ssh-connection-item ${conn.connected ? 'connected' : ''}" onclick="window.neonTerm.connectFromManager('${conn.sessionId}', ${conn.connected})">
                            <div class="ssh-connection-info">
                                <div class="ssh-connection-name">${escapeHtml(conn.name)}</div>
                                <div class="ssh-connection-host">${escapeHtml(conn.username)}@${escapeHtml(conn.host)}:${conn.port}</div>
                            </div>
                            <div class="ssh-connection-status ${conn.connected ? 'connected' : 'disconnected'}">
                                ${conn.connected ? '● Connected' : '○ Saved'}
                            </div>
                        </div>
                    `).join('')
                    : '<p style="color: var(--text-secondary); padding: 20px; text-align: center;">No saved connections. Click "New Connection" to add one.</p>';

                const content = `
                    <div class="ssh-connections-list" id="sshConnectionsList">
                        ${listHtml}
                    </div>
                    <button class="btn btn-primary" onclick="window.neonTerm.showNewConnectionModal()">+ New Connection</button>
                `;

                showModal('🔌 SSH Connections', content);
            }

            connectFromManager(sessionId, alreadyConnected) {
                if (alreadyConnected) {
                    this.switchSession(sessionId);
                    closeModal();
                    this.togglePanel();
                } else {
                    const connections = document.querySelectorAll('.ssh-connection-item');
                    connections.forEach(el => el.style.opacity = '0.5');

                    fetch('?action=ssh_list')
                    .then(r => r.json())
                    .then(data => {
                        const conn = data.connections.find(c => c.sessionId === sessionId);
                        if (conn) {
                            this.showReconnectModal(conn);
                        }
                        connections.forEach(el => el.style.opacity = '');
                    });
                }
            }

            showReconnectModal(conn) {
                const content = `
                    <div class="form-group">
                        <label>Connection Name</label>
                        <input type="text" id="sshConnName" value="${escapeHtml(conn.name)}">
                    </div>
                    <div class="form-group">
                        <label>Host</label>
                        <input type="text" id="sshConnHost" value="${escapeHtml(conn.host)}" readonly>
                    </div>
                    <div class="form-group">
                        <label>Port</label>
                        <input type="number" id="sshConnPort" value="${conn.port}" readonly>
                    </div>
                    <div class="form-group">
                        <label>Username</label>
                        <input type="text" id="sshConnUsername" value="${escapeHtml(conn.username)}" readonly>
                    </div>
                    <div class="form-group">
                        <label>Password</label>
                        <input type="password" id="sshConnPassword" placeholder="Enter password to connect">
                    </div>
                `;

                showModal('🔌 Connect to ' + conn.name, content, () => {
                    const password = document.getElementById('sshConnPassword').value;
                    if (!password) {
                        showToast('Password required', 'error');
                        return false;
                    }

                    this.connect(conn.host, conn.port, conn.username, password, conn.name);
                    return false;
                });
            }

            showNewConnectionModal() {
                const content = `
                    <div class="form-group">
                        <label>Connection Name</label>
                        <input type="text" id="sshConnName" placeholder="Production Server">
                    </div>
                    <div class="form-group">
                        <label>Host</label>
                        <input type="text" id="sshConnHost" placeholder="192.168.1.1">
                    </div>
                    <div class="form-group">
                        <label>Port</label>
                        <input type="number" id="sshConnPort" value="22">
                    </div>
                    <div class="form-group">
                        <label>Username</label>
                        <input type="text" id="sshConnUsername" placeholder="root">
                    </div>
                    <div class="form-group">
                        <label>Password</label>
                        <input type="password" id="sshConnPassword">
                    </div>
                    <div style="margin-top: 16px; font-size: 12px; color: var(--warning);">
                        ⚠️ Password is stored encrypted in session only
                    </div>
                `;

                showModal('🔌 New SSH Connection', content, () => {
                    const name = document.getElementById('sshConnName').value || document.getElementById('sshConnHost').value;
                    const host = document.getElementById('sshConnHost').value;
                    const port = document.getElementById('sshConnPort').value;
                    const username = document.getElementById('sshConnUsername').value;
                    const password = document.getElementById('sshConnPassword').value;

                    if (!host || !username || !password) {
                        showToast('Please fill in all required fields', 'error');
                        return false;
                    }

                    this.connect(host, port, username, password, name);
                    return false;
                });
            }

            connect(host, port, username, password, name) {
                showLoading();

                fetch('?action=ssh_connect', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: 'host=' + encodeURIComponent(host) +
                          '&port=' + encodeURIComponent(port) +
                          '&username=' + encodeURIComponent(username) +
                          '&password=' + encodeURIComponent(password) +
                          '&name=' + encodeURIComponent(name)
                })
                .then(r => r.json())
                .then(data => {
                    hideLoading();

                    if (data.success) {
                        closeModal();
                        this.addTab(data);
                        this.togglePanel();
                        showToast('Connected to ' + data.host, 'success');
                    } else {
                        showToast(data.message, 'error');
                    }
                })
                .catch(err => {
                    hideLoading();
                    showToast('Connection failed: ' + err.message, 'error');
                });
            }

            toggleQuickBar() {
                showToast('Quick Bar - Feature coming soon', 'info');
            }

            loadSnippets() {
                this.snippets = JSON.parse(localStorage.getItem('neonfs_terminal_snippets') || '[]');
            }

            saveSnippet(name, command) {
                this.snippets.push({ name, command, created: Date.now() });
                localStorage.setItem('neonfs_terminal_snippets', JSON.stringify(this.snippets));
                showToast('Snippet saved', 'success');
            }
        }

        window.neonTerm = new NeonTerm();

        console.log('🌌 NeonFS v1.1 LEGENDARY EDITION initialized');
        console.log('The blueprint is complete. The grid awaits you.');
    })();
    </script>
</body>
</html>
