<?php
/**
 * 🌌 NeonFS - Cyberpunk File Manager
 *
 * "Where filesystem mastery meets neon glory"
 *
 * A lightweight, blazingly fast, cyberpunk-styled file manager
 * Zero dependencies • Pure vanilla • Single file • Unstoppable
 *
 * The blueprint was never lost. It was waiting... for you.
 */

// ═══════════════════════════════════════════════════════════════════════════
// 🔮 SECTION 1: CONFIGURATION - The Foundation
// ═══════════════════════════════════════════════════════════════════════════

define('NEONFS_VERSION', '1.0.0');
define('NEONFS_REQUIRE_AUTH', false);           // Set true to require password
define('NEONFS_PASSWORD', 'neon2026');          // Change this if auth enabled
define('NEONFS_ROOT_PATH', '/');                // Base path - can restrict to specific dir
define('NEONFS_MAX_UPLOAD_SIZE', 100);          // MB
define('NEONFS_SAFE_MODE', false);              // Disable delete/chmod operations
define('NEONFS_BLACKLIST', [                    // Forbidden paths
    '/etc/shadow',
    '/etc/passwd',
    '/root/.ssh'
]);

// Session setup
session_start();

// Initialize authentication check
if (NEONFS_REQUIRE_AUTH && !isset($_SESSION['neonfs_authenticated'])) {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['password'])) {
        if ($_POST['password'] === NEONFS_PASSWORD) {
            $_SESSION['neonfs_authenticated'] = true;
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        } else {
            $authError = 'Invalid password';
        }
    }

    // Show login screen
    if (!isset($_SESSION['neonfs_authenticated'])) {
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title>🌌 NeonFS - Authentication</title>
            <style>
                body {
                    margin: 0;
                    padding: 0;
                    background: linear-gradient(135deg, #0a0e27 0%, #1a1f3a 100%);
                    font-family: 'Courier New', monospace;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    min-height: 100vh;
                    color: #e0e0ff;
                }
                .login-box {
                    background: rgba(26, 31, 58, 0.95);
                    border: 2px solid #00eaff;
                    border-radius: 8px;
                    padding: 40px;
                    box-shadow: 0 0 30px rgba(0, 234, 255, 0.5);
                    text-align: center;
                    min-width: 300px;
                }
                .login-box h1 {
                    margin: 0 0 30px 0;
                    font-size: 32px;
                    text-shadow: 0 0 20px rgba(0, 234, 255, 0.8);
                }
                .login-box input {
                    width: 100%;
                    padding: 12px;
                    margin: 10px 0;
                    background: rgba(10, 14, 39, 0.8);
                    border: 1px solid #00eaff;
                    border-radius: 4px;
                    color: #e0e0ff;
                    font-family: 'Courier New', monospace;
                    font-size: 14px;
                    box-sizing: border-box;
                }
                .login-box button {
                    width: 100%;
                    padding: 12px;
                    margin: 20px 0 0 0;
                    background: linear-gradient(135deg, #00eaff, #00a8cc);
                    border: none;
                    border-radius: 4px;
                    color: #0a0e27;
                    font-family: 'Courier New', monospace;
                    font-size: 14px;
                    font-weight: bold;
                    cursor: pointer;
                    transition: all 0.3s;
                }
                .login-box button:hover {
                    box-shadow: 0 0 20px rgba(0, 234, 255, 0.8);
                    transform: translateY(-2px);
                }
                .error {
                    color: #ff00ea;
                    margin: 10px 0;
                    font-size: 12px;
                }
            </style>
        </head>
        <body>
            <div class="login-box">
                <h1>🌌 NeonFS</h1>
                <form method="POST">
                    <input type="password" name="password" placeholder="Enter password..." autofocus>
                    <?php if (isset($authError)): ?>
                        <div class="error"><?= htmlspecialchars($authError) ?></div>
                    <?php endif; ?>
                    <button type="submit">ENTER THE GRID</button>
                </form>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// 🛡️ SECTION 2: SECURITY - The Guardian
// ═══════════════════════════════════════════════════════════════════════════

/**
 * Validate that a path is safe to access
 */
function isPathSafe($path) {
    // Resolve to real path
    $realPath = @realpath($path);
    if ($realPath === false) {
        // Path doesn't exist yet (new folder creation)
        $realPath = $path;
    }

    $rootPath = realpath(NEONFS_ROOT_PATH);

    // Must be within root
    if (strpos($realPath, $rootPath) !== 0) {
        return false;
    }

    // Check blacklist
    foreach (NEONFS_BLACKLIST as $blocked) {
        if (strpos($realPath, $blocked) === 0) {
            return false;
        }
    }

    return true;
}

/**
 * Sanitize filename
 */
function sanitizeFilename($filename) {
    // Remove path separators
    $filename = str_replace(['/', '\\'], '', $filename);
    // Remove null bytes
    $filename = str_replace("\0", '', $filename);
    // Remove leading dots (hidden files protection)
    $filename = ltrim($filename, '.');

    return $filename;
}

/**
 * Check if safe mode blocks an operation
 */
function checkSafeMode($operation) {
    if (NEONFS_SAFE_MODE && in_array($operation, ['delete', 'chmod'])) {
        return ['success' => false, 'message' => 'Operation disabled in safe mode'];
    }
    return null;
}

// ═══════════════════════════════════════════════════════════════════════════
// 🔧 SECTION 3: FILE OPERATIONS - The Manipulator
// ═══════════════════════════════════════════════════════════════════════════

/**
 * Get file icon based on extension
 */
function getFileIcon($filename, $isDir) {
    if ($isDir) return '📁';

    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

    $iconMap = [
        // Images
        'jpg' => '🖼️', 'jpeg' => '🖼️', 'png' => '🖼️', 'gif' => '🖼️', 'svg' => '🖼️', 'webp' => '🖼️',
        // Code
        'php' => '💻', 'js' => '💻', 'py' => '💻', 'java' => '💻', 'cpp' => '💻', 'c' => '💻', 'h' => '💻',
        'html' => '💻', 'css' => '💻', 'ts' => '💻', 'jsx' => '💻', 'vue' => '💻',
        // Text
        'txt' => '📄', 'md' => '📄', 'log' => '📄',
        // Documents
        'pdf' => '📝', 'doc' => '📝', 'docx' => '📝', 'odt' => '📝',
        // Archives
        'zip' => '📦', 'tar' => '📦', 'gz' => '📦', 'rar' => '📦', '7z' => '📦',
        // Video
        'mp4' => '🎬', 'avi' => '🎬', 'mkv' => '🎬', 'mov' => '🎬', 'wmv' => '🎬',
        // Audio
        'mp3' => '🎵', 'wav' => '🎵', 'flac' => '🎵', 'ogg' => '🎵',
        // Database
        'db' => '🗄️', 'sql' => '🗄️', 'sqlite' => '🗄️',
        // Config
        'conf' => '⚙️', 'ini' => '⚙️', 'yaml' => '⚙️', 'yml' => '⚙️', 'json' => '⚙️', 'xml' => '⚙️',
    ];

    return $iconMap[$ext] ?? '📄';
}

/**
 * Format file size
 */
function formatSize($bytes) {
    if ($bytes === 0) return '0 B';

    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $factor = floor(log($bytes) / log(1024));

    return sprintf("%.2f %s", $bytes / pow(1024, $factor), $units[$factor]);
}

/**
 * Format time ago
 */
function timeAgo($timestamp) {
    $diff = time() - $timestamp;

    if ($diff < 60) return 'just now';
    if ($diff < 3600) return floor($diff / 60) . 'm ago';
    if ($diff < 86400) return floor($diff / 3600) . 'h ago';
    if ($diff < 604800) return floor($diff / 86400) . 'd ago';
    if ($diff < 2592000) return floor($diff / 604800) . 'w ago';

    return date('Y-m-d', $timestamp);
}

/**
 * Scan directory and return file list
 */
function scanDirectory($path) {
    if (!isPathSafe($path)) {
        return ['success' => false, 'message' => 'Access denied'];
    }

    if (!is_dir($path)) {
        return ['success' => false, 'message' => 'Not a directory'];
    }

    $items = [];
    $files = @scandir($path);

    if ($files === false) {
        return ['success' => false, 'message' => 'Permission denied'];
    }

    foreach ($files as $file) {
        if ($file === '.') continue;

        $fullPath = $path . DIRECTORY_SEPARATOR . $file;
        $isDir = is_dir($fullPath);

        // Get file info
        $stat = @stat($fullPath);

        $items[] = [
            'name' => $file,
            'path' => $fullPath,
            'isDir' => $isDir,
            'icon' => getFileIcon($file, $isDir),
            'size' => $isDir ? null : ($stat['size'] ?? 0),
            'sizeFormatted' => $isDir ? '-' : formatSize($stat['size'] ?? 0),
            'modified' => $stat['mtime'] ?? 0,
            'modifiedFormatted' => timeAgo($stat['mtime'] ?? 0),
            'permissions' => substr(sprintf('%o', $stat['mode'] ?? 0), -4),
            'isParent' => $file === '..'
        ];
    }

    // Sort: directories first, then by name
    usort($items, function($a, $b) {
        if ($a['isParent']) return -1;
        if ($b['isParent']) return 1;
        if ($a['isDir'] !== $b['isDir']) return $b['isDir'] - $a['isDir'];
        return strcasecmp($a['name'], $b['name']);
    });

    // Get disk space
    $freeSpace = @disk_free_space($path);
    $totalSpace = @disk_total_space($path);

    return [
        'success' => true,
        'items' => $items,
        'path' => $path,
        'freeSpace' => $freeSpace ? formatSize($freeSpace) : 'unknown',
        'totalSpace' => $totalSpace ? formatSize($totalSpace) : 'unknown',
        'itemCount' => count($items) - 1 // Exclude parent
    ];
}

/**
 * Delete file or directory recursively
 */
function deleteItem($path) {
    if ($safeCheck = checkSafeMode('delete')) return $safeCheck;
    if (!isPathSafe($path)) return ['success' => false, 'message' => 'Access denied'];

    if (!file_exists($path)) {
        return ['success' => false, 'message' => 'File not found'];
    }

    if (is_dir($path)) {
        $files = @scandir($path);
        if ($files === false) {
            return ['success' => false, 'message' => 'Permission denied'];
        }

        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;

            $fullPath = $path . DIRECTORY_SEPARATOR . $file;
            $result = deleteItem($fullPath);

            if (!$result['success']) {
                return $result;
            }
        }

        if (!@rmdir($path)) {
            return ['success' => false, 'message' => 'Failed to delete directory'];
        }
    } else {
        if (!@unlink($path)) {
            return ['success' => false, 'message' => 'Failed to delete file'];
        }
    }

    return ['success' => true];
}

/**
 * Copy file or directory recursively
 */
function copyItem($source, $destination) {
    if (!isPathSafe($source) || !isPathSafe($destination)) {
        return ['success' => false, 'message' => 'Access denied'];
    }

    if (!file_exists($source)) {
        return ['success' => false, 'message' => 'Source not found'];
    }

    if (is_dir($source)) {
        if (!@mkdir($destination, 0755, true)) {
            return ['success' => false, 'message' => 'Failed to create directory'];
        }

        $files = @scandir($source);
        if ($files === false) {
            return ['success' => false, 'message' => 'Permission denied'];
        }

        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;

            $srcPath = $source . DIRECTORY_SEPARATOR . $file;
            $dstPath = $destination . DIRECTORY_SEPARATOR . $file;

            $result = copyItem($srcPath, $dstPath);
            if (!$result['success']) {
                return $result;
            }
        }
    } else {
        if (!@copy($source, $destination)) {
            return ['success' => false, 'message' => 'Failed to copy file'];
        }
    }

    return ['success' => true];
}

// ═══════════════════════════════════════════════════════════════════════════
// 🌐 SECTION 4: API ENDPOINTS - The Gateway
// ═══════════════════════════════════════════════════════════════════════════

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action'])) {
    header('Content-Type: application/json');

    $action = $_GET['action'];
    $response = ['success' => false, 'message' => 'Unknown action'];

    try {
        switch ($action) {
            case 'list':
                $path = $_POST['path'] ?? NEONFS_ROOT_PATH;
                $response = scanDirectory($path);
                break;

            case 'rename':
                $path = $_POST['path'] ?? '';
                $newName = sanitizeFilename($_POST['newName'] ?? '');

                if (!isPathSafe($path)) {
                    $response = ['success' => false, 'message' => 'Access denied'];
                } elseif (empty($newName)) {
                    $response = ['success' => false, 'message' => 'Invalid name'];
                } else {
                    $dir = dirname($path);
                    $newPath = $dir . DIRECTORY_SEPARATOR . $newName;

                    if (file_exists($newPath)) {
                        $response = ['success' => false, 'message' => 'File already exists'];
                    } elseif (@rename($path, $newPath)) {
                        $response = ['success' => true, 'message' => 'Renamed successfully'];
                    } else {
                        $response = ['success' => false, 'message' => 'Rename failed'];
                    }
                }
                break;

            case 'delete':
                $paths = json_decode($_POST['paths'] ?? '[]', true);
                $deleted = 0;

                foreach ($paths as $path) {
                    $result = deleteItem($path);
                    if ($result['success']) {
                        $deleted++;
                    } else {
                        $response = $result;
                        break;
                    }
                }

                if ($deleted > 0) {
                    $response = ['success' => true, 'deleted' => $deleted];
                }
                break;

            case 'copy':
                $paths = json_decode($_POST['paths'] ?? '[]', true);
                $destination = $_POST['destination'] ?? '';
                $copied = 0;

                foreach ($paths as $path) {
                    $filename = basename($path);
                    $destPath = $destination . DIRECTORY_SEPARATOR . $filename;

                    // Handle duplicates
                    $counter = 1;
                    $origDestPath = $destPath;
                    while (file_exists($destPath)) {
                        $info = pathinfo($origDestPath);
                        $destPath = $info['dirname'] . DIRECTORY_SEPARATOR .
                                   $info['filename'] . " ($counter)" .
                                   (isset($info['extension']) ? '.' . $info['extension'] : '');
                        $counter++;
                    }

                    $result = copyItem($path, $destPath);
                    if ($result['success']) {
                        $copied++;
                    } else {
                        $response = $result;
                        break;
                    }
                }

                if ($copied > 0) {
                    $response = ['success' => true, 'copied' => $copied];
                }
                break;

            case 'newFolder':
                $path = $_POST['path'] ?? '';
                $name = sanitizeFilename($_POST['name'] ?? '');

                if (!isPathSafe($path)) {
                    $response = ['success' => false, 'message' => 'Access denied'];
                } elseif (empty($name)) {
                    $response = ['success' => false, 'message' => 'Invalid name'];
                } else {
                    $newPath = $path . DIRECTORY_SEPARATOR . $name;

                    if (file_exists($newPath)) {
                        $response = ['success' => false, 'message' => 'Folder already exists'];
                    } elseif (@mkdir($newPath, 0755)) {
                        $response = ['success' => true, 'message' => 'Folder created'];
                    } else {
                        $response = ['success' => false, 'message' => 'Failed to create folder'];
                    }
                }
                break;

            case 'upload':
                $path = $_POST['path'] ?? '';

                if (!isPathSafe($path)) {
                    $response = ['success' => false, 'message' => 'Access denied'];
                } elseif (!isset($_FILES['files'])) {
                    $response = ['success' => false, 'message' => 'No files uploaded'];
                } else {
                    $uploaded = 0;
                    $files = $_FILES['files'];

                    $fileCount = is_array($files['name']) ? count($files['name']) : 1;

                    for ($i = 0; $i < $fileCount; $i++) {
                        $filename = is_array($files['name']) ? $files['name'][$i] : $files['name'];
                        $tmpName = is_array($files['tmp_name']) ? $files['tmp_name'][$i] : $files['tmp_name'];
                        $error = is_array($files['error']) ? $files['error'][$i] : $files['error'];

                        if ($error === UPLOAD_ERR_OK) {
                            $filename = sanitizeFilename($filename);
                            $destPath = $path . DIRECTORY_SEPARATOR . $filename;

                            if (@move_uploaded_file($tmpName, $destPath)) {
                                $uploaded++;
                            }
                        }
                    }

                    $response = ['success' => true, 'uploaded' => $uploaded];
                }
                break;

            case 'download':
                $paths = json_decode($_POST['paths'] ?? '[]', true);

                if (count($paths) === 1 && is_file($paths[0])) {
                    // Single file download
                    $file = $paths[0];
                    if (isPathSafe($file) && file_exists($file)) {
                        header('Content-Type: application/octet-stream');
                        header('Content-Disposition: attachment; filename="' . basename($file) . '"');
                        header('Content-Length: ' . filesize($file));
                        readfile($file);
                        exit;
                    }
                } else {
                    // Multiple files - create ZIP
                    $zipFile = sys_get_temp_dir() . '/neonfs_' . uniqid() . '.zip';
                    $zip = new ZipArchive();

                    if ($zip->open($zipFile, ZipArchive::CREATE) === true) {
                        foreach ($paths as $path) {
                            if (isPathSafe($path) && file_exists($path)) {
                                if (is_dir($path)) {
                                    addDirToZip($zip, $path, basename($path));
                                } else {
                                    $zip->addFile($path, basename($path));
                                }
                            }
                        }
                        $zip->close();

                        header('Content-Type: application/zip');
                        header('Content-Disposition: attachment; filename="neonfs_download.zip"');
                        header('Content-Length: ' . filesize($zipFile));
                        readfile($zipFile);
                        unlink($zipFile);
                        exit;
                    }
                }

                $response = ['success' => false, 'message' => 'Download failed'];
                break;
        }
    } catch (Exception $e) {
        $response = ['success' => false, 'message' => $e->getMessage()];
    }

    echo json_encode($response);
    exit;
}

/**
 * Helper to add directory to ZIP recursively
 */
function addDirToZip($zip, $dir, $zipPath) {
    $files = @scandir($dir);
    if ($files === false) return;

    foreach ($files as $file) {
        if ($file === '.' || $file === '..') continue;

        $fullPath = $dir . DIRECTORY_SEPARATOR . $file;
        $zipFilePath = $zipPath . '/' . $file;

        if (is_dir($fullPath)) {
            $zip->addEmptyDir($zipFilePath);
            addDirToZip($zip, $fullPath, $zipFilePath);
        } else {
            $zip->addFile($fullPath, $zipFilePath);
        }
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// 🎨 SECTION 5: HTML TEMPLATE - The Interface
// ═══════════════════════════════════════════════════════════════════════════
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🌌 NeonFS - File Manager</title>
    <style>
        /* ═══════════════════════════════════════════════════════════════
           🎨 CSS VARIABLES - The Palette
           ═══════════════════════════════════════════════════════════════ */
        :root {
            --bg-primary: #0a0e27;
            --bg-secondary: #1a1f3a;
            --bg-elevated: #252b4a;
            --text-primary: #e0e0ff;
            --text-secondary: #a0a0c0;
            --accent-cyan: #00eaff;
            --accent-magenta: #ff00ea;
            --border-color: #00eaff33;
            --hover-bg: #00eaff1a;
            --success: #00ff88;
            --error: #ff0055;
            --warning: #ffaa00;
        }

        [data-theme="light"] {
            --bg-primary: #f5f5ff;
            --bg-secondary: #ffffff;
            --bg-elevated: #fafafe;
            --text-primary: #1a1f3a;
            --text-secondary: #505070;
            --accent-cyan: #0099cc;
            --accent-magenta: #cc0099;
            --border-color: #0099cc33;
            --hover-bg: #0099cc1a;
        }

        /* ═══════════════════════════════════════════════════════════════
           🌌 BASE STYLES - The Foundation
           ═══════════════════════════════════════════════════════════════ */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Courier New', 'Consolas', monospace;
            background: var(--bg-primary);
            color: var(--text-primary);
            line-height: 1.6;
            font-size: var(--font-size, 14px);
            transition: background 0.3s, color 0.3s;
        }

        /* ═══════════════════════════════════════════════════════════════
           🎯 HEADER - The Crown
           ═══════════════════════════════════════════════════════════════ */
        .header {
            background: var(--bg-secondary);
            border-bottom: 2px solid var(--accent-cyan);
            padding: 16px 24px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 16px;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 0 20px rgba(0, 234, 255, 0.3);
        }

        .logo {
            font-size: 24px;
            font-weight: bold;
            color: var(--accent-cyan);
            text-shadow: 0 0 10px rgba(0, 234, 255, 0.8);
            white-space: nowrap;
        }

        .search-box {
            flex: 1;
            max-width: 400px;
        }

        .search-box input {
            width: 100%;
            padding: 8px 12px;
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            color: var(--text-primary);
            font-family: inherit;
            font-size: inherit;
            transition: all 0.2s;
        }

        .search-box input:focus {
            outline: none;
            border-color: var(--accent-cyan);
            box-shadow: 0 0 10px rgba(0, 234, 255, 0.3);
        }

        .toolbar {
            display: flex;
            gap: 8px;
            align-items: center;
        }

        .btn {
            padding: 8px 16px;
            background: var(--bg-elevated);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            color: var(--text-primary);
            cursor: pointer;
            font-family: inherit;
            font-size: inherit;
            transition: all 0.2s;
            white-space: nowrap;
        }

        .btn:hover {
            background: var(--hover-bg);
            border-color: var(--accent-cyan);
            box-shadow: 0 0 10px rgba(0, 234, 255, 0.3);
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--accent-cyan), #00a8cc);
            border: none;
            color: var(--bg-primary);
            font-weight: bold;
        }

        .btn-primary:hover {
            box-shadow: 0 0 15px rgba(0, 234, 255, 0.6);
            transform: translateY(-1px);
        }

        /* ═══════════════════════════════════════════════════════════════
           🗺️ BREADCRUMBS - The Navigator
           ═══════════════════════════════════════════════════════════════ */
        .breadcrumbs {
            padding: 16px 24px;
            background: var(--bg-secondary);
            border-bottom: 1px solid var(--border-color);
            display: flex;
            align-items: center;
            gap: 8px;
            overflow-x: auto;
        }

        .breadcrumb {
            color: var(--accent-cyan);
            cursor: pointer;
            transition: all 0.2s;
            white-space: nowrap;
        }

        .breadcrumb:hover {
            text-shadow: 0 0 10px rgba(0, 234, 255, 0.8);
        }

        .breadcrumb-separator {
            color: var(--text-secondary);
        }

        /* ═══════════════════════════════════════════════════════════════
           📋 FILE LIST - The Matrix
           ═══════════════════════════════════════════════════════════════ */
        .container {
            padding: 24px;
        }

        .file-list {
            background: var(--bg-secondary);
            border: 1px solid var(--border-color);
            border-radius: 8px;
            overflow: hidden;
        }

        .file-list-header {
            display: grid;
            grid-template-columns: 40px 1fr 120px 120px 100px;
            gap: 16px;
            padding: 12px 16px;
            background: var(--bg-elevated);
            border-bottom: 1px solid var(--border-color);
            font-weight: bold;
            color: var(--accent-cyan);
        }

        .file-item {
            display: grid;
            grid-template-columns: 40px 1fr 120px 120px 100px;
            gap: 16px;
            padding: 12px 16px;
            border-bottom: 1px solid var(--border-color);
            cursor: pointer;
            transition: all 0.2s;
        }

        .file-item:hover {
            background: var(--hover-bg);
        }

        .file-item.selected {
            background: var(--accent-cyan);
            color: var(--bg-primary);
        }

        .file-icon {
            font-size: 20px;
            text-align: center;
        }

        .file-name {
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .file-actions {
            display: flex;
            gap: 8px;
        }

        .action-btn {
            padding: 4px 8px;
            font-size: 12px;
            opacity: 0.7;
        }

        .action-btn:hover {
            opacity: 1;
        }

        /* ═══════════════════════════════════════════════════════════════
           🎭 MODAL - The Portal
           ═══════════════════════════════════════════════════════════════ */
        .modal-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(10, 14, 39, 0.95);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }

        .modal-overlay.active {
            display: flex;
        }

        .modal {
            background: var(--bg-secondary);
            border: 2px solid var(--accent-cyan);
            border-radius: 8px;
            padding: 24px;
            max-width: 500px;
            width: 90%;
            box-shadow: 0 0 30px rgba(0, 234, 255, 0.5);
        }

        .modal-header {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 16px;
            color: var(--accent-cyan);
        }

        .modal-body {
            margin-bottom: 16px;
        }

        .modal-footer {
            display: flex;
            gap: 8px;
            justify-content: flex-end;
        }

        .form-group {
            margin-bottom: 16px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: var(--text-secondary);
        }

        .form-group input {
            width: 100%;
            padding: 8px 12px;
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 4px;
            color: var(--text-primary);
            font-family: inherit;
            font-size: inherit;
        }

        /* ═══════════════════════════════════════════════════════════════
           🎬 LOADING & ANIMATIONS - The Magic
           ═══════════════════════════════════════════════════════════════ */
        .loading {
            display: none;
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            z-index: 2000;
        }

        .loading.active {
            display: block;
        }

        .spinner {
            width: 60px;
            height: 60px;
            border: 4px solid var(--border-color);
            border-top-color: var(--accent-cyan);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* ═══════════════════════════════════════════════════════════════
           📱 RESPONSIVE - The Adapter
           ═══════════════════════════════════════════════════════════════ */
        @media (max-width: 768px) {
            .file-list-header,
            .file-item {
                grid-template-columns: 40px 1fr 80px;
            }

            .file-list-header > :nth-child(3),
            .file-list-header > :nth-child(4),
            .file-item > :nth-child(3),
            .file-item > :nth-child(4) {
                display: none;
            }
        }

        /* ═══════════════════════════════════════════════════════════════
           🎨 UTILITIES - The Helpers
           ═══════════════════════════════════════════════════════════════ */
        .hidden {
            display: none !important;
        }

        .text-success {
            color: var(--success);
        }

        .text-error {
            color: var(--error);
        }

        .text-warning {
            color: var(--warning);
        }
    </style>
</head>
<body>
    <!-- 🌌 The Crown - Header -->
    <div class="header">
        <div class="logo">🌌 NeonFS</div>
        <div class="search-box">
            <input type="text" id="searchInput" placeholder="🔍 Search files...">
        </div>
        <div class="toolbar">
            <button class="btn" id="btnFontDecrease" title="Decrease font size">A-</button>
            <button class="btn" id="btnFontIncrease" title="Increase font size">A+</button>
            <button class="btn" id="btnThemeToggle" title="Toggle theme">☀️</button>
            <button class="btn btn-primary" id="btnNewFolder" title="New folder">+ Folder</button>
            <button class="btn btn-primary" id="btnUpload" title="Upload files">⬆️ Upload</button>
        </div>
    </div>

    <!-- 🗺️ The Navigator - Breadcrumbs -->
    <div class="breadcrumbs" id="breadcrumbs">
        <span class="breadcrumb" data-path="/">/</span>
    </div>

    <!-- 📋 The Matrix - File List -->
    <div class="container">
        <div class="file-list">
            <div class="file-list-header">
                <div><input type="checkbox" id="selectAll"></div>
                <div>Name</div>
                <div>Size</div>
                <div>Modified</div>
                <div>Actions</div>
            </div>
            <div id="fileListBody">
                <!-- Files will be inserted here by JavaScript -->
            </div>
        </div>
    </div>

    <!-- Hidden file input for uploads -->
    <input type="file" id="fileInput" multiple style="display: none">

    <!-- 🎭 The Portal - Modals -->
    <div class="modal-overlay" id="modalOverlay">
        <div class="modal" id="modal">
            <div class="modal-header" id="modalHeader">Modal Title</div>
            <div class="modal-body" id="modalBody">Modal content</div>
            <div class="modal-footer">
                <button class="btn" id="modalCancel">Cancel</button>
                <button class="btn btn-primary" id="modalConfirm">Confirm</button>
            </div>
        </div>
    </div>

    <!-- 🎬 The Magic - Loading -->
    <div class="loading" id="loading">
        <div class="spinner"></div>
    </div>

    <script>
    // ═══════════════════════════════════════════════════════════════════════════
    // 🧠 SECTION 6: JAVASCRIPT - The Mind
    // ═══════════════════════════════════════════════════════════════════════════

    (function() {
        'use strict';

        // 🔮 State Management
        const state = {
            currentPath: <?= json_encode(NEONFS_ROOT_PATH) ?>,
            selectedItems: new Set(),
            theme: localStorage.getItem('neonfs_theme') || 'dark',
            fontSize: parseInt(localStorage.getItem('neonfs_font_size')) || 14,
            files: []
        };

        // 🎨 Initialize theme and font size
        document.documentElement.setAttribute('data-theme', state.theme);
        document.documentElement.style.setProperty('--font-size', state.fontSize + 'px');
        updateThemeIcon();

        // 🌌 Load initial directory
        loadDirectory(state.currentPath);

        // ═══════════════════════════════════════════════════════════════
        // 🔄 Core Functions
        // ═══════════════════════════════════════════════════════════════

        function loadDirectory(path) {
            showLoading();

            fetch('?action=list', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'path=' + encodeURIComponent(path)
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (!data.success) {
                    alert('Error: ' + data.message);
                    return;
                }

                state.currentPath = data.path;
                state.files = data.items;
                state.selectedItems.clear();

                renderFileList(data.items);
                renderBreadcrumbs(data.path);
            })
            .catch(err => {
                hideLoading();
                alert('Error loading directory: ' + err.message);
            });
        }

        function renderFileList(items) {
            const container = document.getElementById('fileListBody');
            container.innerHTML = '';

            items.forEach(item => {
                if (item.isParent) return; // Skip parent directory in main list

                const row = document.createElement('div');
                row.className = 'file-item';
                row.dataset.path = item.path;

                row.innerHTML = `
                    <div><input type="checkbox" class="item-checkbox" data-path="${item.path}"></div>
                    <div class="file-name">
                        <span class="file-icon">${item.icon}</span>
                        <span>${escapeHtml(item.name)}</span>
                    </div>
                    <div>${item.sizeFormatted}</div>
                    <div>${item.modifiedFormatted}</div>
                    <div class="file-actions">
                        <button class="btn action-btn" data-action="rename" data-path="${item.path}">✏️</button>
                        <button class="btn action-btn" data-action="delete" data-path="${item.path}">🗑️</button>
                        <button class="btn action-btn" data-action="download" data-path="${item.path}">⬇️</button>
                    </div>
                `;

                // Double-click to open directory
                row.addEventListener('dblclick', () => {
                    if (item.isDir) {
                        loadDirectory(item.path);
                    }
                });

                container.appendChild(row);
            });
        }

        function renderBreadcrumbs(path) {
            const container = document.getElementById('breadcrumbs');
            container.innerHTML = '';

            const parts = path.split('/').filter(p => p);
            let currentPath = '';

            // Root
            const root = document.createElement('span');
            root.className = 'breadcrumb';
            root.textContent = '/';
            root.dataset.path = '/';
            root.addEventListener('click', () => loadDirectory('/'));
            container.appendChild(root);

            // Path segments
            parts.forEach((part, index) => {
                currentPath += '/' + part;

                const sep = document.createElement('span');
                sep.className = 'breadcrumb-separator';
                sep.textContent = ' > ';
                container.appendChild(sep);

                const crumb = document.createElement('span');
                crumb.className = 'breadcrumb';
                crumb.textContent = part;
                crumb.dataset.path = currentPath;
                crumb.addEventListener('click', function() {
                    loadDirectory(this.dataset.path);
                });
                container.appendChild(crumb);
            });
        }

        // ═══════════════════════════════════════════════════════════════
        // 🎯 File Operations
        // ═══════════════════════════════════════════════════════════════

        function deleteItems(paths) {
            if (!confirm(`Delete ${paths.length} item(s)?`)) return;

            showLoading();

            fetch('?action=delete', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'paths=' + encodeURIComponent(JSON.stringify(paths))
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    loadDirectory(state.currentPath);
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(err => {
                hideLoading();
                alert('Error: ' + err.message);
            });
        }

        function renameItem(path) {
            const oldName = path.split('/').pop();
            const newName = prompt('Rename to:', oldName);

            if (!newName || newName === oldName) return;

            showLoading();

            fetch('?action=rename', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'path=' + encodeURIComponent(path) + '&newName=' + encodeURIComponent(newName)
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    loadDirectory(state.currentPath);
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(err => {
                hideLoading();
                alert('Error: ' + err.message);
            });
        }

        function downloadItems(paths) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '?action=download';

            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'paths';
            input.value = JSON.stringify(paths);

            form.appendChild(input);
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }

        function newFolder() {
            const name = prompt('Folder name:');
            if (!name) return;

            showLoading();

            fetch('?action=newFolder', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'path=' + encodeURIComponent(state.currentPath) + '&name=' + encodeURIComponent(name)
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    loadDirectory(state.currentPath);
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(err => {
                hideLoading();
                alert('Error: ' + err.message);
            });
        }

        function uploadFiles(files) {
            const formData = new FormData();
            formData.append('path', state.currentPath);

            for (let i = 0; i < files.length; i++) {
                formData.append('files[]', files[i]);
            }

            showLoading();

            fetch('?action=upload', {
                method: 'POST',
                body: formData
            })
            .then(r => r.json())
            .then(data => {
                hideLoading();

                if (data.success) {
                    loadDirectory(state.currentPath);
                    alert(`Uploaded ${data.uploaded} file(s)`);
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(err => {
                hideLoading();
                alert('Error: ' + err.message);
            });
        }

        // ═══════════════════════════════════════════════════════════════
        // 🎨 UI Functions
        // ═══════════════════════════════════════════════════════════════

        function showLoading() {
            document.getElementById('loading').classList.add('active');
        }

        function hideLoading() {
            document.getElementById('loading').classList.remove('active');
        }

        function updateThemeIcon() {
            const btn = document.getElementById('btnThemeToggle');
            btn.textContent = state.theme === 'dark' ? '☀️' : '🌙';
        }

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        // ═══════════════════════════════════════════════════════════════
        // 🎯 Event Listeners
        // ═══════════════════════════════════════════════════════════════

        // Theme toggle
        document.getElementById('btnThemeToggle').addEventListener('click', () => {
            state.theme = state.theme === 'dark' ? 'light' : 'dark';
            document.documentElement.setAttribute('data-theme', state.theme);
            localStorage.setItem('neonfs_theme', state.theme);
            updateThemeIcon();
        });

        // Font size controls
        document.getElementById('btnFontIncrease').addEventListener('click', () => {
            if (state.fontSize < 24) {
                state.fontSize += 2;
                document.documentElement.style.setProperty('--font-size', state.fontSize + 'px');
                localStorage.setItem('neonfs_font_size', state.fontSize);
            }
        });

        document.getElementById('btnFontDecrease').addEventListener('click', () => {
            if (state.fontSize > 12) {
                state.fontSize -= 2;
                document.documentElement.style.setProperty('--font-size', state.fontSize + 'px');
                localStorage.setItem('neonfs_font_size', state.fontSize);
            }
        });

        // New folder
        document.getElementById('btnNewFolder').addEventListener('click', newFolder);

        // Upload
        document.getElementById('btnUpload').addEventListener('click', () => {
            document.getElementById('fileInput').click();
        });

        document.getElementById('fileInput').addEventListener('change', function() {
            if (this.files.length > 0) {
                uploadFiles(this.files);
                this.value = ''; // Reset
            }
        });

        // Search/filter
        document.getElementById('searchInput').addEventListener('input', function() {
            const query = this.value.toLowerCase();
            const items = document.querySelectorAll('.file-item');

            items.forEach(item => {
                const name = item.querySelector('.file-name span:last-child').textContent.toLowerCase();
                item.style.display = name.includes(query) ? '' : 'none';
            });
        });

        // File action buttons (delegated)
        document.getElementById('fileListBody').addEventListener('click', (e) => {
            const btn = e.target.closest('[data-action]');
            if (!btn) return;

            const action = btn.dataset.action;
            const path = btn.dataset.path;

            switch (action) {
                case 'rename':
                    renameItem(path);
                    break;
                case 'delete':
                    deleteItems([path]);
                    break;
                case 'download':
                    downloadItems([path]);
                    break;
            }
        });

        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => {
            // Delete key
            if (e.key === 'Delete' && state.selectedItems.size > 0) {
                deleteItems(Array.from(state.selectedItems));
            }
        });

        console.log('🌌 NeonFS initialized - The blueprint was never lost...');
    })();
    </script>
</body>
</html>
