# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

**iaCase** is a PHP-based database-driven CRUD framework that automatically generates web interfaces (forms, grids, reports) from MySQL database metadata. Built circa 2012-2016, it uses jQuery, jqGrid, and jQuery UI for the frontend.

## Development Commands

There is no formal build system. The framework is self-contained PHP with no npm/composer dependencies.

- **Test a table handler:** Access `/ia_case_test.php?test=table_name`
- **Configuration:** Edit `inc/config.php` for database credentials and paths
- **Apache:** `.htaccess` configures `Options +Indexes`
- **Transfer files:** Access `/transfer_files.php` to copy generated apps between directories

## Architecture

### Request Flow

1. Entry point loads `inc/config.php` (DB settings, paths)
2. Initialize `appRelate` instance (database metadata configuration)
3. Create `iacase` instance for the target table
4. Call `process_action()` which routes based on the `h` parameter

### Core Components

| File | Purpose |
|------|---------|
| `inc/iacase.php` (242KB) | Main CRUD controller - handles all create/read/update/delete operations |
| `inc/appRelateBase.php` | Base class for database metadata, table definitions, field types, relationships |
| `inc/ia_utilerias.php` (109KB) | SQL builders (`ia_update()`, `ia_insert()`, `ia_sqlArray()`), escaping functions (`strit()`, `dateit()`, `param()`) |
| `inc/iaJQGrid.php` (94KB) | jqGrid integration - colModel generation, pagination, filtering |
| `inc/iaHeader.php` | Smart JS/CSS dependency loader - only includes what's needed |
| `incprivate/table_do.php` | Table operation handlers |

### Ajax Endpoints

- `ajax/jqgrid_read.php` - Grid data provider
- `ajax/jqgrid_combobox.php` - Dropdown/select options
- `ajax/iacaddchld.php` - Child table CRUD operations
- `ajax/export.php` - CSV/data export

## Key Conventions

### Action Parameter (`h`)

The `h` URL parameter determines the operation:
- Empty `''` = list/grid view
- `a` = add new record form
- `e` = edit existing record
- `r` = read/view record
- `b` = delete confirmation
- `d` = execute delete
- `i` = insert (POST submission)
- `s` = save/update (POST submission)
- `qbe` = query by example search

### Global Variable Prefixes

- `$gIAsql`, `$gIApath`, `$gAppRelate`, `$gIaHeader` - Framework globals
- `ia` prefix - iaCase framework functions/classes
- `iap` prefix - Internal parameters

### Form Modes

Set via `iacase->modo`:
- `cardex` - Same window form
- `cardex_window` - Modal dialog form
- `jqgrid_iacform` - Inline jqGrid editing
- `jqgrid_form` - Separate form with jqGrid list

### Field Type Mapping

Database types automatically map to inputs:
- `varchar` → text input
- `int/float/decimal` → number input
- `datetime/date` → datepicker
- `enum/set` → select dropdown
- `text/longtext` → textarea
- `bit` → checkbox

### Permission System

Properties on `iacase`: `permiso_list`, `permiso_insert`, `permiso_update`, `permiso_delete`

Modes: TABLE, FIELD, RECORD, EDITORS, READERS

### SQL Helpers

Always use helper functions for database operations:
```php
$result = ia_update($table, $data_array, $where);
$id = ia_insert($table, $data_array);
$rows = ia_sqlArray($query);
$safe = strit($user_input);  // String escaping
$date = dateit($date_value); // Date formatting
```

### Generated Apps Path Configuration

The path to generated applications is configurable via global variable:

```php
// In inc/config.php (line 76)
$gIApath['GeneratedAppPath'] = '../quantix/';  // Default: quantix directory
```

**Benefits:**
- **Customizable per environment** - Point to different directories without code changes
- **Used by autoloader** - Automatically includes classes from the configured path
- **Used by test files** - All test scripts respect this configuration
- **Backward compatible** - Defaults to `'../quantix/'` if not explicitly set

**Example customization:**
```php
// Point to a different generated app directory
$gIApath['GeneratedAppPath'] = '../myapp/';
```

## CRUD Generation

Access `/ia_case.php` to generate CRUD interfaces. Configure:
- **DB**: Select target database
- **Create path**: Output directory for generated app files (e.g., `/lamp/www/quantix/`)
- **Hojas path**: Output directory for backoffice pages (e.g., `/lamp/www/quantix/backoffice/`)

**Auto-Path Magic:** When you select a database from the dropdown, the Create path and Hojas path fields automatically update to match the database name! Simply select your database and the paths fill themselves using the convention: database name = folder name.

Click "Create" to generate files for all tables in the selected database.

### Transferring Generated Apps

After generating a CRUD application, use `/transfer_files.php` to copy it between environments (e.g., from test to production).

**Features:**
- Preview mode - See what files will be copied without actually copying
- Session persistence - Remembers your origin/destination paths
- Exclusion list - Automatically skips `.git`, `.idea`, `.vscode`, `node_modules`, `__pycache__`, `.DS_Store`
- Path validation - Enforces `/lamp/www/` prefix and prevents directory traversal
- Statistics - Shows files copied, directories created, errors encountered
- Overwrite mode - Existing files in destination are overwritten

**Usage:**
1. Set **Origin** path (e.g., `/lamp/www/quantixTest/`)
2. Set **Destination** path (e.g., `/lamp/www/quantix/`)
3. Click **Preview** to see what will be copied (no files are modified)
4. Click **Copy** to perform the actual file transfer

**Action Parameter (`h`):**
- `preview` = Show files that would be copied without copying
- `copy` = Execute the file copy operation

### Generated Files Structure

For each table (e.g., `reservas_cloudbeds`), iaCase generates:

**1. `app/app_[tablename].php`** - Entity handler class
```php
class app_reservas_cloudbeds extends iacase_base {
    // Constructor sets table name, permissions, mode
    // campos_default() - Auto-generated field definitions from DB schema
    // campos_final() - Customize field order, attributes, display groups
    // Override hooks: insert_*, update_*, delete_*, forma_*, listme_*
}
```

**2. `backoffice/[tablename].php`** - Web page entry point
```php
$f = new app_reservas_cloudbeds();
$f->process_action();  // Route based on 'h' parameter
$f->toolbar_set();     // Setup toolbar
$f->process();         // Render form or grid
```

**3. `app/appRelate.php`** - Central configuration (one per app)
- `$tables` array - All table metadata, relationships, child tables
- `$enums` array - Enum/dropdown definitions
- `$links` array - Foreign key relationships for select boxes

### Field Definition Array (`$this->campos`)

Each field has attributes:
```php
'field_name' => [
    'title' => 'Label',
    'Type' => 'varchar|int|decimal|date|timestamp|enum',
    'maxlength' => '100',
    'required' => true|false,
    'modo' => 'R/W|R/O',        // Read-Write or Read-Only
    'display_group' => 'group', // Group related fields
    'formato' => 'email|url',   // Special formatting
    'PK' => true,               // Primary key
]
```

### Customization Hooks

Override these methods in `app_[tablename].php`:
- `campos_final()` - Modify field definitions, order, display groups
- `insert_validate()`, `update_validate()` - Custom validation
- `insert_pre_save()`, `update_pre_save()` - Pre-save logic
- `insert_post_save()`, `update_post_save()` - Post-save logic
- `forma_pre()`, `forma_post()` - Form rendering hooks
- `listme_pre()`, `listme_postRender()` - Grid rendering hooks
- `permiso_por_record()` - Record-level permissions

### Smart Field Order Preservation (campos_final + $orden)

**Important:** When you customize field order using `$orden=array(...)` in `campos_final()`, iaCase now **preserves your custom ordering** when regenerating files!

**How it works:**
- When you run "Create" in `/ia_case.php`, the generator extracts your existing `$orden` array
- New database fields are **intelligently inserted** at their schema position relative to existing fields
- Deleted database fields are **automatically removed** from `$orden`
- Your custom field arrangement is **fully preserved**

**Example:**
```php
// Your custom $orden in campos_final()
$orden = array('property', 'name', 'email', 'phone_number', ...);

// After adding 'propiedad_id' column to DB (between property and name)
// Then regenerating → iaCase produces:
$orden = array('property', 'propiedad_id', 'name', 'email', 'phone_number', ...);
// ✅ Custom order preserved, new field inserted intelligently!
```

**Benefits:**
- ✅ Safe to regenerate CRUD files after database schema changes
- ✅ Rearrange fields for better UX without fear of losing changes
- ✅ New fields automatically appear in logical positions
- ✅ No manual field list maintenance after schema updates

**Implementation:** See `smart_merge_field_order()` and `extract_existing_orden()` in `incprivate/table_do.php`

### Child Tables

Configure parent-child relationships:
```php
$this->childTables = ['child_table_name'];
$this->editChilds = true;   // Show child grids in forms
$this->saveChilds = true;   // Save children with parent
```

## Directory Structure

- `inc/` - Core framework libraries
- `incprivate/` - Application-specific private includes (table_do.php, table_create.php)
- `backoffice/` - Sample application template
- `ajax/` - AJAX endpoints
- `js/` - jQuery, jqGrid, jQuery UI libraries
- `css/` - Stylesheets and themes
- `template/` - Template/demo structure for new applications
- `copiar/` - Template files copied during generation
