<?php
/** @noinspection PhpMissingFieldTypeInspection */
/** @noinspection PhpMissingParamTypeInspection */
/** @noinspection PhpMissingReturnTypeInspection */

namespace Iac\inc\sql;

use function array_merge;
use function array_key_exists;
use function array_slice;
use function array_pop;
use function count;
use function microtime;
use function reset;
use function str_ireplace;
use Exception;
use JetBrains\PhpStorm\Pure;
use mysqli;
use mysqli_driver;
use mysqli_sql_exception;
use mysqli_stmt;

error_reporting(E_ALL);
ini_set('display_errors',1);
ini_set('memory_limit', -1);
ignore_user_abort(1);
ini_set('max_execution_time',0);
/* *
 * mysqli db interface and convience methods.
 *
 *
 * @package sql
 * @author Informatica Asociada
 * @copyright 2015
 * @version 1.0.0
 */



require_once( dirname(__FILE__) . "/IacStringer.php");

/**
 * mysqli db interface and convience methods.
 *
 * @version 1.0.0
 */
class IacMysqli {
    use IacStringer;

    /**
     * $mysqli_options
     *
     * @var array $mysqli_options, change before connecting, default: array( rray(MYSQLI_OPT_CONNECT_TIMEOUT, 5), array(MYSQLI_INIT_COMMAND, 'SET AUTOCOMMIT=1') ),
     *
     * @link http://php.net/manual/en/mysqli.options.php
     * @link http://php.net/manual/en/mysqli.constants.php
     * @access public
     */
    public $mysqli_options = array(
        array(MYSQLI_OPT_CONNECT_TIMEOUT, 5),
        array(MYSQLI_INIT_COMMAND, 'SET AUTOCOMMIT=1'),
    );

    /**
     * Character set for connection, usually set to the same as database's characterset.
     *
     * @var string $charset connection character set, defaults to ut8, change before connecting, default utf8
     * @access public
     */
    public $charset = 'utf8mb4';

    /**
     * $coalition
     * Coalition for conection, usually set to the same coallition as database's.
     *
     * @var string $coalition connection coalition defaults to ut8_general_ci, change before connecting, default: utf8_general_ci
     * @access public
     */
    public $coalition = 'utf8mb4_0900_ai_ci';

    /**
     * $metDataOn
     * On true stores field_info of last select statement
     *
     * @var bool $metaDataOn on true store field_info for each query, default false
     * @see IacMysqli::metaData_get() IacMysqli::metaData_get()
     * @access public
     */
    public $metaDataOn = false;

    /**
     * $traceOn
     * True stores sql commands.
     *
     * @var bool $traceOn on true keep log of sql commands issued, default false
     * @access public
     */
    public $traceOn = false;

    /**
     * $register_warnings
     * On true stores last 5 warnings.
     *
     * @var bool $register_warnings on true keep small log of sql warnings
     *
     * @access public
     */
    public $register_warnings = false;

    /**
     * $mysqli
     * php class mysqli for connection
     *
     * @var object $mysqli mysqli instance used
     * @access public
     */
    public $mysqli = /*.(mysqli).*/  null;

    /**
     * $affected_rows
     * Affected rows of last SQL statement (insert, update, delete, ...)
     *
     * @var int $affected_rows number of affected rows by last command
     * @access public
     */
    public $affected_rows=0;

    /**
     * $num_rows
     * Rows retreived in last select stament
     *
     * @var int $num_rows number of retreived rows by last command
     * @access public
    */
    public $num_rows;

    /**
     * $autocommitDefault
     * After a commit or rollback resets autocommit mode to this value
     *
     * @var bool $autocommitDefault default true. autocommit mode to set on connect, reconnect, commit & rollback
     * @access public
     */
    public $autocommitDefault = /*.(bool).*/ true;


    /**
     * $retries
     * Number of times to resend transaction (or single query outside a transaction) when instructed by server
     *
     * @var int $retries number of retries to do, default 3
     * @access public
     */
    public $retries = 3;

    /**
     * $connInfo
     * Connection info and credentials
     *
     * @var array $connInfo
     * @see IacMysqli::connectionInfo_set IacMysqli::connectionInfo_set
     * @access public
     */
    public $connInfo=array();

    /**
     * $connected_to
     * Last connected to, connection info and credentials.
     *
     * @var array $connected_to
     * @see IacMysqli::real_connect IacMysqli::real_connect
     * @access public
     */
    public $connected_to = array();

    /**
     * $retry_usleep
     * milli-seconds to wait before re-issuing statements or trying to reconnect to db
     *
     * @var int $retry_sleep number milli-seconds to wait before re-issuing statements
     * @access public
     */
    public $retry_usleep = 100;

    /**
     * $preparedLast
     * Last prepared statement's sql command
     *
     * @var string $preparedLast last prepared sql command
     *
     * @see IacMysqli::prepare() IacMysqli::prepare()
     * @access public
     */
    public $preparedLast='';


    /**
     * EXEC_ARRAY_INDEX
     * Return recordset as zero based numerical array of arrays (either asociative by field name or numerically)
     *
     * @const int EXEC_ARRAY_INDEX return resultset as numeric array
     * @see IacMysqli::selectArrayIndex() IacMysqli::selectArrayIndex()
     * @access protected
    */
    protected $EXEC_ARRAY_INDEX = 1;

    /**
     * EXEC_SINGLE_READ
     * Return first column of first row, from recordset
     *
     * @const int EXEC_SINGLE_READ return first column of first row
     * @see IacMysqli::single_read() IacMysqli::single_read()
     * @access protected
    */
    protected $EXEC_SINGLE_READ = 2;

    /**
     * EXEC_SIGLETON
     * Return an array from the first row only, (either asociative by field name or numerically)
     *
     * @const int EXEC_SIGLETON return resultset first row as array
     * @see IacMysqli::singleton() IacMysqli::singleton()
     * @access protected
    */
    protected $EXEC_SINGLETON = 3;

    /**
     * EXEC_SIGLETON_FULL
     * Return an asociative array from the first row only, if not found return an associative array with all the fields set to ''
     *
     * @const int EXEC_SIGLETON_FULL return resultset first row as array, if not found array with fields
     * @see IacMysqli::singleton_full() IacMysqli::singleton_full()
     * @access protected
    */
    protected $EXEC_SINGLETON_FULL = 4;


    /**
     * EXEC_VECTOR
     * Return a zero based numerical array with the first column of each row.
     *
     * @const int EXEC_VECTOR return resultset's first column as numeric array
     * @see IacMysqli::selectVector() IacMysqli::selectVector()
     * @access protected
    */
    protected $EXEC_VECTOR = 5;

    /**
     * EXEC_KEY_VALUE
     * Return an associative array: array( col1=>col2, col1=>col2, ... ).
     *
     * @const int EXEC_KEY_VALUE return resultset's second column as associative array indexed by first column
     * @see IacMysqli::selectKeyValue() IacMysqli::selectKeyValue()
     * @access protected
    */
    protected $EXEC_KEY_VALUE = 6;

    /**
     * EXEC_ARRAY_KEY
     *
     * @const int EXEC_ARRAY_KEY return resultset's as associative array indexed by column $key
     * @see IacMysqli::select_key_array() IacMysqli::select_key_array()
     * @access protected
    */
    protected $EXEC_ARRAY_KEY = 7;

    /**
     * EXEC_QUERY
     * Return an associative array indexed by the specified field.
     *
     * @const int EXEC_QUERY execute a DML query
     * @see IacMysqli::query() IacMysqli::query()
     * @access protected
    */
    protected $EXEC_QUERY = 8;

    protected $EXEC_RESULT_SET = 9;

    /**
     * $trace
     * Array with al sql statements issued.
     *
     * @var array $trace log of sql commands issued
     * @see IacMysqli::trace_get() IacMysqli::trace_get()
     * @see IacMysqli::log_trace() IacMysqli::log_trace()
     * @access protected
     */
    protected $trace = array();

    /**
     * $metaData
     * Stores field_info of last select statement.
     *
     * @var array $metaData field info of last select query
     * @see IacMysqli::metaData_get() IacMysqli::metaData_get()
     * @access protected
     */
    protected $metaData = array();

    /**
     * $errorLog
     * Stores sql error messages.
     *
     * @var array $errorLog log of sql errors
     * @see IacMysqli::log_sql_error() IacMysqli::log_sql_error()
     * @see IacMysqli::errorLog_get() IacMysqli::errorLog_get()
     *
     * @access protected
     */
    protected $errorLog = array();

    /**
     * $retryOnErrors
     * Mysqli errors on which to retry the transaction or query if not inside a transaction.
     *
     * @var array $retryOnErrors  mysqli errors on wichi to retry if not inside transaction
     * * Error: 1015 SQLSTATE: HY000 (ER_CANT_LOCK) Message: Can't lock file (errno: %d)
     * * Error?: 1027 SQLSTATE: HY000 (ER_FILE_USED) Message: '%s' is locked against change
     * * Error: 1689 SQLSTATE: HY000 (ER_LOCK_ABORTED) Message: Wait on a lock was aborted due to a pending exclusive lock
     * * Error: 1205 SQLSTATE: HY000 (ER_LOCK_WAIT_TIMEOUT) Message: Lock wait timeout exceeded; try restarting transaction
     * * Error: 1206 SQLSTATE: HY000 (ER_LOCK_TABLE_FULL) Message: The total number of locks exceeds the lock table size
     * * Error: 1213 SQLSTATE: 40001 (ER_LOCK_DEADLOCK) Message: Deadlock found when trying to get lock; try restarting transaction
     * * Error: 1622 SQLSTATE: HY000 (ER_WARN_ENGINE_TRANSACTION_ROLLBACK) Message: Storage engine %s does not support rollback for this statement.
     *      Transaction rolled back and must be restarted
     * * Error: 1614 SQLSTATE: XA102 (ER_XA_RBDEADLOCK) Message: XA_RBDEADLOCK: Transaction branch was rolled back: deadlock was detected
     * * Error: 2006 (CR_SERVER_GONE_ERROR) Message: MySQL server has gone away
     * * Error: 2013 (CR_SERVER_LOST) Message: Lost connection to MySQL server during query
     *
     * @see IacMysqli::runSql() IacMysqli::runSql()
     * @access protected
     */
    protected $retryOnErrors = array(1015=>1, 1689=>1, 1205=>1, 1206=>1, 1213=>1, 1622=>1, 1614=>1, 2006=>1, 2013=>1 );
    
    /**
     * $reconnectOnErrors
     * Mysqli errors on which to try to reconnect.
     *
     * @var array $reconnectOnErrors mysql errorcodes on which to reconnect
     *
     * Error: 2006 (CR_SERVER_GONE_ERROR) Message: MySQL server has gone away
     * Error: 2013 (CR_SERVER_LOST) Message: Lost connection to MySQL server during query
     * Error: 2024 (CR_PROBE_SLAVE_CONNECT) Message: Error connecting to slave:
     * Error: 2025 (CR_PROBE_MASTER_CONNECT) Message: Error connecting to master:
     * Error: 2026 (CR_SSL_CONNECTION_ERROR) Message: SSL connection error: %s
     * --
     * Error: 2020 (CR_NET_PACKET_TOO_LARGE) Message: Got packet bigger than 'max_allowed_packet' bytes
     *  On queries larger than max_ a lost connection error may be returned
     *
     * @access protected
     */
     protected $reconnectOnErrors = array(2006=>1, 2013=>1);

    /**
     * $begins
     * Counts the number of begins pending commit or rollback.
     *
     * @var int $begins number of unclosed begins
     *
     * @access protected
     */
    protected $begins = 0;

    protected $IacSqlInfo;

//////////////////////////////////////////////////
// CONSTRUCTOR & DESTRUCTOR
/////////////////////////////////////////////////
    /**
     * iacMysqli::__construct()
     * constructor.
     *
     * @param string $host host to connect, defaults to php.ini setting
     * @param string $username login, to connect, defaults to php.ini setting
     * @param string $passwd password, to connect, defaults to php.ini setting
     * @param string $dbname databasename
     * @param string $port to connect, defaults to php.ini setting
     * @param string $socket socket to use for connection, defaults to php.ini setting
     * @return void
     * @access public
     */
    public function __construct($host=null, $username=null, $passwd=null, $dbname='', $port=null, $socket=null) {
        $this->connectionInfo_set($host, $username, $passwd, $dbname, $port, $socket);
        $this->throwSqlException_set(false);
    }

    /**
     * IacMysqli::__destruct()
     * Resests mysqli_report_mode to MYSQLI_REPORT_OFF
     *
     * @return void
     */
    #[Pure] public function __destruct() {
        /** @noinspection PhpObjectFieldsAreOnlyWrittenInspection */
        $driver = new mysqli_driver();
        $driver->report_mode = MYSQLI_REPORT_OFF;
    }


//////////////////////////////////////////////////
// CONNECTION CREDENTIALS
/////////////////////////////////////////////////

    /**
     * IacMysqli::connectionInfo_set()
     * Adds connection information & credentials for database connection to the connection array.
     *
     * @param string $host host to connect, defaults to php.ini setting
     * @param string $username login, to connect, defaults to php.ini setting
     * @param string $passwd password, to connect, defaults to php.ini setting
     * @param string $dbname databasename
     * @param string $port to connect, defaults to php.ini setting
     * @param string $socket socket to use for connection, defaults to php.ini setting
     * @return void
     *
     * @access public
     */
    public function connectionInfo_set($host=null, $username=null, $passwd=null, $dbname='', $port=null, $socket=null) {
        $this->connInfo[] =
                array(
                    'host' => $host === null ? ini_get("mysqli.default_host") : $host,
                    'username' => $username === null ? ini_get("mysqli.default_user") : $username,
                    'passwd' => $passwd === null ? ini_get("mysqli.default_pw") : $passwd,
                    'dbname' => $dbname,
                    'port' => $port === null ? ini_get("mysqli.default_port") : $port,
                    'socket' => $socket === null ? ini_get("mysqli.default_socket") : $socket
                );
    }

    /**
     * IacMysqli::connect_toString()
     * A description of the connection.
     *
     * @param array $con connection info
     * @return string text display for connection info
     * @access public
     */
    public function connect_toString($con=null) {
        if($con === null)
            $con = $this->connected_to;
        if(empty($con))
            return "Not connected";
        return "$con[host]:$con[port]/$con[socket] $con[dbname] as $con[username]";
    }

//////////////////////////////////////////////////
// CONNECT, RE-CONNECT & CLOSE
/////////////////////////////////////////////////

    /**
     * iacMysqli::connect()
     * connect or reconnect to the db.
     *
     * @return bool true connected, false conection failed
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     */
    public function connect() {
        //$this->log_trace("-- mysqli_init()");
        //$this->mysqli = mysqli_init();
        $this->mysqli = new mysqli();
        // @codeCoverageIgnoreStart
        if(!$this->mysqli) {
            $this->log_sql_error('-- new mysqli() Error', 0, false);
            if($this->throwSqlException_get()) {
                throw new IacSqlException("new mysqli() error",0,null,"-- error initializing mysqli");
            }
            return false;
        }
        // @codeCoverageIgnoreEnd

        try {
            foreach($this->mysqli_options as $option) {
                //$this->log_trace("mysqli->option($option[0], $option[1])");
                if(!$this->mysqli->options($option[0], $option[1]) ) {
                    $this->log_sql_error("-- setting option: $option[0] = $option[1]", 0, false);
                    if($this->throwSqlException_get()) {
                        throw new IacSqlException("-- setting option: $option[0] = $option[1]",0,null,"-- setting option");
                    }
                    return false;
                }
            }
        } catch(mysqli_sql_exception  $mysqliException) {
            $this->log_sql_error("-- setting option: $option[0] = $option[1]", 0, false);
            throw new IacSqlException($mysqliException,0,null,"-- setting option");
        }

        return $this->connect_real('Connect to DB');
        //DUDA: set quality of service, ie consistency. http://php.net/manual/en/function.mysqlnd-ms-set-qos.php
    }

    /**
     * IacMysqli::connect_real()
     *
     * @param string $msg
     * @return bool
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access protected
     */
    protected function connect_real($msg) {
        foreach($this->connInfo as $con) {
            $retries = 0;
            do {
                //$this->log_trace("$msg to ".$this->connect_toString($con), $retries);
                try {
                    if(@$this->mysqli->real_connect($con['host'], $con['username'], $con['passwd'], $con['dbname'], $con['port'], $con['socket'])) {
                        $this->connected_to = $con;
                        $this->mysqli->set_charset($this->charset);
                        $this->runSql("SET NAMES '$this->charset' COLLATE '$this->coalition'", $this->EXEC_QUERY);
                        return true;
                    }
                } catch(Exception  $mysqliException) {
                   // Do Nothing $throwMysqliException = new IacSqlException($mysqliException,0,null,"-- $msg");
                }
                $this->log_sql_error("-- ".mysqli_connect_errno().': '.mysqli_connect_error()." $msg to ".$this->connect_toString($con), $retries, false);
                usleep($this->retry_usleep); // wait to reissue
            } while($retries++ < $this->retries);
        }

        $this->connected_to = array();
        //if($this->throwSqlException_get()) {
            throw new IacSqlException($this->mysqli->error, $this->mysqli->errno,null,"-- $msg");
        //}
        //return false;
    }

    /**
     * IacMysqli::reconnect()
     *
     * @return bool true on success, false on failure
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     */
    public function reconnect() {
        // check if reconnection needed, ie max_allowed_packet could cause errno 2006
        try {
            if($this->mysqli->query('DO 1'))//if($this->mysqli->ping())
                return true;
        } catch(mysqli_sql_exception  $mysqliException) {
            // Do Nothing
        }
        return $this->connect_real('Reconnect');
    }

    /**
     * iacMysqli::close()
     * close db connection, silently.
     *
     * @return bool true on success, false on failure
     * @access public
     */
    public function close() {
        if(get_class($this->mysqli) === 'mysqli' && method_exists($this->mysqli, 'close')) {
                $this->log_trace("-- close db connection");
            try {
                if($this->mysqli->close()) {
                    $this->connected_to = array();
                    return true;
                }
            }
            catch(Exception  $mysqliException) {
                // Do Nothing error is reported below
            }
        }
        $this->log_sql_error('-- mysqli->close()', 0, false);
        return false;
    }

//////////////////////////////////////////////////
// TRANSACTIONS
/////////////////////////////////////////////////

    /**
     * IacMysqli::begin()
     * start or begin a transaction.
     *
     * @return bool true ok, false error encountered
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     */
    public function begin() {
        $started = $this->autocommit(false, " -- START TRANSACTION");
        if($started)
            $this->begins++;
        return $started;
    }



    /**
     * IacMysqli::commit()
     * commit a transaction.
     *
     * @return bool true ok, false error encountered
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @todo:  MYSQLI_TRANS_COR_*
     */
    public function commit() {
        $this->log_trace("commit");
        try {
            if($this->mysqli->commit()) {
                $this->begins--;
                $this->autocommitToDefault();
                return true;
            }
        } catch(mysqli_sql_exception $mysqliException) {
           // Do Nothing
        }
        $this->begins--;
        $this->log_sql_error('-- commit');
        $errno = $this->mysqli->errno;
        $errmsg = $this->mysqli->error;
        $this->autocommitToDefault();
        if($this->throwSqlException_get()) {
            throw new IacSqlException($errmsg, $errno,null,"commit -- commit()");
        }
        return false;
    }

    /**
     * IacMysqli::rollback()
     * rollback a transaction.
     *
     * @return bool true ok, false error encountered
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     */
    public function rollback() {
        $this->log_trace("rollback");
        try {
            if($this->mysqli->rollback() ) {
                // rollback ok, reset counters and return autocommit
                $this->begins--;
                if($this->begins < 0)
                    $this->begins = 0;
                $this->autocommitToDefault();
                return true;
            }
        } catch(mysqli_sql_exception $mysqliException) {
           // Do nothing $throwMysqliException = new IacSqlException($mysqliException,$this->mysqli->errno,null,"rollback");
        }

        $errno = $this->mysqli->errno;
        $errmsg = $this->mysqli->error;
        $this->log_sql_error('-- rollback');
        $this->autocommitToDefault();
        $this->begins--;
        if($this->begins < 0)
            $this->begins = 0;
        //TODO: que debe ser rollback con error por disconnect?
        //TODO: log: CHECAR ER_WARNING_NOT_COMPLETE_ROLLBACK updating a nontransactional table within a transaction
        if($this->throwSqlException_get()) {
            throw new IacSqlException($errmsg, $errno,null,"rollback -- rollback()");
        }
        return false;
    }

    /**
     * IacMysqli::autocommit()
     * set autocommit.
     *
     * @param bool $mode true set autocommit, false no autocommit
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @return int sucess
     * @access public
     */
    public function autocommit($mode, $comment='') {
        $retries = 0;
        do {
            $this->log_trace("SET autocommit = ".($mode ? "1" : "0" )."$comment");
            try {
                if($this->mysqli->autocommit($mode)) {
                    return true;
                }
            } catch(mysqli_sql_exception $mysqliException) {
                // Do nothing
            }
            $errno = $this->mysqli->errno;
            $errmsg = $this->mysqli->error;
            $this->log_sql_error("SET autocommit = $mode$comment");
            if( $this->begins === 0 && isset($this->reconnectOnErrors[$errno]) ) {
                $this->reconnect();
            } else {
                usleep($this->retry_usleep); // wait to reissue
            }
        } while( $this->begins === 0 &&  $retries++ < $this->retries);
        if($this->throwSqlException_get()) {
            throw new IacSqlException($errmsg, $errno,null,"SET AUTOCOMMIT=1 -- autocommit()");
        }
        return false;
    }


    /**
     * IacMysqli::autocommitToDefault()
     * Sets autocommit to default value.
     *
     * @return bool true on success false on error
     */
    protected function autocommitToDefault() {
        $retries = 0;
        do {
            try {
                if($this->mysqli->autocommit($this->autocommitDefault)) {
                    return true;
                }
            } catch(mysqli_sql_exception  $mysqliException) {}
            $this->reconnect();
        } while($retries++ < $this->retries);
        return false;
    }

    /**
     * IacMysqli::transaction()
     * Runs commands in an array as a transaction running first begin() and after the last command commit or rollback on sql error
     * If servers sends a "rollbacked, resend transaction error" the entire transaction will be resent upto $this->retries times.
     *
     * @param array $sqlArray an array of sql commands
     *        if the index is a string from then on {{key}} will be replaced by that commands lastInsertId
     * @param bool $doSubstitute
     * @param string $replacePrefixQuoted default {{ Will quote and then replace, ie name={{newName}}
     * @param string $replaceSuffixQuoted default }}
     * @param string $replacePrefixUnQuoted default [[ Will replace without quoting ie name='[[name]], Jr'
     * @param string $replaceSuffixUnQuoted default ]]
     * @return bool true transaction succeeded, false failed
     *
     *
     *
     * @example
     * <code>
     * $sql->transaction( array(
     *          "UPDATE person SET title = 'Mr' WHERE id = 2",
     *          'SPOUSE' => "INSERT INTO person(name, relationship) VALUES('Susan', 'My wife')"
     *          "UPDATE person SET spouse_id = '{{SPOUSE}}' WHERE id=1",
     *           ));
     *          Will run within a transaction issuing a commit at the end or rollback on sql error
     *          If servers sends a "rollbacked, resend transaction error" the entire transaction will be resent upto $this->retries times
     * </code>
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     *
     *
     * @see IacMysqli::queryArray() IacMysqli::queryArray()
     * @see IacMysqli::begin() IacMysqli::begin()
     * @see IacMysqli::commit() IacMysqli::commit()
     * @see IacMysqli::rollback() IacMysqli::rollback()
     * @see IacMysqli::$retries IacMysqli::$retries
     *
     * @access public
     */
    public function transaction($sqlArray, $doSubstitute = true, $replacePrefixQuoted='{{', $replaceSuffixQuoted='}}', $replacePrefixUnQuoted='[[', $replaceSuffixUnQuoted=']]',$forceKey=null) {
        $retries = 0;
        do {
            if(!$this->begin())
                return false;
            try {
                if($this->queryArray($sqlArray, $doSubstitute, $replacePrefixQuoted, $replaceSuffixQuoted, $replacePrefixUnQuoted, $replaceSuffixUnQuoted,$forceKey)) {
                    return $this->commit();
                }
            } catch(mysqli_sql_exception  $mysqliException) {
               // Do Nothing $throwMysqliException = new IacSqlException($mysqliException,0,null,'-- transaction');
            }
            $errno = $this->mysqli->errno;
            $errmsg = $this->mysqli->error;
            if(!$this->rollback())
                return false;
        } while($retries++ < $this->retries && isset($this->reconnectOnErrors[$errno]) );
        if($this->throwSqlException_get()) {
            throw new IacSqlException($errmsg, $errno,null,'-- transaction');
        }
        return false;
    }

//////////////////////////////////////////////////
// Data modification & DDL Queries
/////////////////////////////////////////////////

    /**
     * IacMysqli::queryArray()
     * Execues all queries in $sqlArray, substituing string keys for result if $doSubstitute=true.
     *
     * @param array $sqlArray an array of sql commands
     *        if the index is a string from then on {{key}} will be replaced by that commands lastInsertId, when $doSubstitute=true
     *        for begin/commit use instead transaction()
     * @param bool $doSubstitute
     * @param string $replacePrefixQuoted default {{ Will quote and then replace, ie name={{newName}}
     * @param string $replaceSuffixQuoted default }}
     * @param string $replacePrefixUnQuoted default [[ Will replace without quoting ie name='[[name]], Jr'
     * @param string $replaceSuffixUnQuoted default ]]
     * @return bool true success, false failed
     *
     * @example $sql->queryArray( array(
     *            "UPDATE person SET title = 'Mr' WHERE id = 2",
     *            'SPOUSE' => "INSERT INTO person(name, relationship) VALUES('Susan', 'My wife')"
     *            "UPDATE person SET spouse_id = '{{SPOUSE}}' WHERE id=1 -- {{SPOUSE}} gets replaced with last inserted id ",
     *           ));
     *           Will run all commands, stopping if an sql error is encountered
     *
     * @see IacMysqli::transaction() IacMysqli::transaction()
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     */
    public function queryArray($sqlArray, $doSubstitute = true, $replacePrefixQuoted='{{', $replaceSuffixQuoted='}}', $replacePrefixUnQuoted='[[', $replaceSuffixUnQuoted=']]',$forceKey=null) {
        global $gIAsql; //VCA
        $haveLastId = false;
        $search = array();
        $replace = array();
        // var_dump($sqlArray);
        if(!is_array($sqlArray))
            return false;
        foreach($sqlArray as $key=>$sql) if(!empty($sql)) {
            if($haveLastId)
                $sql = str_ireplace($search, $replace, $sql);
            if(!($ret = $this->runSql($sql, $this->EXEC_QUERY)) ) {
                return false;
            }
            elseif($doSubstitute && !is_numeric($key)) {
                $tmpKey = empty($forceKey) ? $key : $forceKey;
                if($ret === true && $this->mysqli->insert_id > 0 && stripos($sql, "insert")!==false) {
                    // assume substituion is last inserted id
                    $search[] = $replacePrefixQuoted.$tmpKey.$replaceSuffixQuoted;
                    $replace[] =  $this->strit($this->mysqli->insert_id);
                    $search[] = $replacePrefixUnQuoted.$tmpKey.$replaceSuffixUnQuoted;
                    $replace[] =  $this->mysqli->insert_id;
                    $this->log_trace("-- Will substitue ".$replacePrefixQuoted.$tmpKey.$replaceSuffixQuoted
                        ." with !insert_id=".$this->strit($this->mysqli->insert_id)
                        .", and $replacePrefixUnQuoted.$tmpKey.$replaceSuffixUnQuoted with ".$this->mysqli->insert_id);
                    $haveLastId = true;
                    $gIAsql['last_id']=$this->mysqli->insert_id; //VCA
                } elseif(is_array($ret)) {
                    $current = reset($ret);
                    if(is_array($current))
                        $current = reset($current);
                    $search[] = $replacePrefixQuoted.$tmpKey.$replaceSuffixQuoted;
                    $replace[] = $this->strit($current);
                    $search[] = $replacePrefixUnQuoted.$tmpKey.$replaceSuffixUnQuoted;
                    $replace[] = $current;
                    $this->log_trace("-- Will substitue ".$replacePrefixQuoted.$tmpKey.$replaceSuffixQuoted
                        ." with selected value=".$this->strit($current)
                        .", and $replacePrefixUnQuoted.$tmpKey.$replaceSuffixUnQuoted with ".$current);
                    $haveLastId = true;
                    $gIAsql['last_id']=$current; //VCA
                }
            }
        }
        return true;
    }

    /**
     * iacMysqli::query()
     * Executes a query like update/insert/delete.
     *
     * string|mysqli_stmt $sql string: sql command, mysqli_stmt binded statement
     * @param int $resultmode MYSQLI_STORE_RESULT [default], MYSQLI_USE_RESULT may use less memory
     * @return mixed bool|array, true success, false on error, associative array if sql returns result object
     *
     * @example query("UPDATE table SET col1=1 WHERE table_id=1");
     *          returns true on success, false on sql error
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     */
    public function query($sql, $resultmode = MYSQLI_STORE_RESULT) {
        return $this->runSql($sql, $this->EXEC_QUERY, MYSQLI_ASSOC, array(), null, $resultmode);
    }

    /**
     * IacMysqli::insertAndGetId()
     * Executes a query (normally insert) if succesfull returns last inserted id, false on error.
     *
     * string|mysqli_stmt $sql
     * @return int|string|false last inserted id or false on error
     */
    public function insertAndGetId($sql) {
        if($this->runSql($sql, $this->EXEC_QUERY) )
            return $this->mysqli->insert_id;
        return false;
    }


//////////////////////////////////////////////////
// Retreive info, selects
/////////////////////////////////////////////////

    /**
     * iacMysqli::single_read()
     * read first column of first returned row.
     *
     * string|mysqli_stmt $sql sql command
     * @param string $onNotFound on no rows found return $onNotFound defaults to ''
     * @return string first column of first row in select written in $sql, on empty returns $onNotFound
     *
     * @exmpale single_read('SELECT name, last_name from person ORDER BY id')
     *          returns 'Dana'
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     *
     * @see iacMysqli::singleton() iacMysqli::singleton()
     * @see iacMysqli::singleton_full() iacMysqli::singleton_full()
     * @see iacMysqli::selectVector() iacMysqli::selectVector()
     * @see iacMysqli::selectKeyValue() iacMysqli::selectKeyValue()
     * @see iacMysqli::selectArrayKey() iacMysqli::selectArrayKey()
     * @see iacMysqli::selectArrayIndex() iacMysqli::selectArrayIndex()
     */
    public function single_read($sql, $onNotFound='') {
        return $this->runSql($sql, $this->EXEC_SINGLE_READ, MYSQLI_NUM, $onNotFound);
    }

    /**
     * iacMysqli::singleton()
     * reads first row.
     *
     * string|mysqli_stmt $sql sql command
     * @param array $onNotFound  on no rows found return $onNotFound defaults to array()
     * @param int $resultType MYSQLI_ASSOC [default], MYSQLI_NUM, or MYSQLI_BOTH
     * @return mixed first row of select command in $sql, on empty returns $onNotFound, false on error
     *
     * @exmpale singleton('SELECT id, name, last_name from person ORDER BY id')
     *          returns array('id'=>1, 'name'=>'Dana', 'last_name'=>'Smith')
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     *
     * @see iacMysqli::single_read() iacMysqli::single_read()
     * @see iacMysqli::singleton_full() iacMysqli::singleton_full()
     * @see iacMysqli::selectVector() iacMysqli::selectVector()
     * @see iacMysqli::selectKeyValue() iacMysqli::selectKeyValue()
     * @see iacMysqli::selectArrayKey() iacMysqli::selectArrayKey()
     * @see iacMysqli::selectArrayIndex() iacMysqli::selectArrayIndex()
     */
    public function singleton($sql, $onNotFound = array(), $resultType = MYSQLI_ASSOC) {
        return $this->runSql($sql, $this->EXEC_SINGLETON, $resultType, $onNotFound);
    }

    /**
     * iacMysqli::singleton_full()
     * reads first row, on not found returns fields with '' as value.
     *
     * string|mysqli_stmt $sql sql command
     * @param mixed $onNull on Null fill value with
     * @return mixed first row of select command in $sql, on empty returns $onNotFound, false on error
     *
     * @exmpale singleton_full('SELECT id, name, last_name from person ORDER BY id')
     *          returns array('id'=>1, 'name'=>'Dana', 'last_name'=>'Smith')
     *
     *
     * @exmpale singleton_full('SELECT id, name, last_name from person WHERE id = -99.9 ORDER BY id -- non existant id')
     *          returns array('id'=>'', 'name'=>'', 'last_name'=>'')
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     *
     * @see iacMysqli::single_read() iacMysqli::single_read()
     * @see iacMysqli::singleton() iacMysqli::singleton()
     * @see iacMysqli::selectVector() iacMysqli::selectVector()
     * @see iacMysqli::selectKeyValue() iacMysqli::selectKeyValue()
     * @see iacMysqli::selectArrayKey() iacMysqli::selectArrayKey()
     * @see iacMysqli::selectArrayIndex() iacMysqli::selectArrayIndex()
     */
    public function singleton_full($sql, $onNull='') {
        return $this->runSql($sql, $this->EXEC_SINGLETON_FULL, MYSQLI_ASSOC,array(),$onNull);
    }

    /**
     * iacMysqli::selectVector()
     * returns select as a vector.
     *
     * string|mysqli_stmt $sql sql command
     * @param mixed $onNotFound on no rows returned return $onNotFound
     * @return mixed array(col_1_row_1, col_1_row_2, ...) or false on error
     *
     * @exmpale selectVector('SELECT id, name, last_name from person ORDER BY id')
     *          returns array( 1, 3 )
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     *
     * @see iacMysqli::single_read() iacMysqli::single_read()
     * @see iacMysqli::singleton() iacMysqli::singleton()
     * @see iacMysqli::singleton_full() iacMysqli::singleton_full()
     * @see iacMysqli::selectKeyValue() iacMysqli::selectKeyValue()
     * @see iacMysqli::selectArrayKey() iacMysqli::selectArrayKey()
     * @see iacMysqli::selectArrayIndex() iacMysqli::selectArrayIndex()
     */
    public function selectVector($sql, $onNotFound=array()) {
        return $this->runSql($sql, $this->EXEC_VECTOR, MYSQLI_NUM, $onNotFound);
    }

    public function selectResultSet($sql) {
        return $this->runSql($sql, $this->EXEC_RESULT_SET, MYSQLI_NUM, null);
    }
    /**
     * iacMysqli::selectKeyValue()
     * returns select as key value array.
     *
     * string|mysqli_stmt $sql sql command
     * @param mixed $onNotFound on no rows returned return $onNotFound
     * @return mixed array or false on error
     *
     * @exmpale selectArrayIndex('SELECT id, name, last_name from person ORDER BY id')
     *          returns array( 1=>'Dana', 3=>'Paul' )
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     *
     * @see iacMysqli::single_read() iacMysqli::single_read()
     * @see iacMysqli::singleton() iacMysqli::singleton()
     * @see iacMysqli::singleton_full() iacMysqli::singleton_full()
     * @see iacMysqli::selectVector() iacMysqli::selectVector()
     * @see iacMysqli::selectArrayKey() iacMysqli::selectArrayKey()
     * @see iacMysqli::selectArrayIndex() iacMysqli::selectArrayIndex()
     */
    public function selectKeyValue($sql, $onNotFound=array()) {
        return $this->runSql($sql, $this->EXEC_KEY_VALUE, MYSQLI_NUM, $onNotFound);
    }

    /**
     * iacMysqli::selectArrayKey()
     * returns select indexed by key.
     *
     * string|mysqli_stmt $sql sql command
     * @param mixed $onNotFound on no rows returned return $onNotFound
     * @param string $key for associate array
     * @return array associative array indexed by $key, each entry contains a selected row
     *
     * @exmpale selectArrayKey('SELECT id, name, last_name from person ORDER BY id', 'id')
     *          returns array(1=>array('id'=>1, 'name'=>'Dana', 'last_name'=>'Smith'),
     *                        3=>array('id'=>3, 'name'=>'Paul', 'last_name'=>'Jones')
     *                  )
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     *
     * @see iacMysqli::single_read() iacMysqli::single_read()
     * @see iacMysqli::singleton() iacMysqli::singleton()
     * @see iacMysqli::singleton_full() iacMysqli::singleton_full()
     * @see iacMysqli::selectVector() iacMysqli::selectVector()
     * @see iacMysqli::selectKeyValue() iacMysqli::selectKeyValue()
     * @see iacMysqli::selectArrayIndex() iacMysqli::selectArrayIndex()
     */
    public function selectArrayKey($sql, $key = 'id', $onNotFound=array(), $resultType = MYSQLI_ASSOC) {
        return $this->runSql($sql, $this->EXEC_ARRAY_KEY, $resultType, $onNotFound, $key);
    }

    /**
     * iacMysqli::selectArrayIndex()
     * returns selecet as a numeric array.
     *
     * string|mysqli_stmt $sql sql command
     * @param mixed $onNotFound on no rows returned return $onNotFound
     * @param int $resultType MYSQLI_ASSOC [default], MYSQLI_NUM, or MYSQLI_BOTH
     * @return mixed false on error
     *
     * @exmpale selectArrayIndex('SELECT id, name, last_name from person ORDER BY id')
     *          returns array(0=>array('id'=>1, 'name'=>'Dana', 'last_name'=>'Smith'),
     *                        1=>array('id'=>3, 'name'=>'Paul'', 'last_name'=>'Jones')
     *                  )
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     *
     * @see iacMysqli::single_read() iacMysqli::single_read()
     * @see iacMysqli::singleton() iacMysqli::singleton()
     * @see iacMysqli::singleton_full() iacMysqli::singleton_full()
     * @see iacMysqli::selectVector() iacMysqli::selectVector()
     * @see iacMysqli::selectKeyValue() iacMysqli::selectKeyValue()
     * @see iacMysqli::selectArrayKey() iacMysqli::selectArrayKey()
     */
    public function selectArrayIndex($sql, $onNotFound=array(), $resultType = MYSQLI_ASSOC) {
        return $this->runSql($sql, $this->EXEC_ARRAY_INDEX, $resultType, $onNotFound);
    }

    //////////////////////////////////////////////////
// Return multikeyed array
/////////////////////////////////////////////////
    /**
     * $arr[key_1][...]['key_'.$numKeys]=$col[$numKeys+1] or =[colName=>v,...] if total columns > $numKeys+1
     *
     * @param string|array $sql
     * @param int $numKeys
     * @param int $resultType MYSQLI_ASSOC || MYSQLI_NUM
     * @return array |bool $arr[key_1][...]['key_'.$numKeys]=$col[$numKeys+1] or =[colName=>v,...] if total columns > $numKeys+1
     * @throws IacSqlException
     */
    public function selectArrayMultiKey($sql, $numKeys, $resultType = MYSQLI_ASSOC, $autoPrefix = false) {
        $ret = array();
        if(!is_array($sql)) {
            $this->selectArrayMultiKeyDo($sql, $ret, $numKeys, $resultType, $autoPrefix);
            return $ret;
        }

        $ok = true;
        foreach($sql as $s) // on ok false it is an error!
            try {
                $ok &= $this->selectArrayMultiKeyDo($s, $ret, $numKeys, $resultType, $autoPrefix);
            } catch(Exception $exception) {
                $ok = false;
                if($this->throwSqlException_get()) {
                    throw $exception; // new \Exception($exception->getMessage() );
                }

            }
        return $ok ? $ret : false;
    }

    /**
     * Execute each query and merge or set multi key array $ret
     *
     * @param string|mysqli_stmt $sql
     * @param array $ret
     * @param int $numKeys
     * @param int $resultType MYSQLI_ASSOC || MYSQLI_NUM
     * @return boolean true Ok, false Error
     * @throws IacSqlException
     */
    protected function selectArrayMultiKeyDoOld($sql, &$ret, $numKeys, $resultType) {
        // store statement for reporting it
        if(is_string($sql)) {
            $sqlStatement = $sql;
        } else {
            $sqlStatement = $this->preparedLast;
        }

        $retries = 0;
        do {
            try {
                $this->log_trace($sqlStatement, $retries);
                if(is_string($sql)) {
                    $result = $this->mysqli->query($sql, MYSQLI_STORE_RESULT);
                } else {
                    if( ($result = $sql->execute() )) {
                        $result = $sql->get_result();
                        $sql->store_result();
                    }
                }

                if($result !== false ) { // Data read successfull,
                    for(; $tmp = $result->fetch_array($resultType);) {
                        if(!isset($countFields)) {
                            $countFields = count($tmp);
                        }
                        $arrRef = &$ret;
                        $theKey = reset($tmp);
                        for($k=0; $k<$numKeys; $k++) {
                            if(!isset($arrRef[$theKey])) {
                                $arrRef[$theKey] = array();
                            }
                            $arrRef = &$arrRef[$theKey];
                            $theKey = next($tmp);
                        }

                        $arrRef = array_merge($arrRef, array_slice($tmp,$numKeys));
                    }

                    $this->num_rows = $result->num_rows;
                    if(is_string($sql)) {
                        $result->free();
                    } else {
                        $sql->free_result();
                    }
                    return  true; // returns the result in the requested format
                }
            }
            catch(mysqli_sql_exception  $mysqliException) {
                // Do Nothing  $throwMysqliException = new IacSqlException($mysqliException,0,null,$sqlStatement);
            }

            // log error and save error number

            $errno = $this->mysqli->errno;
            $errorMessage = $this->mysqli->error;
            // try to reconnect on connection lost and we are not inside a transaction
            if($this->begins === 0 && isset($this->reconnectOnErrors[$errno]) ) {
                $this->log_sql_error($sqlStatement." -- reconnect for retry", $retries);
                $this->reconnect();
            } else {
                $this->log_sql_error($sqlStatement, $retries);
                usleep($this->retry_usleep); // wait to reissue
            }
        } while($this->begins === 0 && $retries++ < $this->retries && isset($this->retryOnErrors[$errno]) );
        $this->num_rows = 0;
        if(!$this->throwSqlException_get())
            return false;
        throw new mysqli_sql_exception($errorMessage, $errno,null);
    }

    /**
     * Execute each query and merge or set multi key array $ret
     *
     * @param string|mysqli_stmt $sql
     * @param array $ret
     * @param int $numKeys
     * @param int $resultType MYSQLI_ASSOC || MYSQLI_NUM
     * @return boolean true Ok, false Error
     * @throws IacSqlException
     */
    protected function selectArrayMultiKeyDo($sql, &$ret, $numKeys, $resultType, $autoPrefix = false) {
        // store statement for reporting it
        if(is_string($sql)) {
            $sqlStatement = $sql;
        } else {
            $sqlStatement = $this->preparedLast;
        }

        $retries = 0;
        do {
            try {
                $this->log_trace($sqlStatement, $retries);
                if(is_string($sql)) {
                    $result = $this->mysqli->query($sql, MYSQLI_STORE_RESULT);
                } else {
                    if( ($result = $sql->execute() )) {
                        $result = $sql->get_result();
                        $sql->store_result();
                    }
                }

                if($result !== false ) { // Data read successfull,
                    $fields = $result->fetch_fields();
                    for(; $row = $result->fetch_array(MYSQLI_NUM);) {
                        if(!isset($countFields)) {
                            $countFields = count($row);
                        }

                        $tmp = [];
                        $i=0;
                        foreach($fields as $f) {
                            if(!$autoPrefix || empty($f->table))
                                $tmp[$f->name] = $row[$i++];
                            else
                                $tmp["$f->table.$f->name"] = $row[$i++];
                        }

                        $arrRef = &$ret;
                        $theKey = reset($tmp);
                        for($k=0; $k<$numKeys; $k++) {
                                if(!isset($arrRef[$theKey])) {
                                    $arrRef[$theKey] = array();
                                }
                                $arrRef = &$arrRef[$theKey];
                                $theKey = next($tmp);
                        }

                        $arrRef = array_merge($arrRef, array_slice($tmp,$numKeys));
                    }

                    $this->num_rows = $result->num_rows;
                    if(is_string($sql)) {
                        $result->free();
                    } else {
                        $sql->free_result();
                    }
                    return  true; // returns the result in the requested format
                }
            }
            catch(mysqli_sql_exception  $mysqliException) {
                // Do Nothing  $throwMysqliException = new IacSqlException($mysqliException,0,null,$sqlStatement);
            }

            // log error and save error number

            $errno = $this->mysqli->errno;
            $errorMessage = $this->mysqli->error;
            // try to reconnect on connection lost and we are not inside a transaction
            if($this->begins === 0 && isset($this->reconnectOnErrors[$errno]) ) {
                $this->log_sql_error($sqlStatement." -- reconnect for retry", $retries);
                $this->reconnect();
            } else {
                $this->log_sql_error($sqlStatement, $retries);
                usleep($this->retry_usleep); // wait to reissue
            }
        } while($this->begins === 0 && $retries++ < $this->retries && isset($this->retryOnErrors[$errno]) );
        $this->num_rows = 0;
        if(!$this->throwSqlException_get())
            return false;
        throw new mysqli_sql_exception($errorMessage, $errno,null);
    }

    /**
     * Returns values as keys
     * SELECT col1,col2,col3 = [col1value1=>[col2Value1=>col3Value1,col2Value2=>col3Value2], ].
     *
     * @param mixed $sql
     * @param null|int $numberOfKeys null=last key is scalar, -1 last value is still a key,
     * @return bool|array [col1value1=>[col2Value1=>col3Value1,col2Value2=>col3Value2], ] false on error
     * @throws IacSqlException
     */
    public function selectValuePath($sql, $numberOfKeys = null) {
        $h = [];
        $flat = $this->selectArrayIndex($sql);
        if($flat === false) {
            return false;
        }
        if($numberOfKeys !== null && $numberOfKeys <= 0) {
            $numberOfKeys = count(reset($flat)) + $numberOfKeys;
        }
        foreach($flat as $r) {
            if($numberOfKeys === null) {
                $last = array_pop($r);
            }
            $nKeys = 0;
            $ptr = &$h;
            foreach($r as $fieldName => $v) {
                if($numberOfKeys !== null && ++$nKeys > $numberOfKeys) {
                    $ptr[$fieldName] = $v;
                    continue;
                }
                if(!array_key_exists($v, $ptr)) {
                    $ptr[$v] = [];
                }
                $ptr = &$ptr[$v];
            }
            if($numberOfKeys === null) {
                $ptr = $last;
            }
        }
        return $h;
    }

//////////////////////////////////////////////////
// INFORMATION ON QUERIES
/////////////////////////////////////////////////

    /**
     * iacMysqli::last_insert_id()
     * returns last inserted id (auto increment value).
     *
     * @return int last inserted id
     * @access public
     */
    public function last_insert_id() {
        return $this->mysqli->insert_id;
    }


    /**
     * iacMysqli::found_rows()
     * returns found rows for last select.
     *
     * @return int Last select found rows, if issueed with SQL_CALC_FOUND_ROWS
     *
     * @example
     *   $db->singleton("SELECT SQL_CALC_FOUND_ROWS col1,col2 FROM table WHERE col1=1 LIMIT 1");
     *   $db->found_rows() => returns number of rows matching query, not considering limit clause
     *
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access public
     */
    public function found_rows() {
        return $this->runSql("SELECT FOUND_ROWS()", $this->EXEC_SINGLE_READ,MYSQLI_NUM,null);
    }

    /**
     * IacMysqli::IacSqlInfo_get()
     *
     * @return object \Iac\inc\sql\IacSqlInfo
     *
     * @see \Iac\inc\sql\IacSqlInfo \Iac\inc\sql\IacSqlInfo
     * @access public
     */
    public function IacSqlInfo_get() {
        if(!isset($this->IacSqlInfo))
            $this->IacSqlInfo = new IacSqlInfo();
        return $this->IacSqlInfo;
    }

    /**
     * IacMysqli::metaData_get()
     * Get metadata on last select query
     *
     * @return array field info, metadata, of last select query
     *
     * @example
     *
     * @access public
     */
    public function metaData_get() {
        return $this->metaData;
    }

    /**
     * IacMysqli::metaData_clear()
     * Clears last metadata retreived
     *
     * @return void
     *
     * @access public
     */
    public function metaData_clear() {
        $this->metaData = array();
    }

//////////////////////////////////////////////////
// SET EXCEPTIONS
/////////////////////////////////////////////////

    /**
     * IacMysqli::throwSqlException_set()
     *
     *
     * @param bool|int $mysqli_report_mode true throws excpetions setting: MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT, , functions with errors trhow
     *        false sets MYSQLI_REPORT_OFF, functions with errors return false
     *        int sets $mysqli_report_mode
     *        options: MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT | MYSQLI_REPORT_ALL | MYSQLI_REPORT_OFF
     * @return void
     * @access public
     */
    public function throwSqlException_set($mysqli_report_mode) {
        /** @noinspection PhpObjectFieldsAreOnlyWrittenInspection */
        $driver = new mysqli_driver();
        if($mysqli_report_mode === false)
            $driver->report_mode = MYSQLI_REPORT_OFF;
        elseif($mysqli_report_mode === true)
            $driver->report_mode = MYSQLI_REPORT_STRICT | MYSQLI_REPORT_ERROR;
        else
            $driver->report_mode = $mysqli_report_mode;
    }

    /**
     * IacMysqli::throwSqlException_get()
     * Does mysqli throw exceptions for sql errors
     *
     * @return bool true mysqli_driver sends exceptions for sql errors, false no exceptions thrown
     */
    #[Pure] public function throwSqlException_get() {
        $driver = new mysqli_driver();
        return ($driver->report_mode & MYSQLI_REPORT_STRICT) === MYSQLI_REPORT_STRICT;
    }

//////////////////////////////////////////////////
// LOG TRACE & ERRORS
/////////////////////////////////////////////////

    /**
     * IacMysqli::trace_get()
     * Returns an array with sql commands issued
     *
     * @return array log of sql commands issued
     * @see IacMysqli::$traceOn IacMysqli::$traceOn
     */
    public function trace_get():array {
        if($this->traceOn === false && empty($this->trace))
            return ["sql trace is off, turn it on with \$sql->traceOn=true"];
        return $this->trace;
    }

    /**
     * IacMysqli::log_trace()
     * Stores issued sql commands in
     *
     * @param string $sql command or message to store in trace
     * @param int $retries number of times this command has been re-sent to the server
     * @return void
     * @see IacMysqli::$traceOn IacMysqli::$traceOn
     * @access public
     */
    public function log_trace($sql, $retries=0) {
        if(!$this->traceOn)
            return;
        if($retries === 0)
            $this->trace[] = $sql;
        else
            $this->trace[] = $sql." -- retry $retries";
    }


    /**
     * IacMysqli::errorLog_get()
     * Returns an array with the sql errors
     *
     * @return array Sql errors
     */
    public function errorLog_get() {
        if($this->begins > 0)
            $this->log_sql_error("-- iacMysqli: ".$this->begins." transactions pending commit/rollback", 0, false);
        return $this->errorLog;
    }

    /**
     * IacMysqli::begins_get()
     * Return number of begin transactions waiting a commit or rollback statement
     *
     * @return int number of begin transactions waiting a commit or rollback statement
     */
    public function begins_get() {
        return $this->begins;
    }

    /**
     * IacMysqli::log_sql_error()
     * Stores an error for later processing
     *
     * @param string $sql sql command that caused the error
     * @param int $retries number of times this command has been re-sent to the server
     * @param bool $putMysqliError true include the error from mysqli, false the error is included in $sql
     * @return void
     * @see IacMysqli::$errorLog IacMysqli::$errorLog
     * @access public
     */
    public function log_sql_error($sql, $retries=0, $putMysqliError=true) {
        // if($this->mysqli->errno == "1213" && $retries < 2) return; // ignora deadlock pero no otros errores de sql
        $retryMessage = " -- " . 
            ($retries < $this->retries && !empty($this->retryOnErrors[$this->mysqli->errno]) ? ' es Warning in:' : ' ' )  . 
            " retries: ".$this->retries2string($retries)."/".$this->retries2string($this->retries,true);
        if($putMysqliError === false)
            $this->errorLog[] =  array('errno'=>0, 'error'=>'', 'sql'=>$sql.$retryMessage);
        else
            $this->errorLog[] =  array('errno'=>$this->mysqli->errno, 'error'=>$this->mysqli->error, 'sql'=>$sql.$retryMessage);
    }
    protected function retries2string($num, $desdeUno=false) {
        switch($num) {
            case '0': return 'primera';
            case '1': return 'segunda';
            case '2': return 'tercera';
            case '3': return $desdeUno ? 'tres' : 'ultima';
            default: return $num;
        }
    }
//////////////////////////////////////////////////
// SEND SQL TO SERVER
/////////////////////////////////////////////////

    /**
     * IacMysqli::runSql()
     * Executes an sql statement or binded mysqlistmt
     *
     * @param mixed $sql string or prepared statment
     * @param int $exec  constant $this->EXEC_
     * @param int $resultType defaults to MYSQLI_ASSOC | MYSQLI_NUM
     * @param mixed $onNotFound on no rows returned return $onNotFound
     * @param string $key key to use in asociative arrows
     * @param int $resultmode defaults to MYSQLI_STORE_RESULT
     * @return mixed false on error, true ok no results, array/string database result
     * @throws IacSqlException if mysqli_report(MYSQLI_REPORT_STRICT) was set, else returns false on error
     * @access protected
     */
    protected function runSql($sql, $exec, $resultType = MYSQLI_ASSOC, $onNotFound = array(), $key = null, $resultmode = MYSQLI_STORE_RESULT) {
        // store statement for reporting it

        if(is_string($sql)) {
            if(empty($sql)) {
                return true;
            }
            $sqlStatement = $sql;
        } else {
            $sqlStatement = $this->preparedLast;
        }

        $retries = 0;
        do {
            try {
                $queryStart = microtime(true);
                if(is_string($sql)) {
                    $result = $this->mysqli->query($sql, $resultmode);
                } else {
                    if( ($result = $sql->execute() )) { // execute binded sql statement
                        $result = $sql->get_result();
                        $sql->store_result();
                    }
                }
                $queryEnd = microtime(true);
                $queryTime = $queryEnd - $queryStart;


                if($result === true ) { // data modification successfull
                    $this->affected_rows = $this->mysqli->affected_rows;
                    $this->log_trace($sqlStatement . "/* t=$queryTime */", $retries);
                    return true;

                } elseif($result !== false ) { // data read successfull, return it in the desired format

                    if($this->metaDataOn) { // save fieldinfo on metadata doit
                        if(!isset($this->IacSqlInfo))
                            $this->IacSqlInfo_get();
                        $this->metaData =  $this->IacSqlInfo->result2fields($result);
                    }

                    if($exec === $this->EXEC_RESULT_SET) {
                        $this->log_trace($sqlStatement . "/* t=$queryTime */", $retries);
                        return $result;
                    }
                    if($exec === $this->EXEC_ARRAY_KEY)
                        for($ret = array(); $tmp = $result->fetch_array($resultType);) {
                            if(empty($key)) // $key not defined, use first item in tuple
                                $key = key($tmp);
                            if(!array_key_exists($key, $tmp)) continue; //VCA warning al hacer un vale uso
                            $ret[$tmp[$key]] = $tmp;
                        }

                    elseif($exec === $this->EXEC_SINGLE_READ) {
                        $tmp = $result->fetch_row();
                        // no results return default value
                        if($tmp === null)
                            $ret = $onNotFound;
                        else
                            $ret = $tmp[0];

                    } elseif($exec === $this->EXEC_SINGLETON) {
                        $ret = $result->fetch_array($resultType);

                    } elseif($exec === $this->EXEC_SINGLETON_FULL) {
                        $ret = $result->fetch_assoc();
                        if($ret === null) {
                            $ret = array();
                            // no results return empty asociative array with selected fileds
                            while ($finfo = $result->fetch_field())
                                $ret[ $finfo->name ] = $key;
                        }

                    } elseif($exec === $this->EXEC_VECTOR ) {
                        for($ret = array(); $tmp = $result->fetch_row();)
                            $ret[] = $tmp[0];

                    } elseif($exec === $this->EXEC_KEY_VALUE ) {
                        for($ret = array(); $tmp = $result->fetch_row();)
                            $ret[$tmp[0]] = $tmp[1];

                    } else { // return array with received each row
                        $ret = $result->fetch_all($resultType);
                    }
                    $this->num_rows = $result->num_rows;
                    if(is_string($sql))
                        $result->free();
                    else
                        $sql->free_result();
                    if($exec !== $this->EXEC_SINGLE_READ && empty($ret)) {
                        $ret = $onNotFound;
                    }
                    $query2DataStructure = $queryEnd - microtime(true);
                    $this->log_trace($sqlStatement . "/* qt=$queryTime q2DSt=$query2DataStructure */", $retries);
                    return  $ret; // returns the result in the requested format
                }
            }
            catch(mysqli_sql_exception  $mysqliException) {
              // Do Nothing  $throwMysqliException = new IacSqlException($mysqliException,0,null,$sqlStatement);
            }
            if(!isset($queryTime))
                $queryTime = 'N/A';
            if(!isset($queryEnd))
                $queryEnd = microtime(true);
            $query2DataStructure = $queryEnd - microtime(true);
            $this->log_trace($sqlStatement . "/* qt=$queryTime q2DSt=$query2DataStructure */", $retries);
            // log error and save error number
            $this->log_sql_error($sqlStatement, $retries);
            $errno = $this->mysqli->errno;
            $errmsg = $this->mysqli->error;
            // try to recconect on connection lost and we are not inside a transaction
            if($this->begins === 0 && isset($this->reconnectOnErrors[$errno]) ) {
                $this->reconnect();
            } else {
                usleep($this->retry_usleep); // wait to reissue
            }
        } while($this->begins === 0 && $retries++ < $this->retries && isset($this->retryOnErrors[$errno]) );
        $this->num_rows = 0;
        if(!$this->throwSqlException_get())
            return false;
        throw new IacSqlException($errmsg, $errno,null,$sqlStatement);
    }

    public function selectArrayKeyPropertyValue($sql,&$ret) {
        $ret = [];
        if(!is_array($sql)) {
            return $this->runSqlPropertyValue($sql,$ret,$this->EXEC_ARRAY_KEY);
        }

        $ok = true;
        foreach($sql as $s)
            try {
                $ok &= $this->runSqlPropertyValue($s,$ret,$this->EXEC_ARRAY_KEY);
            } catch(\Iac\inc\sql\IacSqlException  $iacSqlException) {
                $ok = false;
                $errno = $this->mysqli->errno;
                $errmsg = $this->mysqli->error;
                $sqlStatement = $iacSqlException->getSqlStatement();
            }
        if($ok)
            return true;
        if(!$this->throwSqlException_get()) {
            return false;
        }
        throw new IacSqlException($errmsg, $errno,null,$sqlStatement);
    }

    public function singletonPropertyValue($sql,&$ret) {
        $ret = [];
        if(!is_array($sql)) {
            return $this->runSqlPropertyValue($sql,$ret,$this->EXEC_SINGLETON);
        }

        $ok = true;
        foreach($sql as $s)
            try {
                $ok &= $this->runSqlPropertyValue($s,$ret,$this->EXEC_SINGLETON);
            } catch(\Iac\inc\sql\IacSqlException  $iacSqlException) {
                $ok = false;
                $errno = $this->mysqli->errno;
                $errmsg = $this->mysqli->error;
                $sqlStatement = $iacSqlException->getSqlStatement();
            }
        if($ok)
            return true;
        if(!$this->throwSqlException_get())
            return false;
        throw new IacSqlException($errmsg, $errno,null,$sqlStatement);
    }

    protected function runSqlPropertyValue($sql, &$ret, $exec, $key = null) {
        // store statement for reporting it
        if(is_string($sql)) {
            $sqlStatement = $sql;
        } else {
            $sqlStatement = $this->preparedLast;
        }

        $retries = 0;
        do {
            try {
                $this->log_trace($sqlStatement, $retries);
                if(is_string($sql)) {
                    $result = $this->mysqli->query($sql, MYSQLI_STORE_RESULT);
                } else {
                    if( ($result = $sql->execute() )) {
                        $result = $sql->get_result();
                        $sql->store_result();
                    }
                }

                if($result !== false ) { // data read successfull,

                    if($exec === $this->EXEC_ARRAY_KEY) {
                        for(; $tmp = $result->fetch_row();) {
                            $ret[$tmp[0]][$tmp[1]] = [];
                            $len = count($tmp) -1;
                            $i = 2;
                            while($i < $len ) {
                                $ret[$tmp[0]][$tmp[1]][$tmp[$i]] = $tmp[$i + 1];
                                $i += 2;
                            }
                        }

                    } elseif($exec === $this->EXEC_SINGLETON) {
                        for(; $tmp = $result->fetch_row();) {
                            $len = count($tmp) -1;
                            $i = 0;
                            while($i < $len ) {
                                $ret[$tmp[$i]] = $tmp[$i + 1];
                                $i += 2;
                            }
                        }
                    }

                    $this->num_rows = $result->num_rows;
                    if(is_string($sql)) {
                        $result->free();
                    } else {
                        $sql->free_result();
                    }
                    return  true; // returns the result in the requested format
                }
            }
            catch(mysqli_sql_exception  $mysqliException) {
              // Do Nothing  $throwMysqliException = new IacSqlException($mysqliException,0,null,$sqlStatement);
            }

            // log error and save error number
            $this->log_sql_error($sqlStatement, $retries);
            $errno = $this->mysqli->errno;
            $errmsg = $this->mysqli->error;
            // try to recconect on connection lost and we are not inside a transaction
            if($this->begins === 0 && isset($this->reconnectOnErrors[$errno]) ) {
                $this->reconnect();
            } else {
                usleep($this->retry_usleep); // wait to reissue
            }
        } while($this->begins === 0 && $retries++ < $this->retries && isset($this->retryOnErrors[$errno]) );
        $this->num_rows = 0;
        if(!$this->throwSqlException_get())
            return false;
        throw new IacSqlException($errmsg, $errno,null,$sqlStatement);
    }

//////////////////////////////////////////////////
// PREPARED STATEMENTS
/////////////////////////////////////////////////

    /**
     * IacMysqli::prepare()
     *
     * @param string $sql
     * @return mixed false on failure, prepared mysqli_stmt on success
     */
    public function prepare($sql) {
        $this->preparedLast = $sql;
        return $this->mysqli->prepare($sql);
    }

}

/**
 * IacSqlException
 * Exceptions thrown by IacMysqli
 *
 *
 */
//VCA 01-12-2016
//class IacSqlException extends \mysqli_sql_exception {
class IacSqlException extends Exception {
    protected $sqlStatement;

    /**
     * IacSqlException::__construct()
     *
     * @param mixed $message error message or caught exception
     * @param integer $code error number
     * @param object $previous previous exceptin thrown
     * @param string $sqlStatement sql statement that caused the error
     * @return void
     */
    public function __construct($message = null, $code = 0, $previous = null, $sqlStatement='') {
        parent::__construct($message, $code,$previous);
        $this->sqlStatement = $sqlStatement;
    }

    /**
     * IacSqlException::getSqlStatement()
     *
     * @return string
     */
    public function getSqlStatement() {
        return $this->sqlStatement;
    }


}