<?php
/**
 * IacSqlInfo
 * Provides meta data and information on sql statements and database
 *
 * @package sql
 * @author Informatica Asociada
 * @copyright 2015
 * @version 1.0.0
 * @require php 5.3
 */
/*

create table vacia(
abigint1 bigint,
abigint1u bigint unsigned,
aint1 int,
aint1u int unsigned not null,
amediumint1 mediumint,
amediumint1u mediumint unsigned,
asmallint1 smallint,
asmallint1u smallint unsigned not null,
atinyint1 tinyint,
atinyint1u tinyint unsigned not null,
ayear4 year(4) COMMENT '1901 to 2155 o 0000 for invalid',


adecimal5_2 DECIMAL(5,2) not null default '0.00' COMMENT ' -999.99 to 999.99',
adecimal5_2u DECIMAL(5,2) unsigned not null default '0.00',
anumeric5_2 NUMERIC(5,2) not null default '0.00',
anumeric5_2u NUMERIC(5,2) unsigned not null default '0.00',

anumeric3_0 NUMERIC(3,0) not null default '0',
anumeric3_0u NUMERIC(3,0) unsigned not null default '0',
anumeric3 NUMERIC(3) not null default '0',
anumeric3u NUMERIC(3) unsigned not null default '0',

afloat7_4 FLOAT(7,4) not null default '0.00' COMMENT 'ie  -999.9999',
afloat7_4u FLOAT(7,4) unsigned not null default '0.00',

adouble7_4 DOUBLE(7,4) not null default '0.00' COMMENT 'ie  -999.9999',
adouble7_4u DOUBLE(7,4) unsigned not null default '0.00',

areal7_4 REAL(7,4) not null default '0.00' COMMENT 'ie  -999.9999',
areal7_4u REAL(7,4) unsigned not null default '0.00',

adouble3_0 DOUBLE(3,0) not null default '0' COMMENT ' no decs',
adouble3_0u DOUBLE(3,0) unsigned not null default '0',


abti1 bit(1),
abti16 bit(16),

adate date COMMENT '1000-01-01 to 9999-12-31 ',
adatetime datetime COMMENT '1000-01-01 to 9999-12-31 fractional seconds part in up to microseconds (6 digits) as of MySQL 5.6.4 prev ignored',
atimestamp timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '1970-01-01 00:00:01 UTC to 2038-01-19 03:14:07 UTC  fractional seconds part in up to microseconds (6 digits) as of MySQL 5.6.4 prev ignored',


achar4 char(4),
avarchar129 varchar(129),
abinary12 binary(12),
avarbinary12 varbinary(12),

atinytext tinytext,
atext text,
amediumtext mediumtext,
alongtext longtext,

atinyblob tinyblob,
ablob blob,
amediumblob mediumblob,
alongblob longblob,

geo geometry, pt point, ln linestring, poly polygon,

geo_coll geometrycollection, pt_multi multipoint, ln_multi multilinestring, poly_multi multipolygon

--ayear2 year(2) COMMENT '1 to 99, depreciated, creates year4!',

)

*/

namespace Iac\inc\sql;

/**
 * IacSqlInfo
 *
 * @author Informatica Asociada
 * @copyright 2015
 * @version 1.0.0
 * @require php 5.3
 */
class IacSqlInfo {
    protected $flags;
    protected $types;

    protected $apiType2mysql = array(
        'bigint'=> 'longlong',
        'int'=> 'long',
        'mediumint'=> 'int24',
        'smallint'=> 'short',
        'tinyint'=> 'tinyint',
        'year'=> 'year',
        'numeric'=> 'decimal',

    );

    // https://dev.mysql.com/doc/refman/5.5/en/integer-types.html
    protected $dataTypesUnsigned = array(
        'tinyint'=>array('max'=>255),
        'smallint'=>array('max'=>65535),
        'mediumint'=>array('max'=>16777215),
        'int'=>array('max'=>4294967295),
        'bigint'=>array('max'=>'18446744073709551615'),
        'year' => array('max'=>2155,'min'=>1901),
    );


    protected $dataTypesTrans = array(
        'tinyint'=>array('type'=>'tinyint', 'numeric'=>true,'bytes'=>1, 'min'=>-128, 'max'=>127,),

        'decimal' => array( 'type'=>'decimal', 'numeric'=>true, 'floatingPoint'=>true,
            ),
        'char' => array( 'type'=>'char',
            ),
        'short' => array( 'type'=>'smallint', 'numeric'=>true,'bytes'=>2, 'min'=>-32768, 'max'=>32767,
            ),
        'long' => array( 'type'=>'int', 'numeric'=>true,'bytes'=>4, 'min'=>-2147483648, 'max'=>2147483647,
            ),
        'float' => array( 'type'=>'float', 'numeric'=>true, 'floatingPoint'=>true,
            ),
        'double' => array( 'type'=>'double', 'numeric'=>true, 'floatingPoint'=>true,
            ),
        'null' => array( 'type'=>'null',
            ),
        'timestamp' => array( 'type'=>'timestamp',
            ),
        'longlong' => array( 'type'=>'bigint', 'numeric'=>true,'bytes'=>8, 'min'=>'-9223372036854775808', 'max'=>'9223372036854775807',
            ),
        'int24' => array( 'type'=>'mediumint', 'numeric'=>true,'bytes'=>3, 'min'=>-8388608, 'max'=>8388607,
            ),
        'date' => array( 'type'=>'date',
            ),
        'time' => array( 'type'=>'time',
            ),
        'datetime' => array( 'type'=>'datetime',
            ),
        'year' => array( 'type'=>'year', 'numeric'=>true,
            ),
        'newdate' => array( 'type'=>'date',
            ),
        'interval' => array( 'type'=>'interval',
            ),
        'set' => array( 'type'=>'set',
            ),
        'tiny_blob' => array( 'type'=>'tinytext', 'blob'=>true,
            ),
        'medium_blob' => array( 'type'=>'mediumtext', 'blob'=>true,
            ),
        'long_blob' => array( 'type'=>'longtext', 'blob'=>true,
            ),
        'blob' => array( 'type'=>'text', 'blob'=>true,
            ),
        'var_string' => array( 'type'=>'varchar',
            ),
        'string' => array( 'type'=>'binary',
            ),
        'geometry' => array( 'type'=>'geometry',
            ),
        'newdecimal' => array( 'type'=>'decimal', 'numeric'=>true, 'floatingPoint'=>true,
            ),
        'bit' => array( 'type'=>'bit',
            ),
    );
// Art 60 risr deduccion casa habitacion por uso o goce, depositos efvo regla 3.5.11
    /**
     * IacSqlInfo::result2fields()
     *
     * @param object $result mysqli resultset
     * @return array associative array indexed by field with field's metadata
     */
    public function result2fields($result) {
        $primary_keys = 0;
        $fields = array();
        $finfo = $result->fetch_fields();
        foreach ($finfo as $f) {
            $binary = ($f->flags & MYSQLI_BINARY_FLAG) === MYSQLI_BINARY_FLAG;

            $asMysqli = $type = $this->type2Text($f->type,$f->flags);
            // mysqli type to mysql data type

            //if(($f->flags & MYSQLI_GROUP_FLAG) && $asMysqli === 'CHAR')
            if($f->charsetnr === 63 && $asMysqli === 'CHAR')
                $type = 'TINYINT';
            elseif(($f->flags & MYSQLI_ENUM_FLAG) === MYSQLI_ENUM_FLAG)
                $type = 'ENUM';
            elseif(($f->flags & MYSQLI_SET_FLAG) === MYSQLI_SET_FLAG)
                $type = 'SET';
            elseif(!$binary && $asMysqli === 'STRING')
                $type = 'CHAR';
            elseif($binary && $asMysqli === 'VAR_STRING')
                $type = "VARBINARY";
            $type = strtolower($type);
            if(array_key_exists($type, $this->apiType2mysql))
                $type = $this->apiType2mysql[$type];

            $field = array(
                'type' => strtolower($type),
                'typeMysqli' => $asMysqli,
                'flags' => $this->flags2Array($f->flags),
                'metaData' => $f,
                //'max_chars' => $f->length, // ?
            );

           if(($f->flags & MYSQLI_PRI_KEY_FLAG) === MYSQLI_PRI_KEY_FLAG) {
                $primary_keys++;
                $field['pri_key'] = true;
                $field['pri_key_multiple'] = $primary_keys > 1; // on multi coolumn primar key the first column will be fixed after this loop
                $field['auto_increment'] = ($f->flags & MYSQLI_AUTO_INCREMENT_FLAG) === MYSQLI_AUTO_INCREMENT_FLAG;
           }
           $field['null'] = !( ($f->flags & MYSQLI_NOT_NULL_FLAG) === MYSQLI_NOT_NULL_FLAG );
           $field['required'] = !$field['null'];

            if(isset($this->dataTypesTrans[$type]))
                $field = array_merge($field,$this->dataTypesTrans[$type]);
            $type = $field['type'];

            if($type === 'char' || $type === 'varchar' || $type === 'binary' || $type === 'varbinary') {
                $field['max_chars'] = $this->charsetnr2chars($f->length,$f->charsetnr);
                $field['max_multibytes'] = $f->length;
            }
            elseif(isset($field['numeric'])) {
                $unsigned = ($f->flags & MYSQLI_UNSIGNED_FLAG) === MYSQLI_UNSIGNED_FLAG;
                $field['decimals'] = $f->decimals;
                if(isset($field['floatingPoint'])) {
                    if($f->decimals == 0) {
                        $decs = '';
                        if($type==='decimal') {
                            if($unsigned) {
                                $integers = $field['integers'] = $f->length - $f->decimals;
                            } else {
                                $integers = $field['integers'] = $f->length - $f->decimals -1;
                            }
                        } else {
                            $integers = $field['integers'] = $f->length - $f->decimals;
                        }
                    } else {
                        $decs = '.'.str_repeat('9',$f->decimals);
                        if($type==='decimal') {
                            if($unsigned) {
                                $integers = $field['integers'] = $f->length - $f->decimals -1;
                            } else {
                                $integers = $field['integers'] = $f->length - $f->decimals -2;
                            }
                        } else {
                            $integers = $field['integers'] = $f->length - $f->decimals;
                        }
                    }
                    $field['min'] = '-'.str_repeat('9',$integers).$decs;
                    $field['max'] = str_repeat('9',$integers).$decs;
                } else {
                    if($unsigned) {
                        $field['integers'] = $f->length - $f->decimals;
                    } else {
                        $field['integers'] = $f->length - $f->decimals - 1;
                    }
                }
                if($unsigned) {
                    $field['min'] = 0;
                    if(isset($this->dataTypesUnsigned[$type]))
                        $field = array_merge($field,$this->dataTypesUnsigned[$type]);
                    $field['max_chars'] = strlen($field['max']);
                } else {
                    $field['max_chars'] = strlen($field['min']);
                }
            }
            elseif(isset($field['blob'])) {
                if($binary)
                    $field['type'] = str_replace('text','blob',$field['type']);
                if($f->length > 0) {
                    $field['max_multibytes'] = $f->length;
                    if(!$binary)
                        $field['max_chars'] = $this->charsetnr2chars($f->length,$f->charsetnr);
                }
            }
            elseif($type === 'bit') {
                $field['min'] = 0;
                $field['max'] = $f->length; //FIXME
                $field['max_chars'] = $f->length;
            }
           $fields[$f->name] = $field;
        }
        // on multiple column primary key, mark the first column as multiple
        if($primary_keys > 1)
            foreach($fields as &$field)
                if(isset($field['pri_key'])) {
                    $field['pri_key_multiple'] = true;
                    break;
                }
        return $fields;
    }

    /**
     * IacSqlInfo::charsetnr2chars()
     * Returns length in characters given the length in bytes, for the given $charsetnr
     *
     * @param int $lengthInBytes length in  bytes
     * @param int $charsetnr mysql characterset code
     * @return int
     */
    protected function charsetnr2chars($lengthInBytes,$charsetnr) {
        if($charsetnr == 33) // utf8
            return $lengthInBytes/3;
        if($charsetnr == 8) // latin1
            return $lengthInBytes;
        //TODO SELECT * FROM information_schema.collations;
        return $lengthInBytes;
    }

    /**
     * iacMysqli::type2Text()
     * return mysqli type for the field
     *
     * @param integer $type_id from $result->fetch_field()->type or $result->fetch_field_direct($i)->type
     * @return string mysql type for id
     */
    public function type2Text($type_id,$flags_num) {
        // fill constants if not done yet
        $this->constantsFill();
        // unkown type
        if(!array_key_exists($type_id, $this->types))
            return null;
        return $this->types[$type_id];
    }

    /**
     * iacMysqli::flags2Array()
     * Get mysqli flags
     *
     * @param integer $flags_num from $result->fetch_field()->flags or $result->fetch_field_direct($i)->flags
     * @return array associative array with flag names indexed by mysqli constant
     */
    public function flags2Array($flags_num) {
        // fill constants if not done yet
        //if(!isset($this->flags))
        $this->constantsFill();
        $result = array();
        foreach($this->flags as $n => $t)
            if(($flags_num & $n) === $n) {
                $result[$n] = $t;
            }
        return $result;
    }

    /**
     * IacSqlInfo::constantsFill()
     * Fills mysqli's type names into $this->types and flag names into $this>$flags
     *
     * @return void
     * @see http://php.net/manual/en/mysqli-result.fetch-fields.php
     * @see http://php.net/manual/en/mysqli.constants.php
     */
    protected function constantsFill() {
        if(isset($this->types)) {
            return; // constants have been filled
        }
        $flags = array();
        $types = array();
        $constants = get_defined_constants(true); // era true
        foreach($constants['mysqli'] as $c => $n)
            if(preg_match('/^MYSQLI_TYPE_(.*)/', $c, $m))
                $types[$n] = $m[1];
            elseif(preg_match('/MYSQLI_(.*)_FLAG$/', $c, $mFlag))
                $flags[$n] = $mFlag[1];
        $this->types = $types;
        $this->flags = $flags;
    }

}
