<?php
/**
 * Returns column meta info for a table, query or result set in a standardized array
 *
 * @version 1.0.0
 */

/** @noinspection SqlNoDataSourceInspection */
/** @noinspection SqlResolve */

use Iac\inc\sql\IacMysqli;
use Iac\inc\sql\IacSqlException;
use JetBrains\PhpStorm\ArrayShape;
use JetBrains\PhpStorm\Pure;


/**
 * Class Columns
 * Returns column meta info for a table, query or result set in a standardized array
 *
 */
class Columns {
    /** @var IacMysqli */
    protected $db;

    /** @var array info limits per db datatype  */
    protected $dataTypesTrans = [
        'tinyint'=> ['type'=>'tinyint', 'numeric'=>true,'bytes'=>1, 'min'=>-128, 'max'=>127, 'max_len'=>4],
        'tinyint unsigned'=> ['type'=>'tinyint unsigned', 'numeric'=>true,'bytes'=>1, 'min'=>0, 'max'=>255, 'max_len'=>3],

        'short' => ['type'=>'smallint', 'numeric'=>true,'bytes'=>2, 'min'=>-32768, 'max'=>32767, 'max_len'=>6],
        'short unsigned' => ['type'=>'smallint unsigned', 'numeric'=>true,'bytes'=>2, 'min'=>0, 'max'=>65535, 'max_len'=>5],

        'int24' => ['type'=>'mediumint', 'numeric'=>true,'bytes'=>3, 'min'=>-8388608, 'max'=>8388607, 'max_len'=>8],
        'int24 unsigned' => ['type'=>'mediumint unsigned', 'numeric'=>true,'bytes'=>3, 'min'=>0, 'max'=>16777215, 'max_len'=>8],


        'long' => ['type'=>'int', 'numeric'=>true,'bytes'=>4, 'min'=>-2147483648, 'max'=>2147483647, 'max_len' => 11],
        'long unsigned' => ['type'=>'int unsigned', 'numeric'=>true,'bytes'=>4, 'min'=>0, 'max'=>4294967295, 'max_len'=>10],

        'longlong' => ['type'=>'bigint', 'numeric'=>true,'bytes'=>8, 'min'=>'-9223372036854775808', 'max'=>'9223372036854775807', 'max_len' => 20],
        'longlong unsigned' => ['type'=>'bigint unsigned', 'numeric'=>true,'bytes'=>8, 'min'=>'0', 'max' =>'18446744073709551615', 'max_len' => 20],


        'decimal' => ['type'=>'decimal', 'numeric'=>true],
        'decimal unsigned' => ['type'=>'decimal unsigned', 'numeric'=>true, 'min'=>0.0],

        'float' => ['type'=>'float', 'numeric'=>true],
        'float unsigned' => ['type'=>'float unsigned', 'numeric'=>true, 'min'=>0.0],

        'double' => ['type'=>'double', 'numeric'=>true],
        'double unsigned' => ['type'=>'double unsigned', 'numeric'=>true, 'min'=>0.0],

        'newdecimal' => ['type'=>'decimal', 'numeric'=>true],
        'newdecimal unsigned' => ['type'=>'decimal unsigned', 'numeric'=>true, 'min'=>0.0],

        'char' => ['type'=>'char',],
        'var_string' => ['type'=>'varchar',],
        'string' => ['type'=>'binary',],


        'null' => ['type'=>'null', 'null'=>true,
        ],

        'timestamp' => ['type'=>'timestamp',],

        'date' => ['type'=>'date',
        ],
        'time' => ['type'=>'time',
        ],
        'datetime' => ['type'=>'datetime',
        ],
        'year' => ['type'=>'year', 'numeric'=>true, 'min'=>1901, 'max'=>2155, 'max_chars'=>4],

        'newdate' => ['type'=>'date',
        ],
        'interval' => ['type'=>'interval',
        ],

        'enum' => ['type'=>'enum',],
        'set' => ['type'=>'set',],

        'tiny_blob' => ['type'=>'tinytext', 'blob'=>true,
        ],
        'medium_blob' => ['type'=>'mediumtext', 'blob'=>true,
        ],
        'long_blob' => ['type'=>'longtext', 'blob'=>true, 'noTrim'=>true,
        ],
        'blob' => ['type'=>'text', 'blob'=>true, 'noTrim'=>true,
        ],

        'geometry' => ['type'=>'geometry', 'noTrim'=>true,
        ],

        'bit' => ['type'=>'bit',],
    ];

    /** @var mysqli's type names from php constants */
    protected $types;

    /** @var array  caches show column command per table */
    protected $showColumnCache= [];

    /** @var array control field name and usate, data controls for insert, update,
     *   login like registerd, last_changed, last_login
     *  [ 'lower case name'=>usage, ] usage = insert|update|login|..
     */
    protected $controlFields = [
        'alta_db' => 'insert', 'registrado_el' => 'insert',
        'registerd' => 'insert', 'registered' => 'insert',
        'altaDb' => 'insert', 'altadb' => 'insert', 'registradoEl' => 'insert',
        'alta_por' => 'insert', 'registrado_por' => 'insert', 'registerd_by' => 'insert',
        'altaPor' => 'insert', 'registradoPor' => 'insert',
        'registerdBy' => 'insert', 'registeredBy' => 'insert',

        'ultimo_cambio' => 'update', 'cambiado_el' => 'update', 'last_changed' => 'update',
        'updated' => 'update', 'updated_by' => 'update',
        'ultimoCambio' => 'update', 'cambiadoEl' => 'update', 'lastChanged' => 'update',
        'updatedBy' => 'update',
        'ultimo_cambio_por' => 'update', 'cambiado_por' => 'update',
        'last_changed_by' => 'update',
        'ultimoCambioPor' => 'update', 'cambiadoPor' => 'update',
        'lastChangedBy' => 'update',

        'logins'=>'user', 'last_login'=>'user',
        'lastLogin'=>'user',
        'ultimo_login'=>'user', 'penultimo_login'=>'user', 'antepenultimo_login'=>'user',
        'ultimoLogin'=>'user', 'penultimoLogin'=>'user', 'antepenultimoLogin'=>'user',
    ];

    /**
     * Columns constructor.
     *
     * @param IacMysqli $db
     */
    public function __construct(IacMysqli $db) {
        $this->db = $db;
        $this->constantsFill();
        $this->dataTypesTrans = array_merge(
            array_combine(array_column($this->dataTypesTrans, 'type'), $this->dataTypesTrans),
            $this->dataTypesTrans
        );
    }

    /** functionality */

    /**
     * Get standarized column info array for table
     *   issues query SELECT * FROM $table $alias LIMIT 0
     *
     * @param string $table
     * @param string $alias optional alias for table
     * @return array
     * @noinspection PhpUnused
     * @throws IacSqlException
     */
    #[ArrayShape(['columns' => "array", 'tables' => "array", 'primaryKey' => "array"])]
    public function columnsForTable(string $table, string $alias=''):array {
        return $this->columnsForQuery(
            'SELECT * FROM '.fieldit($table).
            ( empty($alias) ? '' : ' ' . fieldit($alias)) .
            ' LIMIT 0'." /*".__METHOD__."*/"
        );
    }

    /**
     * Get standarized column info array select statement suggested: LIMIT 0
     *
     * @param string $selectQuery
     * @return array
     * @throws IacSqlException
     */
    #[ArrayShape(['columns' => "array", 'tables' => "array", 'primaryKey' => "array"])]
    public function columnsForQuery(string $selectQuery):array {
        $resultSet = $this->db->mysqli->query($selectQuery);
        $columns = $this->columns($resultSet);
        $resultSet->free();
        return $columns;
    }

    /**
     * @param mysqli_result $resultSet
     * @return array
     * @noinspection PhpUnused
     * @throws IacSqlException
     */
    #[ArrayShape(['columns' => "array", 'tables' => "array", 'primaryKey' => "array"])]
    public function columns(mysqli_result $resultSet):array {
        if(empty($resultSet)) {
            throw new IacSqlException("Invalid query for column meta-data");
        }
        $tables = [];
        $primaryKeys = [];
        $fields = [];

        $fieldInfo = $resultSet->fetch_fields();
        foreach ($fieldInfo as $f) {
            if($f === false) {
                $fields[] = [

                ];
                continue;
            }
            if(!array_key_exists($f->orgtable, $tables)) {
                $tables[$f->orgtable] = [];
            }
            $type = $this->deduceType($f);

            $column = array_merge(
                [
                    'name' => $f->name,
                    'table' => $f->table,
                    'orgname' => $f->orgname,
                    'orgtable' => $f->orgtable,
                    'label' => $this->toLabel($f->name),

                    'type' => $type,

                    'null' => !(($f->flags & MYSQLI_NOT_NULL_FLAG) === MYSQLI_NOT_NULL_FLAG),
                    'required' => ($f->flags & MYSQLI_NOT_NULL_FLAG) === MYSQLI_NOT_NULL_FLAG,

                ],
                array_key_exists($type, $this->dataTypesTrans) ? $this->dataTypesTrans[$type] : ['Not Deduced'=>true]
            );

            if($type === 'char' || $type === 'varchar' || $type === 'binary' || $type === 'varbinary') {
                $column['max_chars'] = $this->charsetnr2chars($f->length, $f->charsetnr);
                $column['max_multibytes'] = $f->length;
            } elseif(!empty($column['numeric'])) {
                $this->numericField($f, $column);
            } elseif( ($f->flags & MYSQLI_ENUM_FLAG) === MYSQLI_ENUM_FLAG || ($f->flags & MYSQLI_SET_FLAG) === MYSQLI_SET_FLAG ) {
                $column['values'] = $this->enumSetValues(5, $f->orgtable, $f->orgname);
            } elseif($type === 'bit') {
                $column['min'] = 0;
                $column['max'] = $f->length;
                $column['max_chars'] = $f->length;
            }

            if(MYSQLI_NO_DEFAULT_VALUE_FLAG !== ($f->flags & MYSQLI_NO_DEFAULT_VALUE_FLAG) && !empty($f->orgtable)) {
                $column['default'] = $this->get_default($f->orgtable, $f->orgname);
            }

            if(array_key_exists($f->orgname, $this->controlFields)) {
                $column['controlField'] = $this->controlFields[$f->orgname];
                $column['readOnly'] = true;
            }

            $this->virtualColumn($f->orgtable, $f->orgname, $column);

            if(($f->flags & MYSQLI_PRI_KEY_FLAG) === MYSQLI_PRI_KEY_FLAG) {
                $tables[$f->orgtable]['primaryKey'][$f->orgname] = $f->name;
                $primaryKeys["$f->orgtable.$f->orgname"] = $f->name;
                $column['primaryKey'] = "$f->orgtable.$f->orgname";
                $column['readOnly'] = true;
            }
            if(($f->flags & MYSQLI_AUTO_INCREMENT_FLAG) === MYSQLI_AUTO_INCREMENT_FLAG) {
                $column['auto_increment'] = $column['null'] = true;
                $column['required'] = false;
                $column['readOnly'] = true;
            }

            $fields[$f->name] = $column;
        }
        return ['columns'=>$fields, 'tables'=>$tables, 'primaryKey'=>$primaryKeys];
    }

    /**
     * Retruns and caches sql SHOW COLUMNS FROM orgtable
     *
     * @param string $orgtable original table name, vs query alias
     * @return array
     * @throws IacSqlException
     */
    public function showColumns(string $orgtable):array {
        if(empty($orgtable)) {
            return [];
        }
        if(isset($this->showColumnCache[$orgtable])) {
            return $this->showColumnCache[$orgtable];
        }
        return $this->showColumnCache[$orgtable] =
            $this->db->selectArrayKey("SHOW COLUMNS FROM ".fieldit($orgtable)." /*".__METHOD__."*/", 'Field');
    }

    /** Getters & setters */

    /**
     * Get control fields, data controls for insert, update, login like registerd, last_changed, last_login, ...
     * are marked as readonly.
     *
     * @return array
     * @noinspection PhpUnused
     */
    public function getControlFields(): array
    {
        return $this->controlFields;
    }

    /**
     * Set control fields, data controls for insert, update, login like registerd, last_changed, last_login, ...
     *
     * @param array $controlFields
     * @noinspection PhpUnused
     */
    public function setControlFields(array $controlFields): void
    {
        $this->controlFields = $controlFields;
    }

    /**
     * get show column command per table, as generated by method showColumns
     * ['table'=>[..], ]
     *
     * @return array
     * @noinspection PhpUnused
     */
    public function getShowColumnCache(): array
    {
        return $this->showColumnCache;
    }


    /** helpers */

    /**
     * Gets the column's mysql data type
     *
     * @param object|bool $f contains field definition information or false if no info
     * @return string
     */
    protected function deduceType(object|bool $f):string {
        $binary = ($f->flags & MYSQLI_BINARY_FLAG) === MYSQLI_BINARY_FLAG;
        $type = array_key_exists($f->type, $this->types) ? $this->types[$f->type] : $f->type;
        $type = strtolower($type);
        // mysqli type to mysql data type
        //if(($f->flags & MYSQLI_GROUP_FLAG) && $asMysqli === 'char')
        if($f->charsetnr === 63 && $type === 'char')
            $type = 'tinyint';
        elseif(($f->flags & MYSQLI_ENUM_FLAG) === MYSQLI_ENUM_FLAG)
            $type = 'enum';
        elseif(($f->flags & MYSQLI_SET_FLAG) === MYSQLI_SET_FLAG)
            $type = 'set';
        elseif(!$binary && $type === 'string')
            $type = 'char';
        elseif($binary && $type === 'var_string')
            $type = "varbinary";

        if(array_key_exists($type, $this->dataTypesTrans)) {
            $type = strtolower($this->dataTypesTrans[$type]['type']);
        }
        if(($f->flags & MYSQLI_UNSIGNED_FLAG) === MYSQLI_UNSIGNED_FLAG) {
            return $type . ' unsigned';
        }
        return $type;
    }

    /**
     * Returns length in characters given the length in bytes, for the given $charsetnr
     *
     * @param int $lengthInBytes length in  bytes
     * @param int $charsetnr mysql characterset code
     * @return int length in chars
     */
    protected function charsetnr2chars(int $lengthInBytes, int $charsetnr):int {
        if($charsetnr == 255)
            return $lengthInBytes/4;
        if($charsetnr == 33) // utf8
            return $lengthInBytes/3;
        if($charsetnr == 8) // latin1
            return $lengthInBytes;
        // SELECT co.id as charsetnr, ca.maxlen FROM information_schema.collations co JOIN information_schema.character_sets ca ON ca.character_set_name=co.character_set_name;

        // SELECT co.id as charsetnr, ca.maxlen, co.CHARACTER_SET_NAME, co.COLLATION_NAME FROM information_schema.collations co JOIN information_schema.character_sets ca ON ca.character_set_name=co.character_set_name;
        // utf8mb4, utf8mb4_0900_ai_ci
        return $lengthInBytes;
    }

    /**
     * Sets column numeric attributes if column is a number
     *
     * @param object|false $f contains field definition information or false if no info
     * @param array $column
     */
    protected function numericField(object|false $f, array &$column) {
        if(array_key_exists($column['type'],['timestamp'=>true,'year'=>true])) {
            $column['coma'] = false;
        } else {
            $column['coma'] = !$this->nonComaFieldNames($f);
        }
        if(!empty($f->decimals)) {
            if($f->decimals <= 0) {
                $column['integers'] = $f->length;
            } else {
                $column['integers'] = $f->length - $f->decimals -1;
            }
            $integers = 0 === $column['integers'] ? "0" : str_repeat('9',$column['integers']);
            $column['decimals'] = $f->decimals;
            if($this->isLatitude($f)) {
                $column['max'] = 90.00;
                $column['min'] = -90.00;
            } elseif($this->isLongitude($f)) {
                $column['max'] = 180.00;
                $column['min'] = -180.00;
            } else {
                $decs = $f->decimals == 0 ? '' : '.'.str_repeat('9',$f->decimals);
                $column['max'] = $integers.$decs;
                $column['min'] = ($f->flags & MYSQLI_UNSIGNED_FLAG) === MYSQLI_UNSIGNED_FLAG ?
                    0 : '-'.$integers.$decs;
            }
            $column['max_chars'] = max(strlen($column['max']),strlen($column['min']));
        }
    }

    /**
     * Determines if all or part of a column's named is in $lowerCaseWords
     *
     * @param object|false $f contains field definition information or false if no info
     * @param array $lowerCaseWords
     * @return bool true if it is named as an element of $lowerCaseWords
     */
    protected function containsWord(object|false $f, array $lowerCaseWords):bool {
        $name = strtolower($f->name);
        if(in_array($name, $lowerCaseWords,  true)) {
            return true;
        }
        foreach($lowerCaseWords as $word) {
            if($name === $word) {
                return true;
            }
            if(str_contains($name, '_' . $word)) {
                return true;
            }
            if(str_contains($name, $word . '_')) {
                return true;
            }
        }
        return false;
    }

    /**
     * Does the column represent latitude
     *
     * @param object|false $f contains field definition information or false if no info
     * @return bool
     */
    #[Pure] protected function isLatitude(object|false $f):bool {
        return $this->containsWord($f, ['lat','latitude','latitud']);
    }

    /**
     * Does the column represent longitud
     *
     * @param object|false $f contains field definition information or false if no info
     * @return bool
     */
    #[Pure] protected function isLongitude(object|false $f):bool {
        return $this->containsWord($f, ['lon', 'lng', 'long', 'longitude', 'longitud']);
    }

    /**
     * The number should not be formatead with thousands separators
     *
     * @param object|false $f contains field definition information or false if no info
     * @return bool true the number should not be formatead with thousands separators
     */
    #[Pure] protected function nonComaFieldNames(object|false $f):bool {
        return $this->containsWord($f, ['ano', 'year',]);
    }

    /**
     * Returns an array with the enum or set values
     *
     * @param string|int $typeLength
     * @param string $orgtable original table name, vs query alias
     * @param string $orgname orginal column name in table vs query alias
     * @return array
     * @throws IacSqlException
     */
    protected function enumSetValues(string|int $typeLength, string $orgtable, string $orgname):array {
        $tableInfo = $this->showColumns($orgtable);
        if(empty($tableInfo[$orgname])) {
            return [];
        }
        $values = $this->csv2KeyArray(substr($tableInfo[$orgname]['Type'], $typeLength, -1));
        foreach($values as $key => &$v) {
            $v['label'] = $this->toLabel($key);
        }
        return $values;
    }

    /**
     * Marks a virtual column as virtual and read only
     *
     * @param string $orgtable original table name, vs query alias
     * @param string $orgname orginal column name in table vs query alias
     * @param array $column
     * @throws IacSqlException
     */
    protected function virtualColumn(string $orgtable, string $orgname, array &$column) {
        if(empty($orgtable) || empty($orgname)) {
            return;
        }
        $tableInfo = $this->showColumns($orgtable);
        if(!empty($tableInfo[$orgname]['GENERATION_EXPRESSION'])) {
            $column['virtual'] = $tableInfo[$orgname]['GENERATION_EXPRESSION'];
            $column['readOnly'] = true;
            return;
        }
        if(!empty($tableInfo[$orgname]['Extra'])) {
            if(strpos($tableInfo[$orgname]['Extra'], 'VIRTUAL')) {
                $column['virtual'] = $tableInfo[$orgname]['Extra'];
                $column['readOnly'] = true;
            }
        }
    }

    /**
     * Gets the sql default value for the column
     *
     * @param string $orgtable original table name, vs query alias
     * @param string $orgname orginal column name in table vs query alias
     * @return string
     * @throws IacSqlException
     */
    protected function get_default(string $orgtable, string $orgname):string|null  {
        $tableInfo = $this->showColumns($orgtable);
        return $tableInfo[$orgname]['Default'];
    }

    /**
     * Converts a csv string to an array
     * example (12,2) for decimal type, ('enum1',...)
     *
     * @param string $csv
     * @return array
     */
    protected function csv2KeyArray(string $csv):array {
        return array_fill_keys( array_values( str_getcsv($csv, ',', "'", "'") ), [] );
    }

    /**
     * Fills mysqli's type names into $this->types and flag names into $this->$flags
     *
     * @return void
     * @see http://php.net/manual/en/mysqli-result.fetch-fields.php
     * @see http://php.net/manual/en/mysqli.constants.php
     */
    protected function constantsFill() {
        if(isset($this->types)) {
            return; // constants have been filled
        }
        if(version_compare(PHP_VERSION, '7.3.6', '<=') ) {
            $this->types = [
                0 => 'DECIMAL',
                1 => 'CHAR',
                2 => 'SHORT',
                3 => 'LONG',
                4 => 'FLOAT',
                5 => 'DOUBLE',
                6 => 'NULL',
                7 => 'TIMESTAMP',
                8 => 'LONGLONG',
                9 => 'INT24',
                10 => 'DATE',
                11 => 'TIME',
                12 => 'DATETIME',
                13 => 'YEAR',
                14 => 'NEWDATE',
                247 => 'INTERVAL',
                248 => 'SET',
                249 => 'TINY_BLOB',
                250 => 'MEDIUM_BLOB',
                251 => 'LONG_BLOB',
                252 => 'BLOB',
                253 => 'VAR_STRING',
                254 => 'STRING',
                255 => 'GEOMETRY',
                245 => 'JSON',
                246 => 'NEWDECIMAL',
                16 => 'BIT',
            ];
            return;
        }
        $types = array();
        $constants = get_defined_constants(true);
        foreach($constants['mysqli'] as $c => $n)
            if(preg_match('/^MYSQLI_TYPE_(.*)/', $c, $m))
                $types[$n] = $m[1];
        // elseif(preg_match('/MYSQLI_(.*)_FLAG$/', $c, $mFlag)) $flags[$n] = $mFlag[1];
        $this->types = $types;
    }

    /** @noinspection PhpUnused */
    /**
     * @throws IacSqlException
     */
    protected function getRelatedTables(array $columns): array
    {
        $tables = $this->db->selectKeyValue(
            "SELECT table_name,  table_name FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA=database()  /* ".__METHOD__."*/"
        );
        $related = [];
        foreach($columns['columns'] as $fieldName => $column) {
            $fieldSnake = $this->camel2snake($fieldName);
            if(!empty($column['primaryKey']) || strcasecmp("_id", substr($fieldSnake, -3)) !== 0) {
                continue;
            }
            $foundTable = $this->deduceRelatedTable(substr($fieldSnake, 0, -3), $tables);
            if($foundTable === null) {
                $related[$fieldName] = [];
            } else {
                $related[$fieldName] = [
                    'join' => [$foundTable => $foundTable.'_id'],
                    'field' => $foundTable
                ];
            }
        }
        return $related;
    }

    protected function deduceRelatedTable(string $tableSnake, array $tables):string|null {
        $foundTable = $this->arrayKeyInsensitive($tableSnake, $tables);
        if($foundTable !== null) {
            return $foundTable;
        }
        $foundTable = $this->arrayKeyInsensitive($this->snake2camel($tableSnake), $tables);
        if($foundTable !== null) {
            return $foundTable;
        }
        return null;
    }

    /**
     *  get $array[$case_insensitive_key]
     *
     * @param array $array
     * @param string $key
     * @param mixed $notFoundValue
     * @return mixed the value for $array[$case_insensitive_key]
     */
    protected function arrayKeyInsensitive(string $key, array $array, mixed $notFoundValue = null):mixed {
        if(array_key_exists($key, $array))
            return $array[$key];
        $keyLower = strtolower($key);
        $case_insensitive = array_change_key_case($array, CASE_LOWER);
        return array_key_exists($keyLower, $case_insensitive) ? $case_insensitive[$keyLower] : $notFoundValue;
    }

    protected function camel2snake(string $s):string {
        if(empty($s)) {
            return $s;
        }
        $s[0] = mb_convert_case($s[0], MB_CASE_LOWER);
        return mb_convert_case(preg_replace("/([A-ZÁÉÍÓÚÑ])/u", "_$1", $s), MB_CASE_LOWER);
    }

    protected function snake2camel(string $s):string {
        if(empty($s)) {
            return $s;
        }
        $words = [];
        $string = str_replace(" ", "_", strim($s));
        foreach(explode('_',$string) as $w) {
            $words[] =  mb_convert_case($w, MB_CASE_TITLE);
        }
        $word = implode('', $words);
        $word[0] = mb_convert_case($word[0], MB_CASE_LOWER);
        return $word;
    }

    protected function toLabel(string $s):string {
        $s = $this->camel2snake($s);
        $label = preg_replace('/[\s_]+/uiS', ' ', $s);
        if(empty($label)) {
            $label = str_replace('_',' ',$s);
        }
        return ucwords($label);
    }
}