<?php
/**
 * transfer_files.php
 * Directory copy utility for iaCase
 * Copies files from origin to destination with overwrite
 */

require_once("inc/config.php");

global $gIApath, $gIAsql;

// Excluded folders - skip these during copy
$EXCLUDED = array('.git', '.idea', '.vscode', 'node_modules', '__pycache__', '.DS_Store');

////////////////////////////////////
// Get parameters with session persistence
////////////////////////////////////
$h = param('h');
$origin = param('origin', isset($_SESSION['transfer_origin']) ? $_SESSION['transfer_origin'] : '/lamp/www/quantixTest/');
$destination = param('destination', isset($_SESSION['transfer_destination']) ? $_SESSION['transfer_destination'] : '/lamp/www/quantix/');

// Ensure trailing slash
if (substr($origin, -1) != '/') $origin .= '/';
if (substr($destination, -1) != '/') $destination .= '/';

// Save to session
$_SESSION['transfer_origin'] = $origin;
$_SESSION['transfer_destination'] = $destination;

// Results storage
$results = array(
    'files' => 0,
    'dirs' => 0,
    'skipped' => 0,
    'errors' => 0,
    'messages' => array(),
    'file_list' => array()
);
$msgErr = '';

////////////////////////////////////
// Validation function
////////////////////////////////////
function validate_path($path) {
    // Must start with /lamp/www/
    if (strpos($path, '/lamp/www/') !== 0) {
        return "Path must start with /lamp/www/";
    }
    // No directory traversal
    if (strpos($path, '..') !== false) {
        return "Path cannot contain '..'";
    }
    // Must exist for origin
    return true;
}

////////////////////////////////////
// Recursive copy function
////////////////////////////////////
function copy_directory_recursive($src, $dst, &$results, $preview = false, $excluded = array()) {
    // Validate source exists
    if (!is_dir($src)) {
        $results['errors']++;
        $results['messages'][] = "Source directory does not exist: $src";
        return false;
    }

    // Create destination if not exists (unless preview)
    if (!$preview && !is_dir($dst)) {
        if (!mkdir($dst, 0755, true)) {
            $results['errors']++;
            $results['messages'][] = "Failed to create directory: $dst";
            return false;
        }
        $results['dirs']++;
        $results['messages'][] = "Created directory: $dst";
    }

    // Open source directory
    $dir = opendir($src);
    if ($dir === false) {
        $results['errors']++;
        $results['messages'][] = "Failed to open directory: $src";
        return false;
    }

    // Loop through files
    while (($file = readdir($dir)) !== false) {
        // Skip . and ..
        if ($file == '.' || $file == '..') {
            continue;
        }

        // Skip excluded folders
        if (in_array($file, $excluded)) {
            $results['skipped']++;
            continue;
        }

        $srcPath = $src . $file;
        $dstPath = $dst . $file;

        if (is_dir($srcPath)) {
            // Recurse into subdirectory
            copy_directory_recursive($srcPath . '/', $dstPath . '/', $results, $preview, $excluded);
        } else {
            // Copy file
            if ($preview) {
                $results['files']++;
                $results['file_list'][] = $srcPath . ' -> ' . $dstPath;
            } else {
                if (copy($srcPath, $dstPath)) {
                    $results['files']++;
                } else {
                    $results['errors']++;
                    $results['messages'][] = "Failed to copy: $srcPath";
                }
            }
        }
    }

    closedir($dir);
    return true;
}

////////////////////////////////////
// Execute action
////////////////////////////////////
$actionResult = '';

if ($h == 'preview' || $h == 'copy') {
    // Validate paths
    $originValid = validate_path($origin);
    $destValid = validate_path($destination);

    if ($originValid !== true) {
        $msgErr .= "<li>Origin: $originValid</li>";
    }
    if ($destValid !== true) {
        $msgErr .= "<li>Destination: $destValid</li>";
    }
    if (!is_dir($origin)) {
        $msgErr .= "<li>Origin directory does not exist: $origin</li>";
    }

    // Check not copying to itself
    if ($origin == $destination) {
        $msgErr .= "<li>Origin and destination cannot be the same!</li>";
    }

    if (empty($msgErr)) {
        $preview = ($h == 'preview');
        copy_directory_recursive($origin, $destination, $results, $preview, $EXCLUDED);

        if ($preview) {
            $actionResult = "<span style='color:blue;'>Preview complete - no files were copied</span>";
        } else {
            if ($results['errors'] == 0) {
                $actionResult = "<span style='color:green;'>Copy complete!</span>";
            } else {
                $actionResult = "<span style='color:orange;'>Copy completed with errors</span>";
            }
        }
    }
}

?>
<!DOCTYPE html>
<html>
<head>
<title>Transfer Files - iaCase</title>
<link href="template/css/iastyles.css" rel="stylesheet" type="text/css" />
<style>
HR { color: #800000; }
TH { color: #800000; }
.success { color: green; }
.error { color: red; }
.info { color: blue; }
.file-list {
    max-height: 400px;
    overflow-y: auto;
    background: #f5f5f5;
    padding: 10px;
    font-family: monospace;
    font-size: 12px;
    border: 1px solid #ddd;
}
.stats {
    background: #e8f5e9;
    padding: 10px;
    margin: 10px 0;
    border-left: 4px solid #4caf50;
}
.stats.has-errors {
    background: #ffebee;
    border-left-color: #f44336;
}
</style>
</head>
<body>

<nav>
    <a href="#form">Form</a> |
    <a href="#results">Results</a> |
    <a href="ia_case.php">Back to iaCase</a>
    <?php if (!empty($msgErr)) echo " | <span style='color:red;'>Errors**</span>"; ?>
</nav>

<hr/>

<h2>Transfer Files</h2>
<p>Copy files from origin to destination directory. Existing files will be overwritten.</p>

<?php if (!empty($msgErr)): ?>
<div style="color:red; margin: 10px 0;">
    <strong>Errors:</strong>
    <ol><?php echo $msgErr; ?></ol>
</div>
<?php endif; ?>

<hr/>
<a name="form"></a>

<form method="get">
<table class="tabla">
    <tr>
        <th>Origin</th>
        <td><input type="text" name="origin" value="<?php echo htmlentities($origin); ?>" size="60" /></td>
        <td><input type="submit" value="Set" /></td>
    </tr>
    <tr>
        <th>Destination</th>
        <td><input type="text" name="destination" value="<?php echo htmlentities($destination); ?>" size="60" /></td>
        <td><input type="submit" value="Set" /></td>
    </tr>
    <tr>
        <td colspan="3">
            <hr/>
            <input type="submit" name="h" value="preview" style="padding: 5px 15px;" />
            <input type="submit" name="h" value="copy" style="padding: 5px 15px; background: #4caf50; color: white; border: none; cursor: pointer;"
                   onclick="return confirm('Are you sure you want to copy all files? This will overwrite existing files in the destination.');" />
        </td>
    </tr>
</table>
</form>

<p><small>
    <strong>Excluded folders:</strong> <?php echo implode(', ', $EXCLUDED); ?>
</small></p>

<hr/>
<a name="results"></a>

<?php if ($h == 'preview' || $h == 'copy'): ?>
<h3>Results <?php echo $actionResult; ?></h3>

<div class="stats <?php echo ($results['errors'] > 0) ? 'has-errors' : ''; ?>">
    <strong>Summary:</strong><br/>
    Files <?php echo ($h == 'preview') ? 'to copy' : 'copied'; ?>: <strong><?php echo $results['files']; ?></strong><br/>
    Directories created: <strong><?php echo $results['dirs']; ?></strong><br/>
    Folders skipped (excluded): <strong><?php echo $results['skipped']; ?></strong><br/>
    Errors: <strong style="color: <?php echo ($results['errors'] > 0) ? 'red' : 'green'; ?>"><?php echo $results['errors']; ?></strong>
</div>

<?php if (!empty($results['messages'])): ?>
<h4>Messages:</h4>
<ul>
<?php foreach ($results['messages'] as $msg): ?>
    <li><?php echo htmlentities($msg); ?></li>
<?php endforeach; ?>
</ul>
<?php endif; ?>

<?php if ($h == 'preview' && !empty($results['file_list'])): ?>
<h4>Files to be copied (<?php echo count($results['file_list']); ?>):</h4>
<div class="file-list">
<?php foreach ($results['file_list'] as $file): ?>
<?php echo htmlentities($file); ?><br/>
<?php endforeach; ?>
</div>
<?php endif; ?>

<?php endif; ?>

<hr/>
<div align="right"><a href="#form">Top</a></div>

</body>
</html>
