# Importer Fixes Applied - January 10, 2026

## Summary
Fixed 3 bugs that were causing the importer tracker and importer to fail.

---

## Bug #1: ImportLogger "No Database Selected" Error

**File:** `/lamp/www/importer/lib/ImportLogger.php`
**Method:** `ensureLogTablesExist()`

### Problem
- DatabaseHelper connected without selecting any database
- `ensureLogTablesExist()` was called before database was selected
- MySQL threw "No database selected" error when trying to create log tables

### Solution
Added automatic database selection at the start of `ensureLogTablesExist()`:

```php
// Check if a database is currently selected
$result = $conn->query("SELECT DATABASE()");
if ($result && $result->num_rows > 0) {
    $row = $result->fetch_row();
    $currentDb = $row[0] ?? null;

    // If no database is selected, select default database
    if ($currentDb === null) {
        if (defined('DB_NAME') && DB_NAME !== '') {
            $conn->select_db(DB_NAME);
            ErrorHandler::logError('Auto-selected default database: ' . DB_NAME);
        }
    }
}
```

### Evidence
**Before:**
```
[2026-01-09 22:09:35] ERROR: Failed to ensure log tables exist: No database selected
```

**After:**
```
[2026-01-09 23:23:06] ERROR: Auto-selected default database: importer_db
```

---

## Bug #2: ProgressTracker Resets Percent to 0 on Phase Change

**File:** `/lamp/www/importer/lib/ProgressTracker.php`
**Method:** `setPhase()` (line 123)

### Problem
When phase changed (e.g., parsing → ready → inserting), percent was hardcoded to 0:

```php
$data['percent'] = 0; // Reset percent on phase change  ← BUG!
```

### Real-World Scenario
1. Upload parses all 571 rows → 100% complete
2. Calls `setPhase($key, 'ready', 'Ready to insert')`
3. Percent resets to 0 even though all rows are processed
4. Progress bar shows 0% to user → looks broken!

### Solution
Recalculate percent instead of resetting:

```php
// Recalculate percent based on current progress instead of resetting to 0
$data['percent'] = $data['total'] > 0 ? round(($data['current'] / $data['total']) * 100) : 0;
```

### Evidence
**Before:**
```json
{
  "current": 571,
  "total": 571,
  "percent": 0,  ← Should be 100!
  "phase": "ready"
}
```

**After:**
```json
{
  "current": 571,
  "total": 571,
  "percent": 100,  ✅ Correct!
  "phase": "ready"
}
```

---

## Bug #3: Infinite Polling When Upload Completes

**File:** `/lamp/www/importer/arrival.php`
**Location:** Line 1912 (ProgressMonitor polling logic)

### Problem
JavaScript only stopped polling when `status === 'completed'`, but upload.php leaves status as `'in_progress'` with phase `'ready'` after parsing completes. This caused infinite polling - the browser called getProgress.php over and over forever!

### Evidence from User
```
https://dev-app.filemonprime.net/importer/api/getProgress.php?key=prog_69618f61d181c0.31908286 
→ Called 1000+ times after upload finished
→ Stuck at "Ready 100% 571 of 571 Ready to insert"
```

### Solution
Also recognize 'ready' phase at 100% as completion for uploads:

```javascript
// Stop polling when progress is completed OR when upload is ready (100% at ready phase)
if (data.progress.status === 'completed' ||
   (data.progress.phase === 'ready' && data.progress.percent === 100)) {
    stop();
    if (onComplete) onComplete(data.progress);
    return;
}
```

### Behavior Change
- **Old:** Only stopped when `status === 'completed'` (never happens after upload)
- **New:** Stops when `status === 'completed'` OR `phase === 'ready' AND percent === 100`

---

## Test Results

All tests passing:

| Test | Result |
|-------|--------|
| `test_progress_system.php` | ✅ 7/7 PASS |
| `test_integration_flow.php` | ✅ PASS |
| `ImportLogger::ensureLogTablesExist()` | ✅ PASS |
| `ProgressTracker::setPhase()` | ✅ PASS (preserves 51%) |
| Real upload flow simulation | ✅ 100% preserved after phase change |
| Polling logic verification | ✅ Stops at ready/100% |

---

## Files Modified

1. `/lamp/www/importer/lib/ImportLogger.php` - Added database selection logic
2. `/lamp/www/importer/lib/ProgressTracker.php` - Fixed percent reset bug
3. `/lamp/www/importer/arrival.php` - Fixed infinite polling
4. `/lamp/www/importer/test_progress_system.php` - Updated test expectations
5. `/lamp/www/importer/arrival.php.backup` - Backup created

---

## Verification

✅ No more "No database selected" errors in logs
✅ Progress bar shows accurate percentage during uploads/inserts
✅ Polling stops correctly when upload completes (ready/100%)
✅ Schema memory functionality working
✅ All import operations logging successfully

---

## How to Test

1. **Upload a file** via the web interface
2. **Watch progress bar** - should show accurate percentage
3. **When parsing completes** - progress bar should stop and show data preview
4. **Insert data** - progress should continue and complete properly
5. **No infinite polling** - getProgress.php should not be called repeatedly after completion

---

## Notes

- Progress files are stored in `/tmp/importer_progress_*.json`
- Import log files stored in `/lamp/www/importer/logs/importer.log`
- Default database is `importer_db`
- Progress tracking uses file-based storage for thread safety
