# 📊 Import Logger System - Complete Guide

**Version:** 1.0  
**Database:** `importer_system`  
**Status:** ✅ Operational

---

## 🎯 Overview

The Import Logger is a comprehensive audit system that tracks **every import operation** in the Cyberpunk Data Importer. It captures file metadata, schema definitions, CREATE TABLE operations, insertion statistics, and error details.

### Key Features

✅ **Automatic table creation** - Creates log tables on first use  
✅ **Version management** - Zero-data-loss migrations  
✅ **Full schema capture** - Logs every column with complete metadata  
✅ **Error tracking** - Records failures at row and operation level  
✅ **CSV/XLSX support** - Handles both file types with specific metadata  
✅ **Easy retrieval** - Query methods for Step 2 analysis  

---

## 🗄️ Database Schema

### Database: `importer_system`

All import logs are stored in a dedicated system database separate from user data.

### Table 1: `import_logs` (Master Log)

**Primary Key:** `import_log_id` (CHAR 36, UUID)

| Column | Type | Description |
|--------|------|-------------|
| `import_log_id` | CHAR(36) | UUID primary key |
| `database_name` | VARCHAR(64) | Target database name |
| `table_name` | VARCHAR(64) | Target table name |
| `file_name` | VARCHAR(255) | Original uploaded filename |
| `file_type` | ENUM('XLSX','CSV') | File type |
| `file_size` | INT | File size in bytes |
| `operation_status` | ENUM | 'pending', 'success', 'error', 'partial' |
| `create_table_status` | ENUM | 'pending', 'created', 'already_exists', 'error' |
| `create_table_sql` | TEXT | Full CREATE TABLE statement |
| `create_table_error` | TEXT | Error message if CREATE TABLE failed |
| `rows_processed` | INT | Total rows in file |
| `rows_inserted` | INT | Successfully inserted rows |
| `rows_failed` | INT | Failed row count |
| `error_details` | JSON | Detailed error information |
| `csv_delimiter` | VARCHAR(10) | Detected CSV delimiter (CSV only) |
| `csv_encoding` | VARCHAR(20) | Detected encoding (CSV only) |
| `csv_normalized_rows` | INT | Rows normalized (CSV only) |
| `alta_db` | TIMESTAMP | Import timestamp |
| `alta_por` | VARCHAR(32) | User who performed import |
| `ultimo_cambio` | TIMESTAMP | Last modification |
| `ultimo_cambio_por` | VARCHAR(32) | Last modifier |

**Indexes:**
- `idx_database_name` - Fast filtering by database
- `idx_table_name` - Fast filtering by table
- `idx_status` - Fast filtering by status
- `idx_alta_db` - Chronological ordering

---

### Table 2: `import_schema_logs` (Schema Details)

**Primary Key:** `import_schema_log_id` (CHAR 36, UUID)  
**Foreign Key:** `import_log_id` → `import_logs.import_log_id` (CASCADE DELETE)

| Column | Type | Description |
|--------|------|-------------|
| `import_schema_log_id` | CHAR(36) | UUID primary key |
| `import_log_id` | CHAR(36) | FK to import_logs |
| `column_name` | VARCHAR(64) | Column name (sanitized) |
| `column_order` | INT | Position in table (1-based) |
| `data_type` | VARCHAR(50) | MySQL data type (VARCHAR, INT, etc.) |
| `length_values` | VARCHAR(255) | Length/values (255, ('A','B'), 10,2) |
| `default_value` | VARCHAR(255) | Default value if specified |
| `is_nullable` | BOOLEAN | NULL constraint |
| `is_indexed` | BOOLEAN | Whether column has an index |
| `column_comment` | TEXT | Column comment/description |
| `was_auto_detected` | BOOLEAN | Auto-detected vs manual |
| `original_column_name` | VARCHAR(64) | Original name before sanitization |
| `alta_db` | TIMESTAMP | Record creation timestamp |
| `alta_por` | VARCHAR(32) | User who created record |

**Indexes:**
- `idx_import_log_id` - Fast lookup by import
- `idx_column_name` - Fast column searches

---

### Table 3: `import_logger_meta` (Version Control)

| Column | Type | Description |
|--------|------|-------------|
| `meta_key` | VARCHAR(64) | Metadata key (PRIMARY KEY) |
| `meta_value` | TEXT | Metadata value |
| `ultima_actualizacion` | TIMESTAMP | Last update timestamp |

**Current Version:** 1

---

## 🔧 API Reference

### Class: `ImportLogger`

Location: `/lamp/www/importer/lib/ImportLogger.php`

#### Constants

```php
const CURRENT_VERSION = 1;
const LOG_DATABASE = 'importer_system';
```

---

#### Method: `ensureLogTablesExist()`

**Description:** Creates log tables if missing, migrates if version outdated.

**Returns:** `bool` - True on success

**Usage:**
```php
ImportLogger::ensureLogTablesExist();
```

**Note:** Called automatically by other methods. Manual invocation not required.

---

#### Method: `startImport($metadata)`

**Description:** Creates a new import log entry and returns the ID.

**Parameters:**
```php
$metadata = [
    'file_name' => 'productos.xlsx',     // Required
    'file_type' => 'XLSX',               // Required: 'XLSX' or 'CSV'
    'file_size' => 123456,               // Optional: bytes
    'database_name' => 'ventas_db',      // Optional: set later
    'table_name' => 'productos',         // Optional: set later
    'alta_por' => 'system',              // Optional: defaults to 'system'
    
    // CSV-specific (optional)
    'csv_delimiter' => 'comma',
    'csv_encoding' => 'UTF-8',
    'csv_normalized_rows' => 5
];
```

**Returns:** `string` - UUID of created import log

**Usage:**
```php
$importLogId = ImportLogger::startImport($metadata);
$_SESSION['import_log_id'] = $importLogId; // Store for later use
```

---

#### Method: `logCreateTable($importLogId, $data)`

**Description:** Logs CREATE TABLE operation (success or failure).

**Parameters:**
```php
$data = [
    'database_name' => 'ventas_db',      // Required
    'table_name' => 'productos',         // Required
    'create_sql' => 'CREATE TABLE...',   // Required: full SQL
    'status' => 'success',               // Required: 'success' or 'error'
    'error' => 'Error message...'        // Optional: if status is 'error'
];
```

**Usage:**
```php
// On success
ImportLogger::logCreateTable($importLogId, [
    'database_name' => $databaseName,
    'table_name' => $tableName,
    'create_sql' => $createSQL,
    'status' => 'success'
]);

// On failure
ImportLogger::logCreateTable($importLogId, [
    'database_name' => $databaseName,
    'table_name' => $tableName,
    'create_sql' => $createSQL,
    'status' => 'error',
    'error' => $conn->error
]);
```

---

#### Method: `logSchema($importLogId, $schema)`

**Description:** Logs schema details (column by column).

**Parameters:**
```php
$schema = [
    [
        'name' => 'nombre',              // Required: sanitized name
        'type' => 'VARCHAR',             // Required: data type
        'length' => '100',               // Optional: length/values
        'default' => null,               // Optional: default value
        'nullable' => false,             // Optional: defaults to true
        'indexed' => true,               // Optional: defaults to false
        'comment' => 'Nombre del...',    // Optional: column comment
        'auto_detected' => true,         // Optional: defaults to true
        'original' => 'Nombre'           // Optional: original name
    ],
    // ... more columns
];
```

**Usage:**
```php
ImportLogger::logSchema($importLogId, $schema);
```

---

#### Method: `updateImportStatus($importLogId, $results)`

**Description:** Updates import with final statistics.

**Parameters:**
```php
$results = [
    'rows_processed' => 100,             // Required: total rows
    'rows_inserted' => 95,               // Required: successful inserts
    'rows_failed' => 5,                  // Required: failed inserts
    'errors' => [                        // Optional: error details
        ['row' => 10, 'reason' => '...'],
        ['row' => 25, 'reason' => '...']
    ]
];
```

**Usage:**
```php
ImportLogger::updateImportStatus($importLogId, [
    'rows_processed' => count($rows),
    'rows_inserted' => $insertedCount,
    'rows_failed' => $failedCount,
    'errors' => ErrorHandler::getErrors()
]);
```

**Note:** Automatically determines `operation_status`:
- `success` - All rows inserted (failed = 0)
- `partial` - Some rows failed (inserted > 0, failed > 0)
- `error` - All rows failed or other error

---

#### Method: `logError($importLogId, $error)`

**Description:** Logs an error for a specific import.

**Parameters:**
- `$importLogId` - UUID of import log
- `$error` - Error message string

**Usage:**
```php
ImportLogger::logError($importLogId, $e->getMessage());
```

---

#### Method: `getImportHistory($filters)`

**Description:** Retrieves import history with optional filters.

**Parameters:**
```php
$filters = [
    'database_name' => 'ventas_db',      // Optional
    'table_name' => 'productos',         // Optional
    'operation_status' => 'success',     // Optional: success|error|partial|pending
    'limit' => 50                        // Optional: defaults to 100
];
```

**Returns:** `array` - Array of import log records

**Usage:**
```php
// Get all imports
$allLogs = ImportLogger::getImportHistory();

// Get only successful imports for specific table
$logs = ImportLogger::getImportHistory([
    'table_name' => 'productos',
    'operation_status' => 'success',
    'limit' => 10
]);
```

---

#### Method: `getSchemaByImportId($importLogId)`

**Description:** Retrieves schema details for a specific import.

**Parameters:**
- `$importLogId` - UUID of import log

**Returns:** `array` - Array of schema column records (ordered by column_order)

**Usage:**
```php
$schema = ImportLogger::getSchemaByImportId($importLogId);

foreach ($schema as $col) {
    echo "{$col['column_name']} - {$col['data_type']}\n";
}
```

---

## 🔄 Integration Flow

### Step 1: File Upload (upload.php)

```php
session_start();

// After schema detection...
$logMetadata = [
    'file_name' => $file['name'],
    'file_type' => strtoupper($fileExt),
    'file_size' => $file['size'],
    'table_name' => $tableName,
    'csv_delimiter' => $csvMetadata['delimiter_name'] ?? null,
    'csv_encoding' => $csvMetadata['encoding'] ?? null
];

$importLogId = ImportLogger::startImport($logMetadata);
$_SESSION['import_log_id'] = $importLogId;
```

### Step 2: Table Creation (insert.php)

```php
session_start();

// Retrieve import log ID from session
$importLogId = $_SESSION['import_log_id'] ?? null;

// After CREATE TABLE executes...
ImportLogger::logCreateTable($importLogId, [
    'database_name' => $databaseName,
    'table_name' => $tableName,
    'create_sql' => $createSQL,
    'status' => 'success'
]);

// Log schema details
ImportLogger::logSchema($importLogId, $schema);
```

### Step 3: Final Results (insert.php)

```php
// After insertion completes...
ImportLogger::updateImportStatus($importLogId, [
    'rows_processed' => count($rows),
    'rows_inserted' => $insertedCount,
    'rows_failed' => $failedCount,
    'errors' => ErrorHandler::getErrors()
]);
```

---

## 📊 Sample Queries

### Get All Imports from Last 7 Days

```sql
SELECT 
    table_name,
    operation_status,
    rows_inserted,
    rows_failed,
    alta_db
FROM importer_system.import_logs
WHERE alta_db >= DATE_SUB(NOW(), INTERVAL 7 DAY)
ORDER BY alta_db DESC;
```

### Get Schema for Specific Table Import

```sql
SELECT 
    il.table_name,
    isl.column_name,
    isl.data_type,
    isl.length_values,
    isl.is_nullable,
    isl.column_comment
FROM importer_system.import_logs il
JOIN importer_system.import_schema_logs isl 
    ON il.import_log_id = isl.import_log_id
WHERE il.table_name = 'productos'
    AND il.operation_status = 'success'
ORDER BY il.alta_db DESC, isl.column_order ASC
LIMIT 1;
```

### Get Import Success Rate

```sql
SELECT 
    operation_status,
    COUNT(*) as count,
    ROUND(COUNT(*) * 100.0 / SUM(COUNT(*)) OVER(), 2) as percentage
FROM importer_system.import_logs
GROUP BY operation_status;
```

### Get Failed Imports with Errors

```sql
SELECT 
    table_name,
    create_table_error,
    error_details,
    alta_db
FROM importer_system.import_logs
WHERE operation_status = 'error'
ORDER BY alta_db DESC;
```

---

## 🧪 Testing

Run the test script to verify the logger is working:

```bash
/lamp/php/bin/php /lamp/www/importer/test_import_logger.php
```

**Expected Output:**
- ✅ Log tables created/verified
- ✅ All tables exist in database
- ✅ Version matches current version
- ✅ Import log created
- ✅ CREATE TABLE logged
- ✅ Schema logged
- ✅ Import status updated
- ✅ Import history retrieved
- ✅ Schema retrieved by import ID
- ✅ Data integrity verified

---

## 🚀 Step 2: Using Logs for Analysis

The logged data is now ready for Step 2 analysis. Use the query methods:

```php
// Get recent imports
$recentImports = ImportLogger::getImportHistory(['limit' => 50]);

// Analyze each import
foreach ($recentImports as $import) {
    $importLogId = $import['import_log_id'];
    $schema = ImportLogger::getSchemaByImportId($importLogId);
    
    // Do analysis on schema...
    // Compare with previous imports...
    // Detect schema changes...
    // Generate reports...
}
```

---

## 🔧 Version Management

The system supports zero-downtime migrations:

1. **Version Check:** On first use, checks current version
2. **Auto-Create:** If tables don't exist, creates v1 tables
3. **Auto-Migrate:** If version < CURRENT_VERSION, runs migration
4. **Preserve Data:** All migrations use ALTER TABLE (non-destructive)

### Future Version Migrations

Add to `migrateLogTables()` method:

```php
private static function migrateLogTables($conn, $fromVersion) {
    // Version 1 → 2
    if ($fromVersion < 2) {
        $conn->query("ALTER TABLE import_logs ADD COLUMN new_field VARCHAR(100)");
    }
    
    // Version 2 → 3
    if ($fromVersion < 3) {
        $conn->query("ALTER TABLE import_schema_logs ADD INDEX idx_new_field (new_field)");
    }
}
```

---

## 📋 What Gets Logged

### ✅ File Metadata
- Original filename
- File type (XLSX/CSV)
- File size in bytes
- Upload timestamp

### ✅ CSV-Specific Metadata
- Detected delimiter (comma, semicolon, tab, pipe)
- Detected encoding (UTF-8, ISO-8859-1, Windows-1252)
- Number of rows normalized

### ✅ Database Operations
- Target database name
- Target table name
- Full CREATE TABLE SQL statement
- CREATE TABLE success/error status
- CREATE TABLE error message (if failed)

### ✅ Schema Details (Per Column)
- Sanitized column name
- Original column name (before sanitization)
- Column order/position
- Data type (VARCHAR, INT, ENUM, etc.)
- Length/Values (e.g., '255', '(10,2)', "('A','B')")
- Default value
- NULL constraint
- Index status
- Column comment
- Whether auto-detected or manually specified

### ✅ Insertion Statistics
- Total rows processed
- Successfully inserted rows
- Failed row count
- Detailed error information (JSON)

### ✅ Audit Fields
- Import timestamp (alta_db)
- User who performed import (alta_por)
- Last modification timestamp (ultimo_cambio)
- Last modifier (ultimo_cambio_por)

---

## 🎨 Cyberpunk Philosophy

The Import Logger follows the **Cyberpunk Data OS** principles:

- **Spanish Canonical Schema** - Uses `alta_db`, `alta_por`, etc.
- **Dynamic Primary Keys** - Table-specific UUIDs
- **Zero Friction** - Automatic table creation and migration
- **Full Sovereignty** - Complete audit trail for all operations
- **Myth-Tech Architecture** - Self-documenting, future-proof

---

## 📄 License

Private & proprietary — Phoenix / Filemón Prime.
