<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>🌊 Cyberpunk Data Importer</title>

<style>
    * { box-sizing: border-box; }

    body {
        background: radial-gradient(circle at top, #020d18 0%, #000000 70%);
        font-family: 'Consolas', 'Monaco', monospace;
        color: #e0f7ff;
        margin: 0;
        padding: 0;
        min-height: 100vh;
    }

    .panel {
        background: rgba(0,0,0,0.35);
        border: 1px solid #00eaff55;
        box-shadow: 0 0 20px #00eaff44;
        margin: 20px auto;
        padding: 25px;
        border-radius: 12px;
        width: 90%;
        max-width: 1200px;
        animation: fadein 0.6s ease;
    }

    .panel.cathedral {
        border: 2px solid #00eaff;
        box-shadow: 0 0 40px #00eaffaa;
        animation: cathedral-pulse 2s ease-in-out infinite;
    }

    @keyframes fadein {
        from { opacity:0; transform: translateY(20px); }
        to { opacity:1; transform: translateY(0); }
    }

    @keyframes cathedral-pulse {
        0%, 100% { box-shadow: 0 0 40px #00eaffaa; }
        50% { box-shadow: 0 0 60px #00eaffdd, 0 0 80px #00eaff88; }
    }

    h2 {
        text-shadow: 0 0 12px #00eaff;
        font-weight: bold;
        margin-bottom: 12px;
        margin-top: 0;
    }

    h3 {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
        margin-top: 20px;
    }

    /* Input Styling */
    input[type="text"], input[type="number"], input[type="file"], select {
        background: #000;
        color: #00eaff;
        border: 1px solid #00eaff55;
        padding: 8px 12px;
        border-radius: 4px;
        font-family: 'Consolas', monospace;
        transition: all 0.3s ease;
    }

    input[type="text"]:focus, input[type="number"]:focus, select:focus {
        outline: none;
        border-color: #00eaff;
        box-shadow: 0 0 12px #00eaff88;
    }

    input[type="file"] {
        padding: 6px;
    }

    input[type="checkbox"] {
        cursor: pointer;
        width: 18px;
        height: 18px;
        accent-color: #00eaff;
    }

    label {
        color: #00ffaa;
        font-weight: bold;
        margin-right: 10px;
    }

    /* Button Styling */
    button {
        background: #00eaff;
        color: #000;
        padding: 12px 28px;
        font-size: 16px;
        border-radius: 8px;
        border: none;
        cursor: pointer;
        font-weight: bold;
        box-shadow: 0 0 14px #00eaffcc;
        transition: all 0.2s ease;
        font-family: 'Consolas', monospace;
    }

    button:hover {
        transform: scale(1.05);
        box-shadow: 0 0 22px #00eaff;
    }

    button:active {
        transform: scale(0.98);
    }

    /* Table Styling */
    .table-wrap {
        overflow-x: auto;
        margin-top: 15px;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 12px;
    }

    th, td {
        border: 1px solid #00eaff33;
        padding: 10px 12px;
        text-align: left;
    }

    th {
        background: rgba(0,234,255,0.15);
        text-shadow: 0 0 6px #00eaff;
        font-weight: bold;
    }

    tr:hover td {
        background: rgba(0,234,255,0.08);
        transition: background 0.2s ease;
    }

    tbody tr {
        animation: row-pulse 0.3s ease;
    }

    @keyframes row-pulse {
        from { opacity: 0.5; }
        to { opacity: 1; }
    }

    /* SQL Preview */
    #sqlPreview {
        background: #000;
        color: #00ff88;
        border: 1px solid #00eaff77;
        padding: 15px;
        border-radius: 8px;
        font-family: 'Courier New', monospace;
        font-size: 13px;
        white-space: pre-wrap;
        overflow-x: auto;
        box-shadow: inset 0 0 12px #00eaff44;
        max-height: 400px;
        overflow-y: auto;
    }

    /* Status Messages */
    .status {
        margin-top: 15px;
        padding: 10px;
        border-radius: 6px;
        animation: fadein 0.4s ease;
    }

    .status.success {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
    }

    .status.error {
        color: #ff3366;
        text-shadow: 0 0 12px #ff3366;
        font-weight: bold;
    }

    /* Schema Table Specific */
    #schemaTable input[type="text"],
    #schemaTable input[type="number"],
    #schemaTable select {
        width: 100%;
        padding: 6px 8px;
        font-size: 13px;
    }

    .db-table-inputs {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
        margin-bottom: 20px;
    }

    .input-group {
        display: flex;
        flex-direction: column;
        gap: 8px;
    }

    /* Scrollbar Styling */
    ::-webkit-scrollbar {
        width: 10px;
        height: 10px;
    }

    ::-webkit-scrollbar-track {
        background: #000;
    }

    ::-webkit-scrollbar-thumb {
        background: #00eaff55;
        border-radius: 5px;
    }

    ::-webkit-scrollbar-thumb:hover {
        background: #00eaff88;
    }

    .back-link {
        display: inline-block;
        margin-bottom: 20px;
        color: #00eaff;
        text-decoration: none;
        font-weight: bold;
        transition: all 0.3s;
    }
    .back-link:hover {
        text-shadow: 0 0 8px #00eaff;
        transform: translateX(-5px);
    }

    /* Floating Back to Top Button */
    #backToTopBtn {
        position: fixed;
        bottom: 30px;
        right: 30px;
        width: 55px;
        height: 55px;
        background: rgba(0, 234, 255, 0.9);
        color: #000;
        border: 2px solid #00eaff;
        border-radius: 50%;
        font-size: 24px;
        font-weight: bold;
        cursor: pointer;
        box-shadow: 0 0 20px #00eaffcc, 0 0 40px #00eaff66;
        transition: all 0.3s ease;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transform: scale(0.8);
        display: flex;
        align-items: center;
        justify-content: center;
        font-family: 'Consolas', monospace;
        animation: none;
    }

    #backToTopBtn.show {
        opacity: 1;
        visibility: visible;
        transform: scale(1);
    }

    #backToTopBtn:hover {
        background: #00eaff;
        box-shadow: 0 0 30px #00eaff, 0 0 60px #00eaff88;
        transform: scale(1.1) translateY(-3px);
    }

    #backToTopBtn:active {
        transform: scale(0.95);
    }

    /* Cyberpunk pulse animation for the button */
    @keyframes cyber-pulse {
        0%, 100% {
            box-shadow: 0 0 20px #00eaffcc, 0 0 40px #00eaff66;
        }
        50% {
            box-shadow: 0 0 30px #00eaff, 0 0 50px #00eaff88, 0 0 70px #00eaff44;
        }
    }

    #backToTopBtn.show {
        animation: cyber-pulse 2s ease-in-out infinite;
    }

    /* Autocomplete Dropdown Styles */
    .cyber-autocomplete-dropdown::-webkit-scrollbar {
        width: 8px;
    }

    .cyber-autocomplete-dropdown::-webkit-scrollbar-track {
        background: rgba(0,0,0,0.3);
    }

    .cyber-autocomplete-dropdown::-webkit-scrollbar-thumb {
        background: #00ffaa;
        border-radius: 4px;
    }

    .cyber-autocomplete-item {
        padding: 10px 16px;
        cursor: pointer;
        display: flex;
        justify-content: space-between;
        align-items: center;
        transition: all 0.15s ease;
        border-left: 3px solid transparent;
    }

    .cyber-autocomplete-item:hover,
    .cyber-autocomplete-item.highlighted {
        background: linear-gradient(90deg, rgba(0,234,255,0.15) 0%, transparent 100%);
        border-left-color: #00ffaa;
    }

    .cyber-autocomplete-item.selected {
        background: linear-gradient(90deg, rgba(0,255,170,0.2) 0%, transparent 100%);
        border-left-color: #00ffaa;
    }

    .cyber-autocomplete-item .table-name {
        color: #fff;
        font-size: 14px;
        font-weight: 500;
    }

    .cyber-autocomplete-item .table-meta {
        font-size: 11px;
        color: #888;
        display: flex;
        gap: 12px;
    }

    .cyber-autocomplete-item .usage-count {
        background: rgba(0,234,255,0.2);
        color: #00ffaa;
        padding: 2px 8px;
        border-radius: 10px;
        font-size: 10px;
    }

    .cyber-autocomplete-item .last-used {
        color: #666;
        font-size: 10px;
    }

    /* Input focus glow */
    #destinationTableName:focus {
        outline: none;
        box-shadow: 0 0 15px rgba(0,255,170,0.4), 0 0 30px rgba(0,255,170,0.2) !important;
    }

    /* Dropdown animation */
    .cyber-autocomplete-dropdown {
        animation: dropdown-fade-in 0.15s ease-out;
    }

    @keyframes dropdown-fade-in {
        from {
            opacity: 0;
            transform: translateY(-10px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
</style>

<script src="lib/TableNameMemory.js"></script>

</head>
<body>

<!-- GATE 1: THE ARRIVAL -->
<div class="panel">
    <a href="index.php" class="back-link">← Return to Threshold</a>
    <h2>🌊 Gate 1: The Arrival — Upload Data File</h2>
    <p style="color: #00ffaa; font-size: 14px; margin-bottom: 15px;">
        Accepted formats: <strong>XLSX</strong> (Excel) or <strong>CSV</strong> (Comma-Separated Values)
        <br>CSV files will be auto-detected for encoding and delimiter type.
    </p>

    <div style="margin-bottom: 15px; position: relative;">
        <label for="destinationTableName" style="display: block; margin-bottom: 8px;">Destination Table Name:</label>
        
        <div class="cyber-input-wrapper" style="position: relative; width: 100%; max-width: 400px;">
            <input id="destinationTableName" type="text" 
                   placeholder="Start typing to see suggestions..."
                   autocomplete="off"
                   style="width: 100%; padding-right: 40px;" />
            
            <span class="input-arrow" onclick="toggleTableNameDropdown()"
                  style="position: absolute; right: 12px; top: 50%; transform: translateY(-50%);
                         cursor: pointer; color: #00ffaa; font-size: 12px;">
                ▼
            </span>
        </div>
        
        <div id="tableNameDropdown" class="cyber-autocomplete-dropdown" 
             style="display: none; position: absolute; z-index: 1000;
                    background: linear-gradient(180deg, rgba(10,20,30,0.98) 0%, rgba(5,15,25,0.98) 100%);
                    border: 1px solid #00ffaa; border-radius: 8px;
                    max-height: 300px; overflow-y: auto;
                    box-shadow: 0 0 30px rgba(0,255,170,0.3), 0 10px 40px rgba(0,0,0,0.5);
                    margin-top: 4px; width: 100%; max-width: 400px;">
            
            <div class="dropdown-header" style="padding: 8px 12px; 
                                               background: rgba(0,234,255,0.1);
                                               border-bottom: 1px solid rgba(0,255,170,0.2);
                                               font-size: 11px; color: #00ffaa;">
                <span id="dropdownStatus">Type to search...</span>
            </div>
            
            <div class="dropdown-section" id="recentSection" style="display: none;">
                <div class="section-title" style="padding: 6px 12px; font-size: 10px;
                                                  text-transform: uppercase; color: #ff6600;
                                                  letter-spacing: 1px;">Recent & Frequent</div>
            </div>
            
            <div id="suggestionsList" style="padding: 4px 0;"></div>
            
            <div id="noResults" class="dropdown-no-results" style="display: none;
                                                                   padding: 20px; text-align: center;
                                                                   color: #666;">
                No matching tables found
            </div>
        </div>
        
        <p style="color: #00ffaa; font-size: 12px; margin-top: 5px; margin-bottom: 0;">
            Suggestions based on your import history. Press ↓ to open dropdown.
        </p>
    </div>

    <input id="file" type="file" accept=".xlsx,.csv" />
    <button onclick="uploadXLSX()" style="margin-left: 15px;">Upload & Parse</button>
    <div id="status" class="status"></div>
    
    <!-- Animated Progress Display -->
    <div id="progressContainer" style="display: none; margin: 20px 0; max-width: 600px;">
        
        <!-- Phase Indicator -->
        <div id="progressPhase" style="
            font-size: 16px; 
            color: #00ffaa; 
            margin-bottom: 12px;
            text-transform: uppercase;
            letter-spacing: 2px;
            font-weight: bold;
            display: flex;
            align-items: center;
            gap: 10px;
        ">
            <span class="phase-icon" style="font-size: 20px;">📊</span> 
            <span class="phase-text">Processing...</span>
        </div>
        
        <!-- Main Progress Bar -->
        <div style="
            background: rgba(0,0,0,0.6);
            border: 2px solid #00ffaa;
            border-radius: 10px;
            height: 28px;
            overflow: hidden;
            position: relative;
            box-shadow: 0 0 20px rgba(0,255,170,0.3), inset 0 0 20px rgba(0,0,0,0.5);
        ">
            <!-- Animated Fill -->
            <div id="progressBar" style="
                height: 100%;
                width: 0%;
                background: linear-gradient(90deg, 
                    rgba(0,255,170,0.9) 0%, 
                    rgba(0,234,255,0.9) 50%, 
                    rgba(255,0,255,0.9) 100%);
                transition: width 0.3s ease-out;
                position: relative;
                overflow: hidden;
                border-radius: 8px;
            ">
                <!-- Shimmer Effect -->
                <div style="
                    position: absolute;
                    top: 0; left: 0; right: 0; bottom: 0;
                    background: linear-gradient(90deg, 
                        transparent 0%, 
                        rgba(255,255,255,0.25) 50%, 
                        transparent 100%);
                    animation: shimmer 1.5s infinite;
                "></div>
            </div>
            
            <!-- Percentage Overlay -->
            <div id="progressPercent" style="
                position: absolute;
                top: 50%; left: 50%;
                transform: translate(-50%, -50%);
                color: #fff;
                font-weight: bold;
                font-size: 13px;
                text-shadow: 0 0 10px rgba(0,0,0,0.8);
                font-family: 'Consolas', monospace;
            ">0%</div>
        </div>
        
        <!-- Progress Details -->
        <div id="progressDetails" style="
            margin-top: 12px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            color: #00ffaa;
            font-size: 13px;
            font-family: 'Consolas', monospace;
        ">
            <span><span id="progressCurrent" style="color: #fff; font-size: 16px;">0</span></span>
            <span style="color: #666;">of</span>
            <span><span id="progressTotal" style="color: #fff; font-size: 16px;">0</span></span>
            <span id="progressMessage" style="color: #00eaff; margin-left: auto;">Initializing...</span>
        </div>
        
        <!-- ETA Display -->
        <div id="progressETA" style="
            margin-top: 8px;
            color: #ff6600;
            font-size: 12px;
            font-family: monospace;
            text-align: right;
        "></div>
        
        <!-- Animated Dots -->
        <div id="progressDots" style="margin-top: 12px; text-align: center;">
            <span class="progress-dot" style="
                display: inline-block;
                width: 10px; height: 10px;
                background: #00ffaa;
                border-radius: 50%;
                margin: 0 4px;
                box-shadow: 0 0 10px #00ffaa;
                animation: dot-pulse 1s infinite;
            "></span>
            <span class="progress-dot" style="
                display: inline-block;
                width: 10px; height: 10px;
                background: #00ffaa;
                border-radius: 50%;
                margin: 0 4px;
                box-shadow: 0 0 10px #00ffaa;
                animation: dot-pulse 1s infinite 0.2s;
            "></span>
            <span class="progress-dot" style="
                display: inline-block;
                width: 10px; height: 10px;
                background: #00ffaa;
                border-radius: 50%;
                margin: 0 4px;
                box-shadow: 0 0 10px #00ffaa;
                animation: dot-pulse 1s infinite 0.4s;
            "></span>
        </div>
    </div>
    
    <style>
    @keyframes shimmer {
        0% { transform: translateX(-100%); }
        100% { transform: translateX(100%); }
    }
    @keyframes dot-pulse {
        0%, 100% { opacity: 0.3; transform: scale(0.8); box-shadow: 0 0 5px #00ffaa; }
        50% { opacity: 1; transform: scale(1.2); box-shadow: 0 0 20px #00ffaa, 0 0 30px rgba(0,255,170,0.5); }
    }
    @keyframes progress-glow {
        0%, 100% { box-shadow: 0 0 20px rgba(0,255,170,0.5); }
        50% { box-shadow: 0 0 40px rgba(0,255,170,0.8), 0 0 60px rgba(0,255,170,0.4); }
    }
    .progress-bar-glow {
        animation: progress-glow 2s ease-in-out infinite;
    }
    </style>
</div>

<!-- PREVIEW PANEL -->
<div id="preview" class="panel" style="display:none;"></div>

<!-- GATE 3: THE SCHEMA GATE -->
<div id="schemaPanel" class="panel" style="display:none;">
    <h2>🌌 Gate 3: The Imagination — Schema Builder</h2>
    <p style="color: #00ffaa; margin-bottom: 20px;">Shape the structure before birth. Edit column types, lengths, and constraints.</p>

    <!-- Schema Memory Indicator -->
    <div id="schemaMemoryIndicator" style="display:none; background: rgba(0,234,255,0.15); border: 1px solid #00eaff; border-radius: 8px; padding: 12px 16px; margin-bottom: 20px;">
        <strong style="color: #00eaff;">🧠 Schema Restored from Memory</strong><br>
        <span id="schemaMemoryMessage" style="color: #00ffaa; font-size: 13px;"></span><br>
        <div style="margin-top: 12px;">
            <label style="color: #00ffaa; font-size: 13px; cursor: pointer;">
                <input type="checkbox" id="skipTableRecreation" style="margin-right: 8px;">
                Skip table recreation — insert records only (table already exists)
            </label>
        </div>
        <button onclick="resetToAutoDetectedSchema()" style="margin-top: 10px; background: #ff6600; padding: 8px 16px; font-size: 13px;">Reset to Auto-Detected</button>
    </div>

    <div class="db-table-inputs">
        <div class="input-group">
            <label>Database Name:</label>
            <select id="databaseNameSelect">
                <option value="">Select existing database...</option>
                <!-- Existing databases will be populated -->
            </select>
            <input id="databaseNameInput" type="text" placeholder="Or enter new database name" style="display:none;" />
            <button id="toggleInput" onclick="toggleDatabaseInput()" style="margin-left: 10px;">Toggle</button>
        </div>
        <div class="input-group">
            <label>Table Name:</label>
            <input id="tableName" type="text" placeholder="Table name" readonly
                   style="background: rgba(0,234,255,0.08); cursor: not-allowed;" />
            <p style="color: #00ffaa; font-size: 12px; margin-top: 5px;">
                (Set in Gate 1 - cannot be changed here)
            </p>
        </div>
    </div>

    <h3>Columns (Standard fields {table}_id, alta_db, alta_por, ultimo_cambio, ultimo_cambio_por will be added automatically)</h3>

    <div class="table-wrap">
        <table id="schemaTable">
            <thead>
                <tr>
                <th style="width: 40px;">Skip</th>
                <th>Column Name</th>
                <th>Data Type</th>
                <th>Length</th>
                <th>Nullable</th>
                <th>Indexed</th>
                <th>Comment</th>
                </tr>
            </thead>
            <tbody id="schemaTableBody">
                <!-- Populated dynamically -->
            </tbody>
        </table>
    </div>

    <h3>SQL Preview</h3>
    <pre id="sqlPreview">-- Waiting for schema...</pre>

    <button onclick="createAndInsert()" style="margin-top: 20px;">⚡ Create Table & Insert Data</button>
    <div id="insertStatus" class="status"></div>
</div>

<!-- GATE 4: THE BLESSING - RESULTS -->
<div id="resultsPanel" class="panel" style="display:none;">
    <h2>✨ Gate 4: The Cathedral Moment — Success</h2>
    <div id="insertMessage" class="status success"></div>

    <h3 style="margin-top: 32px;">Inserted Rows (Preview)</h3>
    <div id="results"></div>
</div>

<script>
// Global state
window.parsedData = null;
window.currentSchema = null;
window.autoDetectedSchema = null; // Store auto-detected schema for reset functionality

/**
 * Gate 1: Upload and Parse XLSX
 */
function uploadXLSX() {
    const fileInput = document.getElementById("file");
    const tableNameInput = document.getElementById("destinationTableName");
    const statusDiv = document.getElementById("status");

    if (!fileInput.files[0]) {
        showError(statusDiv, "Please select a file.");
        return;
    }

    const tableName = tableNameInput.value.trim();
    if (!tableName) {
        showError(statusDiv, "Please enter a destination table name.");
        return;
    }

    const formData = new FormData();
    formData.append("xlsx", fileInput.files[0]);
    formData.append("tableName", tableName);

    // Hide previous results
    document.getElementById('preview').style.display = 'none';
    document.getElementById('schemaPanel').style.display = 'none';
    document.getElementById('resultsPanel').style.display = 'none';
    
    // Show progress animation
    ProgressMonitor.start(null, {
        total: 100,
        phase: 'parsing',
        message: 'Uploading file...'
    });

    statusDiv.className = "status";
    statusDiv.innerHTML = "🌊 Uploading and analyzing file...";

    fetch("upload.php", {
        method: "POST",
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.status !== "ok") {
            ProgressMonitor.hide();
            showError(statusDiv, data.message);
            return;
        }

        // Start progress monitoring if we have a progress key
        if (data.progress_key) {
            ProgressMonitor.start(data.progress_key, {
                total: data.stats?.total_rows || 100,
                phase: 'parsing',
                message: 'Analysis complete!',
                onComplete: function(progress) {
                    handleUploadSuccess(data, statusDiv);
                }
            });
        } else {
            handleUploadSuccess(data, statusDiv);
        }
    })
    .catch(err => {
        ProgressMonitor.hide();
        showError(statusDiv, "Network error: " + err.message);
    });
}

function handleUploadSuccess(data, statusDiv) {
    // Record table name for autocomplete learning
    if (window.recordTableNameUsage) {
        recordTableNameUsage(data.tableName);
    }

    // Store parsed data
    window.parsedData = data;

    // Show success
    statusDiv.className = "status success";
    statusDiv.innerHTML = `✓ ${data.message}`;

    // Hide progress
    ProgressMonitor.hide();

    // Show preview
    showPreview(data.rows);

    // Show schema builder with memory indicator
    showSchemaBuilder(data.tableName, data.schema, data.schema_source, data.schema_import_date);

    // Load daemon memory
    loadDaemonMemory();
}

/**
 * Show data preview
 */
function showPreview(rows) {
    const panel = document.getElementById("preview");

    if (!rows || rows.length === 0) {
        panel.style.display = "none";
        return;
    }

    let html = `<h2>📊 Parsed Data Preview (First ${Math.min(rows.length, 10)} rows)</h2>`;
    
    // Show CSV metadata if available
    if (window.parsedData && window.parsedData.stats) {
        const stats = window.parsedData.stats;
        if (stats.file_type === 'CSV') {
            html += `<p style="color: #00ffaa; font-size: 13px; margin-bottom: 10px;">`;
            html += `📋 CSV File • Delimiter: <strong>${escapeHtml(stats.delimiter)}</strong> • `;
            html += `Total Rows: <strong>${stats.total_rows}</strong>`;
            if (stats.inconsistent_rows && stats.inconsistent_rows > 0) {
                html += ` • <span style="color: #ffaa00;">⚠ ${stats.inconsistent_rows} rows normalized</span>`;
            }
            html += `</p>`;
        }
    }
    
    html += `<div class='table-wrap'><table><tr>`;

    const cols = Object.keys(rows[0]);
    cols.forEach(c => html += `<th>${escapeHtml(c)}</th>`);
    html += `</tr>`;

    const previewRows = rows.slice(0, 10);
    previewRows.forEach(r => {
        html += "<tr>";
        cols.forEach(c => html += `<td>${escapeHtml(r[c] || '')}</td>`);
        html += "</tr>";
    });

    html += "</table></div>";

    panel.innerHTML = html;
    panel.style.display = "block";
}

/**
 * Gate 3: Show Schema Builder
 */
function showSchemaBuilder(tableName, schema, schemaSource, schemaImportDate) {
    window.currentSchema = schema;

    // Store auto-detected schema if not already stored (for reset functionality)
    if (window.autoDetectedSchema === null) {
        // Deep copy the schema
        window.autoDetectedSchema = JSON.parse(JSON.stringify(schema));
    }

    // Show memory indicator if schema came from memory
    if (schemaSource === 'memory' && schemaImportDate) {
        const indicator = document.getElementById('schemaMemoryIndicator');
        const message = document.getElementById('schemaMemoryMessage');

        // Get match score and column counts
        const matchScore = window.parsedData.schema_match_score || 100;
        const missingColumns = schema.filter(col => col.missing_in_file === true);
        const newColumns = schema.filter(col => col.is_new_column === true);

        // Determine score color
        let scoreColor = matchScore >= 90 ? '#00ffaa' : matchScore >= 70 ? '#ffaa00' : '#ff6600';

        // Build message with match score and column info
        let messageHTML = `Schema for table "${tableName}" loaded from previous import on ${schemaImportDate}.<br>`;
        messageHTML += `<strong style="color: ${scoreColor};">Match Score: ${matchScore}%</strong>`;

        if (matchScore < 100) {
            messageHTML += ` <span style="color: #888; font-size: 12px;">(fuzzy match - 70% threshold)</span>`;
        }

        if (newColumns.length > 0) {
            messageHTML += `<br><strong style="color: #00ffaa;">✨ ${newColumns.length} new column(s) detected (not in saved schema):</strong>`;
            messageHTML += `<br><span style="font-size: 12px; color: #00ffaa;">` + newColumns.map(c => c.name).join(', ') + `</span>`;
        }

        if (missingColumns.length > 0) {
            messageHTML += `<br><strong style="color: #ffaa00;">⚠ ${missingColumns.length} column(s) missing in file (will use defaults):</strong>`;
            messageHTML += `<br><span style="font-size: 12px; color: #ffaa00;">` + missingColumns.map(c => c.name).join(', ') + `</span>`;
        }

        messageHTML += `<br><span style="color: #00ffaa;">You can review and modify as needed.</span>`;

        message.innerHTML = messageHTML;
        indicator.style.display = 'block';
    } else {
        document.getElementById('schemaMemoryIndicator').style.display = 'none';
    }

    // Set table name
    document.getElementById("tableName").value = tableName;

    // Intelligent database name recall: check table-specific mapping first, then fall back to last used
    let suggestedDbName = 'importer_db';
    try {
        const tableMappings = JSON.parse(localStorage.getItem('tableToDatabaseMap') || '{}');
        if (tableMappings[tableName]) {
            // This table was previously imported to a specific database
            suggestedDbName = tableMappings[tableName];
        } else {
            // Fall back to last used database
            suggestedDbName = localStorage.getItem('lastDatabaseName') || 'importer_db';
        }
    } catch (e) {
        suggestedDbName = localStorage.getItem('lastDatabaseName') || 'importer_db';
    }

    // Populate dropdown with available databases
    fetchDatabaseOptions();

    // Set the dropdown value or switch to text input if needed
    const selectElement = document.getElementById("databaseNameSelect");
    if (selectElement.options.length > 1 && selectElement.options.namedItem(suggestedDbName)) {
        selectElement.value = suggestedDbName;
    } else {
        // Switch to text input mode for new database name
        selectElement.style.display = "none";
        document.getElementById("databaseNameInput").value = suggestedDbName;
        document.getElementById("databaseNameInput").style.display = "inline-block";
        document.getElementById("toggleInput").textContent = "Switch to dropdown";
    }

    // Populate schema table
    const tbody = document.getElementById("schemaTableBody");
    tbody.innerHTML = "";

    schema.forEach((col, idx) => {
        const row = tbody.insertRow();

        // Highlight missing columns (orange) and new columns (green)
        const isMissing = col.missing_in_file === true;
        const isNew = col.is_new_column === true;

        let rowStyle = '';
        let tag = '';

        if (isMissing) {
            rowStyle = 'background: rgba(255,170,0,0.1); border-left: 3px solid #ffaa00;';
            tag = '<br><small style="color: #ffaa00;">⚠ Missing in file</small>';
        } else if (isNew) {
            rowStyle = 'background: rgba(0,255,170,0.1); border-left: 3px solid #00ffaa;';
            tag = '<br><small style="color: #00ffaa;">✨ New column</small>';
        }

        row.innerHTML = `
            <td style="${rowStyle}">
                <input type="checkbox" title="Remove this column from the import" onchange="updateSchema(${idx}, 'remove', this.checked)" />
                ${tag}
            </td>
            <td style="${rowStyle}">
                <input type="text" value="${escapeHtml(col.name)}"
                    onchange="updateSchema(${idx}, 'name', this.value)" />
            </td>
            <td style="${rowStyle}">
                <select onchange="updateSchema(${idx}, 'type', this.value)">
                    <option ${col.type==='VARCHAR'?'selected':''}>VARCHAR</option>
                    <option ${col.type==='TEXT'?'selected':''}>TEXT</option>
                    <option ${col.type==='INT'?'selected':''}>INT</option>
                    <option ${col.type==='DECIMAL'?'selected':''}>DECIMAL</option>
                    <option ${col.type==='DATE'?'selected':''}>DATE</option>
                    <option ${col.type==='DATETIME'?'selected':''}>DATETIME</option>
                    <option ${col.type==='BOOLEAN'?'selected':''}>BOOLEAN</option>
                    <option ${col.type==='ENUM'?'selected':''}>ENUM</option>
                </select>
            </td>
            <td style="${rowStyle}"><input type="text" value="${escapeHtml(col.length || '')}"
                onchange="updateSchema(${idx}, 'length', this.value)"
                ${['TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'INT'].includes(col.type) ? 'disabled' : ''}
                placeholder="precision,scale" /></td>
            <td style="text-align: center; ${rowStyle}"><input type="checkbox" ${col.nullable?'checked':''}
                onchange="updateSchema(${idx}, 'nullable', this.checked)"/></td>
            <td style="text-align: center; ${rowStyle}"><input type="checkbox" ${col.indexed?'checked':''}
                onchange="updateSchema(${idx}, 'indexed', this.checked)"/></td>
            <td style="${rowStyle}"><input type="text" value="${escapeHtml(col.comment || '')}"
                onchange="updateSchema(${idx}, 'comment', this.value)" placeholder="Description"/></td>
        `;
    });

    // Update SQL preview
    updateSQLPreview();

    // Show panel
    document.getElementById("schemaPanel").style.display = "block";
}

/**
 * Update schema when user edits
 */
function updateSchema(idx, field, value) {
        if (field === 'remove') {
            // Mark column as skipped (removed from import)
            window.currentSchema[idx].remove = value;
        } else if (field === 'nullable' || field === 'indexed') {
            window.currentSchema[idx][field] = value;
        } else if (field === 'length') {
            // Special handling for different types
            const columnType = window.currentSchema[idx].type;

            if (columnType === 'DECIMAL') {
                // DECIMAL: Parse comma-separated precision,scale (e.g., "10,2")
                const parts = value.toString().split(',');
                if (parts.length === 2) {
                    const precision = parseInt(parts[0], 10);
                    const scale = parseInt(parts[1], 10);
                    if (!isNaN(precision) && !isNaN(scale)) {
                        window.currentSchema[idx].precision = precision;
                        window.currentSchema[idx].scale = scale;
                        // Store the raw string for display purposes
                        window.currentSchema[idx].length = value;
                    } else {
                        // Fall back to numeric parsing if comma parsing fails
                        const cleanedValue = value.toString().replace(/,/g, '.').replace(/[^0-9.]/g, '');
                        const parsedValue = cleanedValue ? parseFloat(cleanedValue) : null;
                        window.currentSchema[idx][field] = isNaN(parsedValue) ? null : parsedValue;
                    }
                } else {
                    // Fall back to numeric parsing for invalid DECIMAL format
                    const cleanedValue = value.toString().replace(/,/g, '.').replace(/[^0-9.]/g, '');
                    const parsedValue = cleanedValue ? parseFloat(cleanedValue) : null;
                    window.currentSchema[idx][field] = isNaN(parsedValue) ? null : parsedValue;
                }
            } else if (columnType === 'ENUM') {
                // ENUM: Store raw string value (e.g., "('Si','No')")
                // Do NOT parse as number - preserve the entire string including parentheses and quotes
                window.currentSchema[idx].length = value || null;
            } else if (columnType === 'VARCHAR') {
                // VARCHAR: Store as string (e.g., "100")
                // Keep as string for consistency, even though it's numeric
                window.currentSchema[idx].length = value || null;
            } else {
                // Other types: Parse as numeric (for future numeric types)
                let stringValue = '';
                if (typeof value === 'string') {
                    stringValue = value;
                } else if (typeof value === 'number') {
                    stringValue = String(value);
                } else {
                    stringValue = '';
                }
                const cleanedValue = stringValue.replace(/,/g, '.').replace(/[^0-9.]/g, '');
                const parsedValue = cleanedValue ? parseFloat(cleanedValue) : null;
                window.currentSchema[idx][field] = isNaN(parsedValue) ? null : parsedValue;
            }
        } else {
            window.currentSchema[idx][field] = value;
        }

    // Handle length input for different types
    if (field === 'type') {
        const row = document.getElementById("schemaTableBody").rows[idx];
        const lengthInput = row.cells[2].querySelector('input');
        const noLengthTypes = ['TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'INT'];

        if (noLengthTypes.includes(value)) {
            // Type doesn't use length - clear both UI and data
            lengthInput.disabled = true;
            lengthInput.value = '';  // Clear the visible value
            window.currentSchema[idx].length = null;  // Clear from schema data
        } else {
            // Type uses length - enable the input
            lengthInput.disabled = false;
        }
    }

    updateSQLPreview();
}

/**
 * Singularize table name (mirrors PHP SchemaConventions logic)
 * Converts plural table names to singular for primary key naming
 */
function singularizeTableName(tableName) {
    // Explicit whitelist of known plural forms (mirrors PHP SchemaConventions::IRREGULAR_PLURALS)
    const irregulars = {
        'propietarios': 'propietario',
        'departamentos': 'departamento',
        'alumnos_becados': 'alumno_becado',
        'eleyeme_cfdi_emitidos': 'eleyeme_cfdi_emitido',
        'productos': 'producto',
        'servicios': 'servicio',
        'empleados': 'empleado',
        'clientes': 'cliente',
        'proveedores': 'proveedor',
        'contratos': 'contrato',
        'facturas': 'factura',
        'pagos': 'pago',
        'usuarios': 'usuario',
        'roles': 'rol',
        'permisos': 'permiso',
        'catalogos': 'catalogo',
        'categorias': 'categoria',
        'sucursales': 'sucursal',
        'almacenes': 'almacen',
        'inventarios': 'inventario',
        'movimientos': 'movimiento',
        'documentos': 'documento',
        'archivos': 'archivo',
        'reportes': 'reporte',
        'configuraciones': 'configuracion',
        'bitacoras': 'bitacora',
        'auditorias': 'auditoria',
        'notificaciones': 'notificacion',
        'mensajes': 'mensaje',
        'tareas': 'tarea',
        'proyectos': 'proyecto',
        'eventos': 'evento',
        'registros': 'registro'
    };

    // Check explicit whitelist first
    if (irregulars[tableName]) {
        return irregulars[tableName];
    }

    // Fallback rule: strip trailing 's'
    if (tableName.endsWith('s')) {
        return tableName.slice(0, -1);
    }

    // Return as-is if already singular or unknown pattern
    return tableName;
}

/**
 * Generate SQL preview
 */
function updateSQLPreview() {
    const dbNameElement = document.getElementById("databaseNameSelect");
    const dbNameInput = document.getElementById("databaseNameInput");
    
    let dbName;
    if (dbNameElement.style.display !== "none") {
        dbName = dbNameElement.value;
    } else {
        dbName = dbNameInput.value;
    }
    const tableName = document.getElementById("tableName").value;

    // Get singularized table name for primary key
    const singularForm = singularizeTableName(tableName);
    const pkFieldName = `${singularForm}_id`;

    let sql = `-- Database: ${dbName}\n`;
    sql += `DROP TABLE IF EXISTS \`${tableName}\`;\n\n`;
    sql += `CREATE TABLE \`${tableName}\` (\n`;

    // Dynamic primary key field (table-specific, e.g., producto_id for productos table)
    sql += `  \`${pkFieldName}\` VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',\n`;

    // User columns (filter out removed columns)
    window.currentSchema.forEach(col => {
        // Skip this column if marked for removal
        if (col.remove === true) {
            return;
        }

        let colType = col.type;
        if (colType === 'VARCHAR' && col.length) {
            colType += `(${col.length})`;
        } else if (colType === 'ENUM' && col.length) {
            colType += col.length;  // Already formatted as ('value1','value2')
        } else if (colType === 'DECIMAL') {
            if (col.precision !== undefined && col.scale !== undefined) {
                colType += `(${col.precision},${col.scale})`;
            } else if (col.length) {
                // Legacy compatibility - use length as comma-separated string
                const parts = col.length.toString().split(',');
                if (parts.length === 2) {
                    colType += `(${parts.join(',')})`;
                } else {
                    colType += `(${col.length})`;
                }
            }
        } else if (colType === 'DECIMAL' && col.length) {
            colType += `(${col.length})`;
        }

        sql += `  \`${col.name}\` ${colType}`;
        sql += col.nullable ? ' NULL' : ' NOT NULL';
        if (col.comment) {
            // Escape single quotes in comment
            const safeComment = col.comment.replace(/'/g, "''");
            sql += ` COMMENT '${safeComment}'`;
        }
        sql += `,\n`;
    });

    // Standard audit fields (Spanish canonical schema)
    sql += `  \`alta_db\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'Timestamp del momento de inserción del registro',\n`;
    sql += `  \`alta_por\` VARCHAR(32) DEFAULT 'system' COMMENT 'Usuario que insertó el registro',\n`;
    sql += `  \`ultimo_cambio\` TIMESTAMP NULL DEFAULT NULL COMMENT 'Timestamp del último cambio al registro',\n`;
    sql += `  \`ultimo_cambio_por\` VARCHAR(32) NULL DEFAULT NULL COMMENT 'Usuario que hizo el último cambio',\n`;

    // Indexes (skip removed columns)
    window.currentSchema.forEach(col => {
        if (col.indexed && col.remove !== true) {
            sql += `  INDEX \`idx_${col.name}\` (\`${col.name}\`),\n`;
        }
    });

    // Remove trailing comma
    sql = sql.replace(/,\n$/, '\n');

    sql += `) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;`;

    document.getElementById("sqlPreview").textContent = sql;
}

/**
 * Gate 4: Create Table and Insert Data
 */
function createAndInsert() {
    const dbNameElement = document.getElementById("databaseNameSelect");
    const dbNameInput = document.getElementById("databaseNameInput");
    
    let dbName;
    if (dbNameElement.style.display !== "none") {
        dbName = dbNameElement.value.trim();
    } else {
        dbName = dbNameInput.value.trim();
    }
    const tableName = document.getElementById("tableName").value.trim();
    const statusDiv = document.getElementById("insertStatus");

    if (!dbName || !tableName) {
        showError(statusDiv, "Database name and table name are required");
        return;
    }

    if (!window.parsedData || !window.currentSchema) {
        showError(statusDiv, "No data to insert. Please upload a file first.");
        return;
    }

    // Check if user wants to skip table recreation
    const skipTableRecreationCheckbox = document.getElementById("skipTableRecreation");
    const skipTableRecreation = skipTableRecreationCheckbox ? skipTableRecreationCheckbox.checked : false;

    // Hide previous results
    document.getElementById('resultsPanel').style.display = 'none';
    
    // Show progress animation
    const totalRows = window.parsedData?.rows?.length || 0;
    ProgressMonitor.start(null, {
        total: totalRows,
        phase: 'inserting',
        message: skipTableRecreation ? 'Inserting data...' : 'Creating table...'
    });

    statusDiv.className = "status";
    statusDiv.innerHTML = skipTableRecreation
        ? "⚡ Inserting data into existing table..."
        : "⚡ Creating table and inserting data...";

    // Save to daemon memory
    saveDaemonMemory(dbName, tableName);

    fetch("insert.php", {
        method: "POST",
        headers: {"Content-Type":"application/json"},
        body: JSON.stringify({
            databaseName: dbName,
            tableName: tableName,
            schema: window.currentSchema,
            rows: window.parsedData.rows,
            skipTableRecreation: skipTableRecreation,
            progress_key: ProgressMonitor.currentKey || ''
        })
    })
    .then(r => r.json())
    .then(data => {
        if (data.status !== "ok") {
            ProgressMonitor.hide();
            showError(statusDiv, data.message);
            return;
        }

        // Start progress monitoring for insertion
        if (data.progress_key) {
            ProgressMonitor.start(data.progress_key, {
                total: totalRows,
                phase: 'inserting',
                message: 'Inserting rows...',
                onComplete: function(progress) {
                    handleInsertSuccess(data, statusDiv);
                }
            });
        } else {
            handleInsertSuccess(data, statusDiv);
        }
    })
    .catch(err => {
        ProgressMonitor.hide();
        showError(statusDiv, "Network error: " + err.message);
    });
}

function handleInsertSuccess(data, statusDiv) {
    // Show success
    statusDiv.className = "status success";
    statusDiv.innerHTML = `✓ ${data.message}`;

    // Hide progress
    ProgressMonitor.hide();

    // Show cathedral moment
    showCathedralMoment(data);
}

/**
 * Show cathedral moment - the final success display
 */
function showCathedralMoment(data) {
    const panel = document.getElementById("resultsPanel");
    const messageDiv = document.getElementById("insertMessage");
    const resultsDiv = document.getElementById("results");

    // Success message
    let correctedInfo = '';
    if (data.corrected && data.corrected > 0) {
        correctedInfo = `<br><span style="color: #ffaa00;">⚙ Auto-corrected: ${data.corrected} row${data.corrected > 1 ? 's' : ''}</span>`;
    }

    messageDiv.innerHTML = `
        <strong>✨ The Blessing is Complete</strong><br>
        ${data.message}<br>
        Rows inserted: ${data.inserted}${data.failed > 0 ? `, Failed: ${data.failed}` : ''}${correctedInfo}
    `;

    // Show results table
    if (data.rows && data.rows.length > 0) {
        let html = `<div class='table-wrap'><table><tr>`;
        const cols = Object.keys(data.rows[0]);
        cols.forEach(c => html += `<th>${escapeHtml(c)}</th>`);
        html += "</tr>";

        data.rows.forEach(r => {
            html += "<tr>";
            cols.forEach(c => html += `<td>${escapeHtml(r[c] || '')}</td>`);
            html += "</tr>";
        });

        html += "</table></div>";
        resultsDiv.innerHTML = html;
    }

    // Add cathedral pulse effect
    panel.className = "panel cathedral";
    panel.style.display = "block";

    // Scroll to results
    panel.scrollIntoView({ behavior: 'smooth' });
}

/**
 * Daemon Memory: Save preferences
 * Stores table-to-database mappings for future imports
 */
function saveDaemonMemory(dbName, tableName) {
    localStorage.setItem('lastDatabaseName', dbName);
    localStorage.setItem('lastTableName', tableName);

    // Store table-to-database mapping for intelligent defaults
    try {
        let tableMappings = JSON.parse(localStorage.getItem('tableToDatabaseMap') || '{}');
        tableMappings[tableName] = dbName;
        localStorage.setItem('tableToDatabaseMap', JSON.stringify(tableMappings));
    } catch (e) {
        console.error('Failed to save table-database mapping:', e);
    }
}

/**
 * Toggle between dropdown and text input modes for database name
 */
function toggleDatabaseInput() {
    const selectElement = document.getElementById("databaseNameSelect");
    const inputElement = document.getElementById("databaseNameInput");
    const toggleBtn = document.getElementById("toggleInput");
    
    if (selectElement.style.display !== "none") {
        // Switch to text input mode
        selectElement.style.display = "none";
        inputElement.style.display = "inline-block";
        toggleBtn.textContent = "Switch to dropdown";
        inputElement.focus();
    } else {
        // Switch to dropdown mode
        selectElement.style.display = "inline-block";
        inputElement.style.display = "none";
        toggleBtn.textContent = "Toggle";
        
        // Fetch databases when switching to dropdown mode
        fetchDatabaseOptions();
    }
}

/**
 * Fetch available databases and populate dropdown
 */
function fetchDatabaseOptions() {
    fetch("api/getDatabases.php")
        .then(response => response.json())
        .then(data => {
            if (data.status === "ok") {
                const selectElement = document.getElementById("databaseNameSelect");
                const existingOptions = Array.from(selectElement.options).map(option => option.value);
                
                // Clear existing non-default options
                for (let i = selectElement.options.length - 1; i >= 1; i--) {
                    selectElement.remove(i);
                }
                
                // Add new available databases
                data.databases.forEach(dbName => {
                    if (!existingOptions.includes(dbName)) {
                        const option = document.createElement("option");
                        option.value = dbName;
                        option.textContent = dbName;
                        selectElement.appendChild(option);
                    }
                });
                
                // Set the default value if it exists
                const lastDbName = localStorage.getItem('lastDatabaseName') || 'importer_db';
                if (data.databases.includes(lastDbName)) {
                    selectElement.value = lastDbName;
                }
            }
        })
        .catch(error => {
            console.error("Error fetching databases:", error);
        });
}

/**
 * Daemon Memory: Load preferences
 */
function loadDaemonMemory() {
    // This function is now handled by the schema builder logic
    // The dropdown will automatically load the last selected value on page load
}

/**
 * Reset schema to auto-detected values (ignoring memory)
 */
function resetToAutoDetectedSchema() {
    if (!window.autoDetectedSchema) {
        alert('No auto-detected schema available');
        return;
    }

    // Restore auto-detected schema
    window.currentSchema = JSON.parse(JSON.stringify(window.autoDetectedSchema));

    // Hide memory indicator
    document.getElementById('schemaMemoryIndicator').style.display = 'none';

    // Re-populate schema table
    const tbody = document.getElementById("schemaTableBody");
    tbody.innerHTML = "";

    window.currentSchema.forEach((col, idx) => {
        const row = tbody.insertRow();

        // Highlight missing columns (orange) and new columns (green)
        const isMissing = col.missing_in_file === true;
        const isNew = col.is_new_column === true;

        let rowStyle = '';
        let tag = '';

        if (isMissing) {
            rowStyle = 'background: rgba(255,170,0,0.1); border-left: 3px solid #ffaa00;';
            tag = '<br><small style="color: #ffaa00;">⚠ Missing in file</small>';
        } else if (isNew) {
            rowStyle = 'background: rgba(0,255,170,0.1); border-left: 3px solid #00ffaa;';
            tag = '<br><small style="color: #00ffaa;">✨ New column</small>';
        }

        row.innerHTML = `
            <td style="${rowStyle}">
                <input type="checkbox" title="Remove this column from the import" onchange="updateSchema(${idx}, 'remove', this.checked)" />
                ${tag}
            </td>
            <td style="${rowStyle}">
                <input type="text" value="${escapeHtml(col.name)}"
                    onchange="updateSchema(${idx}, 'name', this.value)" />
            </td>
            <td style="${rowStyle}">
                <select onchange="updateSchema(${idx}, 'type', this.value)">
                    <option ${col.type==='VARCHAR'?'selected':''}>VARCHAR</option>
                    <option ${col.type==='TEXT'?'selected':''}>TEXT</option>
                    <option ${col.type==='INT'?'selected':''}>INT</option>
                    <option ${col.type==='DECIMAL'?'selected':''}>DECIMAL</option>
                    <option ${col.type==='DATE'?'selected':''}>DATE</option>
                    <option ${col.type==='DATETIME'?'selected':''}>DATETIME</option>
                    <option ${col.type==='BOOLEAN'?'selected':''}>BOOLEAN</option>
                    <option ${col.type==='ENUM'?'selected':''}>ENUM</option>
                </select>
            </td>
            <td style="${rowStyle}"><input type="text" value="${escapeHtml(col.length || '')}"
                onchange="updateSchema(${idx}, 'length', this.value)"
                ${['TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'INT'].includes(col.type) ? 'disabled' : ''}
                placeholder="precision,scale" /></td>
            <td style="text-align: center; ${rowStyle}"><input type="checkbox" ${col.nullable?'checked':''}
                onchange="updateSchema(${idx}, 'nullable', this.checked)"/></td>
            <td style="text-align: center; ${rowStyle}"><input type="checkbox" ${col.indexed?'checked':''}
                onchange="updateSchema(${idx}, 'indexed', this.checked)"/></td>
            <td style="${rowStyle}"><input type="text" value="${escapeHtml(col.comment || '')}"
                onchange="updateSchema(${idx}, 'comment', this.value)" placeholder="Description"/></td>
        `;
    });

    // Update SQL preview
    updateSQLPreview();

    alert('✓ Schema reset to auto-detected values');
}

/**
 * Show error (red rune)
 */
function showError(element, message) {
    element.className = "status error";
    element.innerHTML = `⚠ ${escapeHtml(message)}`;
}

/**
 * Escape HTML to prevent XSS
 */
function escapeHtml(text) {
    if (text === null || text === undefined) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
    fetchDatabaseOptions();

    // Pre-populate destination table name with last used table name (as a helpful hint)
    const lastTableName = localStorage.getItem('lastTableName');
    if (lastTableName) {
        const tableNameInput = document.getElementById('destinationTableName');
        if (tableNameInput) {
            tableNameInput.placeholder = `e.g., ${lastTableName}`;
        }
    }

    // Back to Top Button - Show/Hide on scroll
    const backToTopBtn = document.getElementById('backToTopBtn');

    window.addEventListener('scroll', function() {
        if (window.scrollY > 300) {
            backToTopBtn.classList.add('show');
        } else {
            backToTopBtn.classList.remove('show');
        }
    });

    // Smooth scroll to top when button is clicked
    backToTopBtn.addEventListener('click', function() {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });
});

/**
 * Table Name Autocomplete Controller
 * Intelligent autocomplete with frequency tracking and server integration
 */
(function() {
    'use strict';
    
    var dropdownOpen = false;
    var highlightedIndex = -1;
    var currentSuggestions = [];
    var fetchTimeout = null;
    
    var input = document.getElementById('destinationTableName');
    var dropdown = document.getElementById('tableNameDropdown');
    var suggestionsList = document.getElementById('suggestionsList');
    var recentSection = document.getElementById('recentSection');
    var noResults = document.getElementById('noResults');
    var dropdownStatus = document.getElementById('dropdownStatus');
    
    // Initialize
    function init() {
        if (!input || !dropdown) return;
        
        // Load initial suggestions from local memory
        loadLocalSuggestions('');
        
        // Event listeners
        input.addEventListener('focus', onFocus);
        input.addEventListener('input', onInput);
        input.addEventListener('keydown', onKeydown);
        input.addEventListener('blur', onBlur);
        
        // Close dropdown on escape
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && dropdownOpen) {
                closeDropdown();
                input.focus();
            }
        });
    }
    
    // Handle focus
    function onFocus() {
        var value = input.value.trim();
        if (!dropdownOpen) {
            openDropdown();
        }
        if (value === '') {
            loadLocalSuggestions('');
            updateStatus('Recent & frequent tables');
        } else {
            searchServer(value);
        }
    }
    
    // Handle input
    function onInput() {
        var value = input.value.trim();
        highlightedIndex = -1;
        
        if (fetchTimeout) {
            clearTimeout(fetchTimeout);
        }
        
        if (value === '') {
            loadLocalSuggestions('');
            updateStatus('Recent & frequent tables');
            return;
        }
        
        // Debounce server search
        fetchTimeout = setTimeout(function() {
            searchServer(value);
        }, 150);
    }
    
    // Load suggestions from local memory
    function loadLocalSuggestions(query) {
        if (!window.TableNameMemory) return;
        
        var recent = TableNameMemory.getRecent(3);
        var frequent = TableNameMemory.getFrequent(5);
        
        // Merge and deduplicate
        var seen = {};
        var all = [];
        
        [].concat(recent, frequent).forEach(function(name) {
            var lower = name.toLowerCase();
            if (!seen[lower] && (!query || name.toLowerCase().indexOf(query.toLowerCase()) !== -1)) {
                seen[lower] = true;
                all.push(name);
            }
        });
        
        all = all.slice(0, 10);
        renderSuggestions(all, true);
    }
    
    // Search server for table names
    function searchServer(query) {
        updateStatus('Searching...');
        
        var xhr = new XMLHttpRequest();
        xhr.open('GET', 'api/getTableNames.php?search=' + encodeURIComponent(query) + '&limit=15', true);
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4) {
                if (xhr.status === 200) {
                    try {
                        var data = JSON.parse(xhr.responseText);
                        if (data.status === 'ok') {
                            var names = data.table_names.map(function(t) { return t.name; });
                            renderSuggestions(names, false);
                            updateStatus(names.length + ' tables found');
                        }
                    } catch (e) {
                        console.error('Parse error:', e);
                        loadLocalSuggestions(query);
                        updateStatus('Using local memory');
                    }
                } else {
                    loadLocalSuggestions(query);
                    updateStatus('Using local memory');
                }
            }
        };
        xhr.onerror = function() {
            loadLocalSuggestions(query);
            updateStatus('Using local memory');
        };
        xhr.send();
    }
    
    // Render suggestions
    function renderSuggestions(names, fromMemory) {
        currentSuggestions = names;
        highlightedIndex = -1;
        
        if (names.length === 0) {
            suggestionsList.innerHTML = '';
            if (recentSection) recentSection.style.display = 'none';
            noResults.style.display = 'block';
            return;
        }
        
        noResults.style.display = 'none';
        
        if (fromMemory && names.length > 0) {
            if (recentSection) recentSection.style.display = 'block';
        }
        
        suggestionsList.innerHTML = names.map(function(name, index) {
            var meta = fromMemory ? getLocalMeta(name) : { count: null, time: null };
            var countBadge = meta.count ? '<span class="usage-count">' + meta.count + ' uses</span>' : '';
            
            return '<div class="cyber-autocomplete-item" ' +
                   'data-index="' + index + '" ' +
                   'data-value="' + escapeHtml(name) + '" ' +
                   'onmouseenter="highlightItem(' + index + ')" ' +
                   'onclick="selectItem(\'' + escapeHtml(name) + '\')">' +
                   '<span class="table-name">' + escapeHtml(name) + '</span>' +
                   '<span class="table-meta">' + countBadge + '</span>' +
                   '</div>';
        }).join('');
    }
    
    // Get local metadata for table name
    function getLocalMeta(name) {
        if (!window.TableNameMemory) return { count: null, time: null };
        var data = TableNameMemory.load ? TableNameMemory.load() : {};
        var key = name.toLowerCase();
        return data[key] || { count: null, time: null };
    }
    
    // Highlight item
    function highlightItem(index) {
        var items = suggestionsList.querySelectorAll('.cyber-autocomplete-item');
        items.forEach(function(item) {
            item.classList.remove('highlighted');
            item.classList.remove('selected');
        });
        
        if (index >= 0 && index < items.length) {
            items[index].classList.add('highlighted');
            highlightedIndex = index;
            
            // Scroll into view
            items[index].scrollIntoView({ block: 'nearest' });
        }
    }
    
    // Select item
    function selectItem(name) {
        input.value = name;
        if (window.TableNameMemory) {
            TableNameMemory.recordUsage(name);
        }
        closeDropdown();
        input.focus();
    }
    
    // Open dropdown
    function openDropdown() {
        dropdown.style.display = 'block';
        dropdownOpen = true;
    }
    
    // Close dropdown
    function closeDropdown() {
        dropdown.style.display = 'none';
        dropdownOpen = false;
        highlightedIndex = -1;
    }
    
    // Toggle dropdown
    window.toggleTableNameDropdown = function() {
        if (dropdownOpen) {
            closeDropdown();
        } else {
            openDropdown();
            input.focus();
            if (input.value.trim() === '') {
                loadLocalSuggestions('');
            }
        }
    };
    
    // Handle keyboard navigation
    function onKeydown(e) {
        if (!dropdownOpen) {
            if (e.key === 'ArrowDown' || e.key === 'ArrowUp') {
                e.preventDefault();
                openDropdown();
                loadLocalSuggestions(input.value.trim());
            }
            return;
        }
        
        var items = suggestionsList.querySelectorAll('.cyber-autocomplete-item');
        
        switch (e.key) {
            case 'ArrowDown':
                e.preventDefault();
                highlightedIndex = Math.min(highlightedIndex + 1, items.length - 1);
                highlightItem(highlightedIndex);
                break;
                
            case 'ArrowUp':
                e.preventDefault();
                highlightedIndex = Math.max(highlightedIndex - 1, 0);
                highlightItem(highlightedIndex);
                break;
                
            case 'Enter':
                e.preventDefault();
                if (highlightedIndex >= 0 && items[highlightedIndex]) {
                    var value = items[highlightedIndex].dataset.value;
                    selectItem(value);
                }
                break;
                
            case 'Escape':
                e.preventDefault();
                closeDropdown();
                input.focus();
                break;
        }
    }
    
    // Handle blur
    function onBlur() {
        setTimeout(function() {
            if (dropdownOpen) {
                var value = input.value.trim();
                if (value && window.TableNameMemory) {
                    TableNameMemory.recordUsage(value);
                }
                closeDropdown();
            }
        }, 150);
    }
    
    // Update status text
    function updateStatus(text) {
        if (dropdownStatus) {
            dropdownStatus.textContent = text;
        }
    }
    
    // Expose selectItem and highlightItem globally
    window.selectItem = selectItem;
    window.highlightItem = highlightItem;
    
    // Record successful imports for autocomplete learning
    window.recordTableNameUsage = function(tableName) {
        if (window.TableNameMemory && tableName) {
            TableNameMemory.recordUsage(tableName);
        }
    };
    
    // Initialize on DOM ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();

/**
 * Animated Progress Monitor
 * Real-time progress tracking with polling
 */
var ProgressMonitor = (function() {
    var pollInterval = null;
    var currentKey = null;
    var startTime = null;
    var retries = 0;
    var maxRetries = 3;
    var currentPhase = 'processing';
    
    function start(key, options) {
        options = options || {};
        currentKey = key;
        startTime = Date.now();
        retries = 0;
        currentPhase = options.phase || 'processing';
        
        // Show progress container
        var container = document.getElementById('progressContainer');
        if (container) {
            container.style.display = 'block';
            // Add glow effect
            container.classList.add('progress-bar-glow');
        }
        
        // Reset progress bar
        updateUI({
            percent: 0,
            current: 0,
            total: options.total || 100,
            message: options.message || 'Starting...',
            phase: currentPhase
        });
        
        // Hide autocomplete dropdown if open
        var autocomplete = document.getElementById('tableNameDropdown');
        if (autocomplete) {
            autocomplete.style.display = 'none';
        }
        
        // Start polling
        poll(options.onComplete);
    }
    
    function poll(onComplete) {
        if (!currentKey) return;
        
        var xhr = new XMLHttpRequest();
        xhr.open('GET', 'api/getProgress.php?key=' + currentKey, true);
        xhr.timeout = 5000;
        
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4) {
                if (xhr.status === 200) {
                    try {
                        var data = JSON.parse(xhr.responseText);
                        if (data.status === 'ok' && data.progress) {
                            updateUI(data.progress);
                            
                            // Stop polling when progress is completed OR when upload is ready (any percent)
                            // The 'ready' phase means parsing is done - no need to wait for exact 100%
                            if (data.progress.status === 'completed' ||
                               data.progress.phase === 'ready') {
                                stop();
                                if (onComplete) onComplete(data.progress);
                                return;
                            }
                            
                            // Continue polling with adaptive interval
                            var interval = data.progress.percent < 50 ? 200 : 150;
                            pollInterval = setTimeout(function() {
                                poll(onComplete);
                            }, interval);
                        } else {
                            handleError('Progress not ready', onComplete);
                        }
                    } catch (e) {
                        handleError('Parse error', onComplete);
                    }
                } else {
                    handleError('HTTP error: ' + xhr.status, onComplete);
                }
            }
        };
        
        xhr.onerror = function() {
            handleError('Network error', onComplete);
        };
        
        xhr.ontimeout = function() {
            handleError('Timeout', onComplete);
        };
        
        xhr.send();
    }
    
    function handleError(msg, onComplete) {
        retries++;
        if (retries >= maxRetries) {
            console.warn('Progress polling failed: ' + msg);
            stop();
            if (onComplete) onComplete(null);
            return;
        }
        // Retry with longer delay
        pollInterval = setTimeout(function() {
            poll(onComplete);
        }, 500);
    }
    
    function updateUI(progress) {
        // Update bar width
        var bar = document.getElementById('progressBar');
        if (bar) {
            bar.style.width = progress.percent + '%';
        }
        
        // Update percentage text
        var percent = document.getElementById('progressPercent');
        if (percent) {
            percent.textContent = progress.percent + '%';
        }
        
        // Update current/total
        var current = document.getElementById('progressCurrent');
        var total = document.getElementById('progressTotal');
        if (current) current.textContent = (progress.current || 0).toLocaleString();
        if (total) total.textContent = (progress.total || 0).toLocaleString();
        
        // Update message
        var message = document.getElementById('progressMessage');
        if (message) {
            message.textContent = progress.message || '';
        }
        
        // Update phase
        var phaseText = document.querySelector('.phase-text');
        var phaseIcon = document.querySelector('.phase-icon');
        if (phaseText) {
            var phaseNames = {
                'parsing': 'Analyzing Data',
                'inserting': 'Inserting Rows',
                'completed': 'Complete!',
                'ready': 'Ready'
            };
            phaseText.textContent = phaseNames[progress.phase] || progress.phase;
        }
        if (phaseIcon) {
            var icons = {
                'parsing': '📊',
                'inserting': '💾',
                'completed': '✅',
                'ready': '⏳'
            };
            phaseIcon.textContent = icons[progress.phase] || '⏳';
        }
        
        // Update ETA
        var eta = document.getElementById('progressETA');
        if (eta && progress.estimated_remaining) {
            var seconds = Math.ceil(progress.estimated_remaining);
            if (seconds > 60) {
                eta.textContent = '⏱ ~' + Math.ceil(seconds / 60) + ' min remaining';
            } else if (seconds > 0) {
                eta.textContent = '⏱ ~' + seconds + ' sec remaining';
            } else {
                eta.textContent = '';
            }
        }
        
        // Add glow effect when near completion
        var container = document.getElementById('progressContainer');
        if (container && progress.percent >= 90) {
            container.style.boxShadow = '0 0 30px rgba(0,255,170,0.6), 0 0 60px rgba(0,255,170,0.3)';
        }
    }
    
    function stop() {
        if (pollInterval) {
            clearTimeout(pollInterval);
            pollInterval = null;
        }
        currentKey = null;
        
        // Hide dots animation
        var dots = document.getElementById('progressDots');
        if (dots) {
            dots.style.opacity = '0.3';
        }
        
        // Remove glow effect
        var container = document.getElementById('progressContainer');
        if (container) {
            container.classList.remove('progress-bar-glow');
        }
    }
    
    function hide() {
        stop();
        var container = document.getElementById('progressContainer');
        if (container) {
            container.style.display = 'none';
        }
    }
    
    return {
        start: start,
        stop: stop,
        hide: hide,
        update: updateUI
    };
})();
</script>

<!-- Floating Back to Top Button -->
<button id="backToTopBtn" title="Back to Top">↑</button>

</body>
</html>
