<?php
include_once(__DIR__ . "/../config.php");
include_once(__DIR__ . "/../lib/DatabaseHelper.php");
include_once(__DIR__ . "/../lib/FormBuilder.php");

// Get table parameter
$table = $_GET['t'] ?? '';
if (empty($table)) {
    die('No table specified');
}

// Get database parameter
$db = $_GET['db'] ?? '';
if (empty($db)) {
    die('No database specified');
}

// Validate database name
if (!preg_match('/^[a-zA-Z0-9_]+$/', $db)) {
    die('Invalid database name');
}

// Sanitize database name
$db = preg_replace('/[^a-zA-Z0-9_]/', '', $db);

// Log parameter propagation
error_log("DEBUG: db parameter value = $db");
error_log("DEBUG: table parameter value = $table");

// Pagination settings
$limit = 20;
$page = $_GET['page'] ?? 1;
$offset = ($page - 1) * $limit;

// Search functionality
$search = $_GET['search'] ?? '';
$searchCondition = '';
$searchParams = [];

if (!empty($search)) {
    $searchColumns = DatabaseHelper::getColumns($table, $db);
    $searchCondition = ' WHERE ';
    $conditions = [];
    foreach ($searchColumns as $column) {
        if ($column['type'] !== 'int') { // Skip integer columns for LIKE search
            $conditions[] = "{$column['name']} LIKE ?";
            $searchParams[] = "%{$search}%";
        }
    }
    $searchCondition .= implode(' OR ', $conditions);
}

// Get table metadata
$metadata = DatabaseHelper::getColumns($table, $db);
// Get rows with pagination
$rows = DatabaseHelper::getRows($table, $db, $limit, $offset);
// Get total count for pagination
$totalRows = DatabaseHelper::getRowCount($table, $db, $searchCondition, $searchParams);
$totalPages = ceil($totalRows / $limit);

// Get columns for display
$columns = $metadata;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($table) ?> CRUD - <?= $metadata['comment'] ?? ucfirst($table) ?></title>
    <link rel="stylesheet" href="../lib/forms-cyberpunk.css">
    <style>
        body {
            background-color: #0a0a0a;
            color: #00ff00;
            font-family: 'Courier New', monospace;
            margin: 0;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
            padding: 20px;
            border-bottom: 1px solid #00ff00;
        }
        
        .search-box {
            margin-bottom: 20px;
            text-align: center;
        }
        
        .search-box input {
            background-color: #111;
            border: 1px solid #00ff00;
            color: #00ff00;
            padding: 10px;
            width: 300px;
            border-radius: 4px;
        }
        
        .search-box button {
            background-color: #00ff00;
            color: #000;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            cursor: pointer;
            font-weight: bold;
        }
        
        .search-box button:hover {
            background-color: #00cc00;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
            background-color: #111;
            border: 1px solid #00ff00;
        }
        
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #00ff00;
        }
        
        th {
            background-color: #003300;
            font-weight: bold;
        }
        
        tr:hover {
            background-color: #002200;
        }
        
        .actions {
            display: flex;
            gap: 10px;
        }
        
        .btn {
            padding: 6px 12px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-family: 'Courier New', monospace;
            font-size: 12px;
        }
        
        .btn-edit {
            background-color: #0066cc;
            color: white;
        }
        
        .btn-delete {
            background-color: #cc0000;
            color: white;
        }
        
        .btn-insert {
            background-color: #00cc00;
            color: black;
            font-weight: bold;
            padding: 10px 20px;
        }
        
        .pagination {
            text-align: center;
            margin-top: 20px;
        }
        
        .pagination a {
            color: #00ff00;
            padding: 8px 16px;
            text-decoration: none;
            border: 1px solid #00ff00;
            margin: 0 4px;
        }
        
        .pagination a:hover {
            background-color: #003300;
        }
        
        .pagination .current {
            background-color: #00ff00;
            color: #000;
        }
        
        .crud-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .crud-title {
            margin: 0;
        }
        
        .crud-actions button {
            margin-left: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1 class="crud-title"><?= htmlspecialchars(ucfirst($table)) ?> CRUD Interface</h1>
            <p><?= htmlspecialchars($metadata['comment'] ?? ucfirst($table)) ?></p>
        </div>
        
        <div class="crud-header">
            <div class="search-box">
                <form method="GET" style="display: inline;">
                    <input type="hidden" name="t" value="<?= htmlspecialchars($table) ?>">
                    <input type="hidden" name="db" value="<?= htmlspecialchars($db) ?>">
                    <input type="text" name="search" placeholder="Search records..." value="<?= htmlspecialchars($search) ?>">
                    <button type="submit">Search</button>
                </form>
            </div>
            <div class="crud-actions">
                <button class="btn btn-insert" onclick="window.location.href='insert.php?t=<?= htmlspecialchars($table) ?>&db=<?= urlencode($db) ?>'">
                    ➕ Insert New Record
                </button>
            </div>
        </div>
        
        <table>
            <thead>
                <tr>
                    <?php foreach ($columns as $column): ?>
                        <th><?= htmlspecialchars($column['name']) ?></th>
                    <?php endforeach; ?>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($rows)): ?>
                    <tr>
                        <td colspan="<?= count($columns) + 1 ?>" style="text-align: center;">
                            No records found
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($rows as $row): ?>
                        <tr>
                            <?php foreach ($columns as $column): ?>
                                <td>
                                    <?= htmlspecialchars($row[$column['name']] ?? '') ?>
                                </td>
                            <?php endforeach; ?>
                            <td class="actions">
                                <button class="btn btn-edit" onclick="window.location.href='edit.php?t=<?= htmlspecialchars($table) ?>&db=<?= urlencode($db) ?>&id=<?= urlencode($row['id'] ?? '') ?>'">
                                    ✏️ Edit
                                </button>
                                <button class="btn btn-delete" onclick="if(confirm('Are you sure you want to delete this record?')) window.location.href='delete.php?t=<?= htmlspecialchars($table) ?>&db=<?= urlencode($db) ?>&id=<?= urlencode($row['id'] ?? '') ?>'">
                                    ❌ Delete
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
        
        <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?t=<?= htmlspecialchars($table) ?>&db=<?= urlencode($db) ?>&page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>">Previous</a>
            <?php endif; ?>
            
            <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                <?php if ($i == $page): ?>
                    <span class="current"><?= $i ?></span>
                <?php else: ?>
                    <a href="?t=<?= htmlspecialchars($table) ?>&db=<?= urlencode($db) ?>&page=<?= $i ?>&search=<?= urlencode($search) ?>"><?= $i ?></a>
                <?php endif; ?>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
                <a href="?t=<?= htmlspecialchars($table) ?>&db=<?= urlencode($db) ?>&page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>">Next</a>
            <?php endif; ?>
        </div>
        <?php endif; ?>
    </div>
</body>
</html>
