<?php
/**
 * Form - The Dynamic Wrapper
 * Wraps generated HTML forms to allow binding data and setting attributes
 */
class Form {
    private string $htmlContent;
    private string $formPath;
    
    public function __construct(string $formPath) {
        if (!file_exists($formPath)) {
            throw new Exception("Form file not found: {$formPath}");
        }
        $this->formPath = $formPath;
        $this->htmlContent = file_get_contents($formPath);
    }
    
    /**
     * Bind data to form fields
     * @param array $data Associative array of field => value pairs
     * @return void
     */
    public function bind(array $data): void {
        foreach ($data as $field => $value) {
            // For input fields, we'll inject the value attribute
            if (strpos($this->htmlContent, "name=\"{$field}\"") !== false) {
                // Handle different input types
                if (strpos($this->htmlContent, "type=\"checkbox\"") !== false) {
                    // Checkbox handling - check if value matches the checkbox value
                    $checkboxPattern = '/<input[^>]*type="checkbox"[^>]*name="' . preg_quote($field, '/') . '"[^>]*value="([^"]*)"[^>]*>/i';
                    if (preg_match($checkboxPattern, $this->htmlContent, $matches)) {
                        $checkboxValue = $matches[1];
                        if ($value == $checkboxValue) {
                            $this->htmlContent = preg_replace(
                                '/(<input[^>]*type="checkbox"[^>]*name="' . preg_quote($field, '/') . '"[^>]*value="' . preg_quote($checkboxValue, '/') . '"[^>]*>)/i',
                                '$1 checked',
                                $this->htmlContent
                            );
                        }
                    }
                } else {
                    // For text inputs, textarea, select
                    $pattern = '/(<input[^>]*name="' . preg_quote($field, '/') . '"[^>]*value=")[^"]*(")/i';
                    if (preg_match($pattern, $this->htmlContent)) {
                        $val = (string)$value;
                        $this->htmlContent = preg_replace(
                            $pattern,
                            '$1' . htmlspecialchars($val, ENT_QUOTES, 'UTF-8') . '$2',
                            $this->htmlContent
                        );
                    } else {
                        // Add value attribute if missing
                        $val = (string)$value;
                        $this->htmlContent = preg_replace(
                            '/(<input[^>]*name="' . preg_quote($field, '/') . '"[^>]*)(>)/i',
                            '$1 value="' . htmlspecialchars($val, ENT_QUOTES, 'UTF-8') . '"$2',
                            $this->htmlContent
                        );
                    }
                    // For textarea
                    $textareaPattern = '/(<textarea[^>]*name="' . preg_quote($field, '/') . '"[^>]*>)([^<]*)(<\/textarea>)/i';
                    if (preg_match($textareaPattern, $this->htmlContent)) {
                        $val = (string)$value;
                        $this->htmlContent = preg_replace(
                            $textareaPattern,
                            '$1' . htmlspecialchars($val, ENT_QUOTES, 'UTF-8') . '$3',
                            $this->htmlContent
                        );
                    }
                }
            }
        }
    }
    
    /**
     * Set form mode (insert/edit)
     * @param string $mode Either 'insert' or 'edit'
     * @return void
     */
    public function setMode(string $mode): void {
        if ($mode === 'edit') {
            // Add hidden input for edit mode
            $this->htmlContent = preg_replace(
                '/(<form[^>]*class="phoenix-form)/i',
                '$1 data-mode="edit"',
                $this->htmlContent
            );
        } else {
            $this->htmlContent = preg_replace(
                '/(<form[^>]*class="phoenix-form)/i',
                '$1 data-mode="insert"',
                $this->htmlContent
            );
        }
    }
    
    /**
     * Set form action URL
     * @param string $action The action URL
     * @return void
     */
    public function setAction(string $action): void {
        $this->htmlContent = preg_replace(
            '/(<form[^>]*class="phoenix-form)/i',
            '$1 action="' . htmlspecialchars($action, ENT_QUOTES, 'UTF-8') . '"',
            $this->htmlContent
        );
    }
    
    /**
     * Render the final HTML form
     * @return string
     */
    public function render(): string {
        return $this->htmlContent;
    }
    
    /**
     * Get the raw HTML content
     * @return string
     */
    public function getHtml(): string {
        return $this->htmlContent;
    }
}
