<?php

declare(strict_types=1);

/**
 * Class FormTemplate
 *
 * A simple and fast template wrapper for PhoenixForms.
 * It loads .tpl files, replaces placeholders, and renders the output.
 */
class FormTemplate
{
    private string $templateContent = '';

    /**
     * Loads a template file from the specified path.
     *
     * @param string $path Full path to the .tpl file.
     * @return self
     * @throws Exception if the template file is not found or readable.
     */
    public function loadTemplate(string $path): self
    {
        if (!file_exists($path) || !is_readable($path)) {
            throw new Exception("Template file not found or not readable: {$path}");
        }
        $this->templateContent = file_get_contents($path);
        return $this;
    }

    /**
     * Replaces placeholders in the template.
     * Placeholders are in the format {{KEY}}.
     *
     * @param string $key The placeholder key (without brackets).
     * @param string $value The value to substitute.
     * @return self
     */
    public function replace(string $key, string $value): self
    {
        $this->templateContent = str_replace('{{' . $key . '}}', $value, $this->templateContent);
        return $this;
    }

    /**
     * Renders the final template content.
     * @return string
     */
    public function render(): string
    {
        return $this->templateContent;
    }
}