<?php
/**
 * ImportLogger - The Chronicle Keeper
 * Logs every import operation with full schema capture and version management
 */

require_once 'DatabaseHelper.php';
require_once 'ErrorHandler.php';

class ImportLogger {
    const CURRENT_VERSION = 1;
    
    /**
     * Ensure log tables exist and are up to date in the CURRENT database
     * Creates tables if missing, migrates if version outdated
     * NOTE: This operates on whatever database is currently selected
     */
    public static function ensureLogTablesExist() {
        try {
            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();

            // Check if a database is currently selected
            $result = $conn->query("SELECT DATABASE()");
            if ($result && $result->num_rows > 0) {
                $row = $result->fetch_row();
                $currentDb = $row[0] ?? null;

                // If no database is selected, select the default database
                if ($currentDb === null) {
                    if (defined('DB_NAME') && DB_NAME !== '') {
                        $conn->select_db(DB_NAME);
                        ErrorHandler::logError('Auto-selected default database: ' . DB_NAME);
                    }
                }
            }

            // Work with currently selected database - NOW GUARANTEED TO HAVE A DATABASE

            // Check if version table exists
            if (!$db->tableExists('import_logger_meta')) {
                self::createMetaTable($conn);
            }
            
            // Check current version
            $currentVersion = self::getCurrentVersion($conn);
            
            // Create or migrate tables
            if ($currentVersion === 0) {
                // Tables don't exist, create them
                self::createLogTables($conn);
                self::setVersion($conn, self::CURRENT_VERSION);
                ErrorHandler::logError('Import logger tables created', ['version' => self::CURRENT_VERSION]);
            } elseif ($currentVersion < self::CURRENT_VERSION) {
                // Tables exist but need migration
                self::migrateLogTables($conn, $currentVersion);
                self::setVersion($conn, self::CURRENT_VERSION);
                ErrorHandler::logError('Import logger tables migrated', [
                    'from_version' => $currentVersion,
                    'to_version' => self::CURRENT_VERSION
                ]);
            }
            
            return true;
        } catch (Exception $e) {
            ErrorHandler::logError('Failed to ensure log tables exist: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Create metadata table for version tracking
     */
    private static function createMetaTable($conn) {
        $sql = "CREATE TABLE import_logger_meta (
            meta_key VARCHAR(64) PRIMARY KEY,
            meta_value TEXT,
            ultima_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if (!$conn->query($sql)) {
            throw new Exception('Failed to create import_logger_meta table: ' . $conn->error);
        }
    }
    
    /**
     * Get current version from metadata table
     */
    private static function getCurrentVersion($conn) {
        $result = $conn->query("SELECT meta_value FROM import_logger_meta WHERE meta_key = 'version'");
        
        if (!$result || $result->num_rows === 0) {
            // Check if import_logs table exists (legacy check)
            $tableCheck = $conn->query("SHOW TABLES LIKE 'import_logs'");
            return ($tableCheck && $tableCheck->num_rows > 0) ? 1 : 0;
        }
        
        $row = $result->fetch_assoc();
        return (int)$row['meta_value'];
    }
    
    /**
     * Set version in metadata table
     */
    private static function setVersion($conn, $version) {
        $stmt = $conn->prepare("INSERT INTO import_logger_meta (meta_key, meta_value) VALUES ('version', ?) 
                                ON DUPLICATE KEY UPDATE meta_value = ?");
        $versionStr = (string)$version;
        $stmt->bind_param("ss", $versionStr, $versionStr);
        
        if (!$stmt->execute()) {
            throw new Exception('Failed to set version: ' . $stmt->error);
        }
    }
    
    /**
     * Create log tables (v1)
     */
    private static function createLogTables($conn) {
        // Create import_logs table
        $sql1 = "CREATE TABLE import_logs (
            import_log_id VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',
            
            -- Import Metadata
            database_name VARCHAR(64) NOT NULL COMMENT 'Target database name',
            table_name VARCHAR(64) NOT NULL COMMENT 'Target table name',
            file_name VARCHAR(255) NULL COMMENT 'Original uploaded filename',
            file_type ENUM('XLSX','CSV') NOT NULL COMMENT 'File type',
            file_size INT NULL COMMENT 'File size in bytes',
            file_path VARCHAR(500) NULL COMMENT 'Persistent file path on server',

            -- Operation Results
            operation_status ENUM('pending','success','error','partial') NOT NULL DEFAULT 'pending' COMMENT 'Overall operation status',
            create_table_status ENUM('pending','created','already_exists','error') NOT NULL DEFAULT 'pending' COMMENT 'CREATE TABLE result',
            create_table_sql TEXT NULL COMMENT 'Full CREATE TABLE statement executed',
            create_table_error TEXT NULL COMMENT 'CREATE TABLE error message if failed',
            
            -- Data Insertion Stats
            rows_processed INT DEFAULT 0 COMMENT 'Total rows processed',
            rows_inserted INT DEFAULT 0 COMMENT 'Successfully inserted rows',
            rows_failed INT DEFAULT 0 COMMENT 'Failed row count',
            error_details JSON NULL COMMENT 'Detailed error information',
            
            -- CSV-Specific Metadata
            csv_delimiter VARCHAR(10) NULL COMMENT 'Detected delimiter for CSV',
            csv_encoding VARCHAR(20) NULL COMMENT 'Detected encoding for CSV',
            csv_normalized_rows INT DEFAULT 0 COMMENT 'Rows normalized in CSV',
            
            -- Audit Fields (Spanish canonical)
            alta_db TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'Import timestamp',
            alta_por VARCHAR(32) DEFAULT 'system' COMMENT 'User who performed import',
            ultimo_cambio TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP COMMENT 'Last modification',
            ultimo_cambio_por VARCHAR(32) NULL DEFAULT NULL COMMENT 'Last modifier',
            
            -- Indexes
            INDEX idx_database_name (database_name),
            INDEX idx_table_name (table_name),
            INDEX idx_status (operation_status),
            INDEX idx_alta_db (alta_db)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if (!$conn->query($sql1)) {
            throw new Exception('Failed to create import_logs table: ' . $conn->error);
        }
        
        // Create import_schema_logs table
        $sql2 = "CREATE TABLE import_schema_logs (
            import_schema_log_id VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',
            import_log_id VARCHAR(32) NOT NULL COMMENT 'FK to import_logs',
            
            -- Column Definition
            column_name VARCHAR(64) NOT NULL COMMENT 'Column name',
            column_order INT NOT NULL COMMENT 'Position in table (1-based)',
            data_type VARCHAR(50) NOT NULL COMMENT 'MySQL data type (VARCHAR, INT, etc)',
            length_values VARCHAR(255) NULL COMMENT 'Length for VARCHAR, values for ENUM, precision for DECIMAL',
            default_value VARCHAR(255) NULL COMMENT 'Default value if specified',
            is_nullable BOOLEAN DEFAULT TRUE COMMENT 'NULL constraint',
            is_indexed BOOLEAN DEFAULT FALSE COMMENT 'Whether column has an index',
            column_comment TEXT NULL COMMENT 'Column comment/description',
            
            -- Metadata
            was_auto_detected BOOLEAN DEFAULT TRUE COMMENT 'Whether schema was auto-detected or manually specified',
            original_column_name VARCHAR(64) NULL COMMENT 'Original column name before sanitization',
            
            -- Audit Fields
            alta_db TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'Record creation timestamp',
            alta_por VARCHAR(32) DEFAULT 'system' COMMENT 'User who created record',
            
            -- Foreign Key
            FOREIGN KEY (import_log_id) REFERENCES import_logs(import_log_id) ON DELETE CASCADE,
            
            -- Indexes
            INDEX idx_import_log_id (import_log_id),
            INDEX idx_column_name (column_name)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if (!$conn->query($sql2)) {
            throw new Exception('Failed to create import_schema_logs table: ' . $conn->error);
        }
    }
    
    /**
     * Migrate log tables from old version to current
     */
    private static function migrateLogTables($conn, $fromVersion) {
        // Future migrations will go here
        // Example: if ($fromVersion < 2) { ALTER TABLE ... }
        ErrorHandler::logError('Migration completed', ['from' => $fromVersion, 'to' => self::CURRENT_VERSION]);
    }
    
    /**
     * Start a new import log entry in the CURRENT database
     * Returns import_log_id
     * NOTE: Works with whatever database is currently selected - NO SWITCHING!
     */
    public static function startImport($metadata) {
        try {
            self::ensureLogTablesExist();
            
            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();
            // NO DATABASE SWITCHING - use current database
            
            $importLogId = $db->generateUUID();
            
            // Extract metadata
            $fileName = $metadata['file_name'] ?? null;
            $fileType = strtoupper($metadata['file_type'] ?? 'XLSX');
            $fileSize = $metadata['file_size'] ?? null;
            $filePath = $metadata['file_path'] ?? null;  // Persistent file path
            $csvDelimiter = $metadata['csv_delimiter'] ?? null;
            $csvEncoding = $metadata['csv_encoding'] ?? null;
            $csvNormalizedRows = $metadata['csv_normalized_rows'] ?? 0;
            $databaseName = $metadata['database_name'] ?? null;
            $tableName = $metadata['table_name'] ?? null;
            $altaPor = $metadata['alta_por'] ?? 'system';

            // Insert initial log entry
            $sql = "INSERT INTO import_logs (
                import_log_id,
                database_name,
                table_name,
                file_name,
                file_type,
                file_size,
                file_path,
                csv_delimiter,
                csv_encoding,
                csv_normalized_rows,
                operation_status,
                create_table_status,
                alta_por
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', 'pending', ?)";

            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                throw new Exception('Failed to prepare insert statement: ' . $conn->error);
            }

            $stmt->bind_param(
                "sssssississ",
                $importLogId,
                $databaseName,
                $tableName,
                $fileName,
                $fileType,
                $fileSize,
                $filePath,
                $csvDelimiter,
                $csvEncoding,
                $csvNormalizedRows,
                $altaPor
            );
            
            if (!$stmt->execute()) {
                throw new Exception('Failed to start import log: ' . $stmt->error);
            }
            
            ErrorHandler::logError('Import log started', ['import_log_id' => $importLogId, 'table' => $tableName]);
            
            return $importLogId;
            
        } catch (Exception $e) {
            ErrorHandler::logError('Failed to start import log: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Log CREATE TABLE operation in CURRENT database
     * NOTE: Works with whatever database is currently selected - NO SWITCHING!
     */
    public static function logCreateTable($importLogId, $data) {
        try {
            if (!$importLogId) {
                return; // Silently skip if no import log ID
            }
            
            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();
            // NO DATABASE SWITCHING - use current database
            
            $databaseName = $data['database_name'] ?? null;
            $tableName = $data['table_name'] ?? null;
            $createSql = $data['create_sql'] ?? null;
            $status = $data['status'] ?? 'error';
            $error = $data['error'] ?? null;
            
            // Map status
            $createTableStatus = ($status === 'success') ? 'created' : 'error';
            
            $sql = "UPDATE import_logs SET
                database_name = ?,
                table_name = ?,
                create_table_sql = ?,
                create_table_status = ?,
                create_table_error = ?,
                ultimo_cambio_por = 'system'
                WHERE import_log_id = ?";
            
            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                throw new Exception('Failed to prepare update statement: ' . $conn->error);
            }
            
            $stmt->bind_param("ssssss", $databaseName, $tableName, $createSql, $createTableStatus, $error, $importLogId);
            
            if (!$stmt->execute()) {
                throw new Exception('Failed to log CREATE TABLE: ' . $stmt->error);
            }
            
            ErrorHandler::logError('CREATE TABLE logged', ['import_log_id' => $importLogId, 'status' => $status]);
            
        } catch (Exception $e) {
            ErrorHandler::logError('Failed to log CREATE TABLE: ' . $e->getMessage());
        }
    }
    
    /**
     * Log schema details (column by column) in CURRENT database
     * NOTE: Works with whatever database is currently selected - NO SWITCHING!
     */
    public static function logSchema($importLogId, $schema) {
        try {
            if (!$importLogId || empty($schema)) {
                return; // Silently skip if no import log ID or schema
            }
            
            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();
            // NO DATABASE SWITCHING - use current database
            
            $sql = "INSERT INTO import_schema_logs (
                import_schema_log_id,
                import_log_id,
                column_name,
                column_order,
                data_type,
                length_values,
                default_value,
                is_nullable,
                is_indexed,
                column_comment,
                was_auto_detected,
                original_column_name,
                alta_por
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'system')";
            
            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                throw new Exception('Failed to prepare schema insert statement: ' . $conn->error);
            }
            
            $order = 1;
            foreach ($schema as $col) {
                $schemaLogId = $db->generateUUID();
                $columnName = $col['name'] ?? '';
                $dataType = $col['type'] ?? 'VARCHAR';
                $lengthValues = $col['length'] ?? null;
                $defaultValue = $col['default'] ?? null;
                $isNullable = isset($col['nullable']) ? (int)$col['nullable'] : 1;
                $isIndexed = isset($col['indexed']) ? (int)$col['indexed'] : 0;
                $columnComment = $col['comment'] ?? null;
                $wasAutoDetected = isset($col['auto_detected']) ? (int)$col['auto_detected'] : 1;
                $originalColumnName = $col['original'] ?? $columnName;
                
                $stmt->bind_param(
                    "sssissississ",
                    $schemaLogId,
                    $importLogId,
                    $columnName,
                    $order,
                    $dataType,
                    $lengthValues,
                    $defaultValue,
                    $isNullable,
                    $isIndexed,
                    $columnComment,
                    $wasAutoDetected,
                    $originalColumnName
                );
                
                if (!$stmt->execute()) {
                    ErrorHandler::logError('Failed to log schema column: ' . $stmt->error, ['column' => $columnName]);
                }
                
                $order++;
            }
            
            ErrorHandler::logError('Schema logged', ['import_log_id' => $importLogId, 'columns' => count($schema)]);
            
        } catch (Exception $e) {
            ErrorHandler::logError('Failed to log schema: ' . $e->getMessage());
        }
    }
    
    /**
     * Update import status with final results in CURRENT database
     * NOTE: Works with whatever database is currently selected - NO SWITCHING!
     */
    public static function updateImportStatus($importLogId, $results) {
        try {
            if (!$importLogId) {
                return; // Silently skip if no import log ID
            }
            
            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();
            // NO DATABASE SWITCHING - use current database
            
            $rowsProcessed = $results['rows_processed'] ?? 0;
            $rowsInserted = $results['rows_inserted'] ?? 0;
            $rowsFailed = $results['rows_failed'] ?? 0;
            $errors = $results['errors'] ?? [];
            
            // Determine overall status
            if ($rowsFailed === 0 && $rowsInserted > 0) {
                $operationStatus = 'success';
            } elseif ($rowsInserted > 0 && $rowsFailed > 0) {
                $operationStatus = 'partial';
            } else {
                $operationStatus = 'error';
            }
            
            // Convert errors to JSON
            $errorJson = !empty($errors) ? json_encode($errors, JSON_UNESCAPED_UNICODE) : null;
            
            $sql = "UPDATE import_logs SET
                operation_status = ?,
                rows_processed = ?,
                rows_inserted = ?,
                rows_failed = ?,
                error_details = ?,
                ultimo_cambio_por = 'system'
                WHERE import_log_id = ?";
            
            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                throw new Exception('Failed to prepare status update: ' . $conn->error);
            }
            
            $stmt->bind_param("siiiss", $operationStatus, $rowsProcessed, $rowsInserted, $rowsFailed, $errorJson, $importLogId);
            
            if (!$stmt->execute()) {
                throw new Exception('Failed to update import status: ' . $stmt->error);
            }
            
            ErrorHandler::logError('Import status updated', [
                'import_log_id' => $importLogId,
                'status' => $operationStatus,
                'inserted' => $rowsInserted,
                'failed' => $rowsFailed
            ]);
            
        } catch (Exception $e) {
            ErrorHandler::logError('Failed to update import status: ' . $e->getMessage());
        }
    }
    
    /**
     * Log an error for a specific import in CURRENT database
     * NOTE: Works with whatever database is currently selected - NO SWITCHING!
     */
    public static function logError($importLogId, $error) {
        try {
            if (!$importLogId) {
                return;
            }
            
            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();
            // NO DATABASE SWITCHING - use current database
            
            $sql = "UPDATE import_logs SET
                operation_status = 'error',
                create_table_error = ?,
                ultimo_cambio_por = 'system'
                WHERE import_log_id = ?";
            
            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                throw new Exception('Failed to prepare error update: ' . $conn->error);
            }
            
            $stmt->bind_param("ss", $error, $importLogId);
            $stmt->execute();
            
        } catch (Exception $e) {
            ErrorHandler::logError('Failed to log import error: ' . $e->getMessage());
        }
    }
    
    /**
     * Get import history with optional filters from CURRENT database
     * NOTE: Works with whatever database is currently selected - NO SWITCHING!
     */
    public static function getImportHistory($filters = []) {
        try {
            self::ensureLogTablesExist();
            
            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();
            // NO DATABASE SWITCHING - use current database
            
            $sql = "SELECT * FROM import_logs WHERE 1=1";
            $params = [];
            $types = '';
            
            // Apply filters
            if (!empty($filters['database_name'])) {
                $sql .= " AND database_name = ?";
                $params[] = $filters['database_name'];
                $types .= 's';
            }
            
            if (!empty($filters['table_name'])) {
                $sql .= " AND table_name = ?";
                $params[] = $filters['table_name'];
                $types .= 's';
            }
            
            if (!empty($filters['operation_status'])) {
                $sql .= " AND operation_status = ?";
                $params[] = $filters['operation_status'];
                $types .= 's';
            }
            
            // Order by most recent first
            $sql .= " ORDER BY alta_db DESC";
            
            // Apply limit
            $limit = $filters['limit'] ?? 100;
            $sql .= " LIMIT ?";
            $params[] = $limit;
            $types .= 'i';
            
            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                throw new Exception('Failed to prepare query: ' . $conn->error);
            }
            
            if (!empty($params)) {
                $stmt->bind_param($types, ...$params);
            }
            
            $stmt->execute();
            $result = $stmt->get_result();
            
            $logs = [];
            while ($row = $result->fetch_assoc()) {
                $logs[] = $row;
            }
            
            return $logs;
            
        } catch (Exception $e) {
            ErrorHandler::logError('Failed to get import history: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get schema details for a specific import from CURRENT database
     * NOTE: Works with whatever database is currently selected - NO SWITCHING!
     */
    public static function getSchemaByImportId($importLogId) {
        try {
            self::ensureLogTablesExist();

            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();
            // NO DATABASE SWITCHING - use current database

            $sql = "SELECT * FROM import_schema_logs
                    WHERE import_log_id = ?
                    ORDER BY column_order ASC";

            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                throw new Exception('Failed to prepare query: ' . $conn->error);
            }

            $stmt->bind_param("s", $importLogId);
            $stmt->execute();
            $result = $stmt->get_result();

            $schema = [];
            while ($row = $result->fetch_assoc()) {
                $schema[] = $row;
            }

            return $schema;

        } catch (Exception $e) {
            ErrorHandler::logError('Failed to get schema: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get the last successful schema for a specific table from CURRENT database
     * Returns schema in format compatible with arrival.php schema builder
     * NOTE: Works with whatever database is currently selected - NO SWITCHING!
     *
     * @param string $tableName The table name to look up
     * @param string|null $databaseName Optional database name filter
     * @return array|null Schema array or null if no history found
     */
    public static function getLastSuccessfulSchema($tableName, $databaseName = null) {
        try {
            // Try to ensure log tables exist, but don't fail if database doesn't exist
            try {
                self::ensureLogTablesExist();
            } catch (Exception $tableEx) {
                // Log tables don't exist yet - return null (no history)
                ErrorHandler::logError('Import log tables not found (expected for first import): ' . $tableEx->getMessage());
                return null;
            }

            $db = DatabaseHelper::getInstance();
            $conn = $db->getConnection();
            // NO DATABASE SWITCHING - use current database

            // Check if import_logs table exists before querying
            $tableCheck = $conn->query("SHOW TABLES LIKE 'import_logs'");
            if (!$tableCheck || $tableCheck->num_rows === 0) {
                // Table doesn't exist - no history available
                return null;
            }

            // Find the most recent successful import for this table WITH SCHEMA LOGS
            // Use INNER JOIN to skip imports that have no schema (e.g., skip-mode imports)
            $sql = "SELECT l.import_log_id, l.alta_db
                    FROM import_logs l
                    INNER JOIN (
                        SELECT import_log_id, COUNT(*) as schema_count
                        FROM import_schema_logs
                        GROUP BY import_log_id
                        HAVING COUNT(*) > 0
                    ) s ON l.import_log_id = s.import_log_id
                    WHERE l.table_name = ?
                    AND l.operation_status = 'success'";

            $params = [$tableName];
            $types = 's';

            // Optionally filter by database name
            if ($databaseName !== null) {
                $sql .= " AND l.database_name = ?";
                $params[] = $databaseName;
                $types .= 's';
            }

            $sql .= " ORDER BY l.alta_db DESC LIMIT 1";

            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                throw new Exception('Failed to prepare query: ' . $conn->error);
            }

            $stmt->bind_param($types, ...$params);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows === 0) {
                return null; // No previous import found
            }

            $importLog = $result->fetch_assoc();
            $importLogId = $importLog['import_log_id'];
            $importDate = $importLog['alta_db'];

            // Retrieve schema for this import
            $schemaLogs = self::getSchemaByImportId($importLogId);

            if (empty($schemaLogs)) {
                return null;
            }

            // Convert from import_schema_logs format to arrival.php schema format
            $schema = [];
            foreach ($schemaLogs as $schemaLog) {
                $schema[] = [
                    'name' => $schemaLog['column_name'],
                    'type' => $schemaLog['data_type'],
                    'length' => $schemaLog['length_values'],
                    'nullable' => (bool)$schemaLog['is_nullable'],
                    'indexed' => (bool)$schemaLog['is_indexed'],
                    'comment' => $schemaLog['column_comment'],
                    'original' => $schemaLog['original_column_name'],
                    'auto_detected' => (bool)$schemaLog['was_auto_detected']
                ];
            }

            return [
                'schema' => $schema,
                'import_date' => $importDate,
                'import_log_id' => $importLogId
            ];

        } catch (Exception $e) {
            ErrorHandler::logError('Failed to get last successful schema: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get list of recent table names from import history
     * 
     * @param string|null $search Optional search term for filtering
     * @param int $limit Maximum number of results to return
     * @return array List of table names with metadata
     */
    public static function getRecentTableNames($search = null, $limit = 20) {
        try {
            self::ensureLogTablesExist();
        } catch (Exception $e) {
            return [];
        }
        
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();
        
        // Check if import_logs table exists
        $tableCheck = $conn->query("SHOW TABLES LIKE 'import_logs'");
        if (!$tableCheck || $tableCheck->num_rows === 0) {
            return [];
        }
        
        $sql = "SELECT DISTINCT table_name, MAX(alta_db) as last_imported, 
                       COUNT(*) as import_count
                FROM import_logs 
                WHERE operation_status = 'success'";
        
        $params = [];
        $types = '';
        
        if ($search !== null && $search !== '') {
            $sql .= " AND table_name LIKE ?";
            $params[] = '%' . $search . '%';
            $types .= 's';
        }
        
        $sql .= " GROUP BY table_name ORDER BY last_imported DESC LIMIT ?";
        $params[] = $limit;
        $types .= 'i';
        
        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            return [];
        }
        
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $tableNames = [];
        while ($row = $result->fetch_assoc()) {
            $tableNames[] = [
                'name' => $row['table_name'],
                'last_imported' => $row['last_imported'],
                'import_count' => (int)$row['import_count']
            ];
        }
        
        return $tableNames;
    }
}
