<?php
/**
 * ProgressTracker - Real-time progress tracking for uploads and inserts
 * Uses file-based storage for thread-safe progress sharing
 */

class ProgressTracker
{
    private static function getProgressFile($key)
    {
        return sys_get_temp_dir() . '/importer_progress_' . md5($key) . '.json';
    }

    /**
     * Initialize progress tracking
     */
    public static function start($key, $total, $phase = 'processing')
    {
        $data = [
            'key' => $key,
            'phase' => $phase,
            'total' => (int) $total,
            'current' => 0,
            'percent' => 0,
            'status' => 'started',
            'started_at' => time(),
            'message' => 'Starting...'
        ];
        @file_put_contents(self::getProgressFile($key), json_encode($data));
        return $data;
    }

    /**
     * Update progress
     */
    public static function update($key, $current, $message = null)
    {
        $file = self::getProgressFile($key);
        if (!@file_exists($file)) {
            return;
        }

        $data = json_decode(@file_get_contents($file), true);
        if (!$data) return;
        
        $data['current'] = (int) $current;
        $data['percent'] = $data['total'] > 0 ? round(($current / $data['total']) * 100) : 0;
        
        if ($message) {
            $data['message'] = $message;
        }
        
        $data['updated_at'] = time();
        @file_put_contents($file, json_encode($data));
    }

    /**
     * Increment progress by 1
     */
    public static function increment($key, $message = null)
    {
        $file = self::getProgressFile($key);
        if (!@file_exists($file)) {
            return;
        }

        $data = json_decode(@file_get_contents($file), true);
        if (!$data) return;
        
        $data['current']++;
        $data['percent'] = $data['total'] > 0 ? round(($data['current'] / $data['total']) * 100) : 0;
        
        if ($message) {
            $data['message'] = $message;
        }
        
        $data['updated_at'] = time();
        @file_put_contents($file, json_encode($data));
    }

    /**
     * Complete progress
     */
    public static function complete($key, $message = 'Completed!')
    {
        $file = self::getProgressFile($key);
        if (!@file_exists($file)) {
            return;
        }

        $data = json_decode(@file_get_contents($file), true);
        if (!$data) return;
        
        $data['current'] = $data['total'];
        $data['percent'] = 100;
        $data['status'] = 'completed';
        $data['message'] = $message;
        $data['completed_at'] = time();
        
        // Calculate duration
        if (isset($data['started_at'])) {
            $data['duration'] = $data['completed_at'] - $data['started_at'];
        }
        
        @file_put_contents($file, json_encode($data));
    }

    /**
     * Set phase (e.g., 'parsing', 'inserting', 'completed')
     */
    public static function setPhase($key, $phase, $message = null)
    {
        $file = self::getProgressFile($key);
        if (!@file_exists($file)) {
            return;
        }

        $data = json_decode(@file_get_contents($file), true);
        if (!$data) return;
        
        $data['phase'] = $phase;
        $data['status'] = 'in_progress';
        // Recalculate percent based on current progress instead of resetting to 0
        $data['percent'] = $data['total'] > 0 ? round(($data['current'] / $data['total']) * 100) : 0;

        if ($message) {
            $data['message'] = $message;
        }
        
        @file_put_contents($file, json_encode($data));
    }

    /**
     * Get current progress
     */
    public static function get($key)
    {
        $file = self::getProgressFile($key);
        if (!@file_exists($file)) {
            return null;
        }
        $content = @file_get_contents($file);
        if (!$content) return null;
        return json_decode($content, true);
    }

    /**
     * Clean up progress file
     */
    public static function cleanup($key)
    {
        $file = self::getProgressFile($key);
        if (@file_exists($file)) {
            @unlink($file);
        }
    }

    /**
     * Generate unique progress key
     */
    public static function generateKey()
    {
        return uniqid('prog_', true);
    }
}
