<?php
/**
 * TableOperations - The Replicator
 * Handles table copy, move, rename, and delete operations across databases
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/DatabaseHelper.php';
require_once __DIR__ . '/ErrorHandler.php';

class TableOperations {

    /**
     * Get list of all databases
     */
    public static function getAllDatabases() {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        $result = $conn->query("SHOW DATABASES");
        $databases = [];

        while ($row = $result->fetch_assoc()) {
            $dbName = $row['Database'];
            // Skip system databases
            if (!in_array($dbName, ['information_schema', 'mysql', 'performance_schema', 'sys'])) {
                $databases[] = $dbName;
            }
        }

        return $databases;
    }

    /**
     * Get list of all tables in a database
     */
    public static function getTablesInDatabase($database) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        $database = $conn->real_escape_string($database);
        $result = $conn->query("SHOW TABLES FROM `$database`");

        $tables = [];
        while ($row = $result->fetch_array()) {
            $tables[] = $row[0];
        }

        return $tables;
    }

    /**
     * Get detailed information about a table
     */
    public static function getTableInfo($database, $tableName) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        $database = $conn->real_escape_string($database);
        $tableName = $conn->real_escape_string($tableName);

        // Get row count
        $result = $conn->query("SELECT COUNT(*) as count FROM `$database`.`$tableName`");
        $rowCount = $result->fetch_assoc()['count'];

        // Get table status (size, engine, etc.)
        $result = $conn->query("SHOW TABLE STATUS FROM `$database` WHERE Name = '$tableName'");
        $status = $result->fetch_assoc();

        return [
            'row_count' => $rowCount,
            'engine' => $status['Engine'] ?? 'Unknown',
            'data_length' => $status['Data_length'] ?? 0,
            'index_length' => $status['Index_length'] ?? 0,
            'auto_increment' => $status['Auto_increment'] ?? null,
            'collation' => $status['Collation'] ?? 'Unknown',
            'create_time' => $status['Create_time'] ?? null,
            'update_time' => $status['Update_time'] ?? null
        ];
    }

    /**
     * Get CREATE TABLE statement for a table
     */
    public static function getCreateTableSQL($database, $tableName) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        $database = $conn->real_escape_string($database);
        $tableName = $conn->real_escape_string($tableName);

        $result = $conn->query("SHOW CREATE TABLE `$database`.`$tableName`");
        $row = $result->fetch_assoc();

        return $row['Create Table'] ?? null;
    }

    /**
     * Copy table to another database
     *
     * @param string $sourceDb Source database name
     * @param string $sourceTable Source table name
     * @param string $targetDb Target database name
     * @param string $targetTable Target table name
     * @param array $options Copy options
     * @return array Result with success status and message
     */
    public static function copyTable($sourceDb, $sourceTable, $targetDb, $targetTable, $options = []) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        // Default options
        $defaults = [
            'structure' => true,
            'data' => true,
            'drop_table' => false,
            'add_constraints' => true,
            'add_auto_increment' => true
        ];
        $options = array_merge($defaults, $options);

        try {
            // Escape names
            $sourceDb = $conn->real_escape_string($sourceDb);
            $sourceTable = $conn->real_escape_string($sourceTable);
            $targetDb = $conn->real_escape_string($targetDb);
            $targetTable = $conn->real_escape_string($targetTable);

            ErrorHandler::logError('Starting table copy operation', [
                'source' => "$sourceDb.$sourceTable",
                'target' => "$targetDb.$targetTable",
                'options' => $options
            ]);

            // Check if source table exists
            $result = $conn->query("SHOW TABLES FROM `$sourceDb` LIKE '$sourceTable'");
            if ($result->num_rows === 0) {
                throw new Exception("Source table `$sourceDb`.`$sourceTable` does not exist");
            }

            // Create target database if it doesn't exist
            $conn->query("CREATE DATABASE IF NOT EXISTS `$targetDb` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

            // DROP target table if option is enabled
            if ($options['drop_table']) {
                $conn->query("DROP TABLE IF EXISTS `$targetDb`.`$targetTable`");
                ErrorHandler::logError('Dropped existing target table', ['table' => "$targetDb.$targetTable"]);
            }

            // Copy structure
            if ($options['structure']) {
                // Get CREATE TABLE statement
                $createSQL = self::getCreateTableSQL($sourceDb, $sourceTable);

                // Replace table name
                $createSQL = preg_replace(
                    '/CREATE TABLE `' . preg_quote($sourceTable, '/') . '`/',
                    "CREATE TABLE `$targetDb`.`$targetTable`",
                    $createSQL
                );

                // Remove AUTO_INCREMENT value if option is disabled
                if (!$options['add_auto_increment']) {
                    $createSQL = preg_replace('/AUTO_INCREMENT=\d+\s*/', '', $createSQL);
                }

                // Execute CREATE TABLE
                if (!$conn->query($createSQL)) {
                    throw new Exception("Failed to create table structure: " . $conn->error);
                }

                ErrorHandler::logError('Table structure copied', ['table' => "$targetDb.$targetTable"]);
            }

            // Copy data
            if ($options['data']) {
                // Check if target table exists (it should after structure copy)
                $result = $conn->query("SHOW TABLES FROM `$targetDb` LIKE '$targetTable'");
                if ($result->num_rows === 0) {
                    throw new Exception("Cannot copy data: target table does not exist. Enable 'structure' option.");
                }

                // Insert data using SELECT
                $insertSQL = "INSERT INTO `$targetDb`.`$targetTable` SELECT * FROM `$sourceDb`.`$sourceTable`";

                if (!$conn->query($insertSQL)) {
                    throw new Exception("Failed to copy data: " . $conn->error);
                }

                $rowCount = $conn->affected_rows;
                ErrorHandler::logError('Table data copied', [
                    'table' => "$targetDb.$targetTable",
                    'rows' => $rowCount
                ]);
            }

            return [
                'success' => true,
                'message' => "Table copied successfully from `$sourceDb`.`$sourceTable` to `$targetDb`.`$targetTable`",
                'rows_copied' => $rowCount ?? 0
            ];

        } catch (Exception $e) {
            ErrorHandler::logError('Table copy failed', [
                'error' => $e->getMessage(),
                'source' => "$sourceDb.$sourceTable",
                'target' => "$targetDb.$targetTable"
            ]);

            return [
                'success' => false,
                'message' => 'Copy failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Move table to another database (copy + delete source)
     */
    public static function moveTable($sourceDb, $sourceTable, $targetDb, $targetTable, $options = []) {
        // First copy the table
        $result = self::copyTable($sourceDb, $sourceTable, $targetDb, $targetTable, $options);

        if (!$result['success']) {
            return $result;
        }

        // If copy succeeded, delete source table
        try {
            $deleteResult = self::deleteTable($sourceDb, $sourceTable);

            if (!$deleteResult['success']) {
                return [
                    'success' => false,
                    'message' => 'Table copied but failed to delete source: ' . $deleteResult['message']
                ];
            }

            return [
                'success' => true,
                'message' => "Table moved successfully from `$sourceDb`.`$sourceTable` to `$targetDb`.`$targetTable`",
                'rows_moved' => $result['rows_copied']
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Table copied but failed to delete source: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Rename a table within the same database
     */
    public static function renameTable($database, $oldName, $newName) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        try {
            $database = $conn->real_escape_string($database);
            $oldName = $conn->real_escape_string($oldName);
            $newName = $conn->real_escape_string($newName);

            // Check if source table exists
            $result = $conn->query("SHOW TABLES FROM `$database` LIKE '$oldName'");
            if ($result->num_rows === 0) {
                throw new Exception("Table `$database`.`$oldName` does not exist");
            }

            // Check if target name already exists
            $result = $conn->query("SHOW TABLES FROM `$database` LIKE '$newName'");
            if ($result->num_rows > 0) {
                throw new Exception("Table `$database`.`$newName` already exists");
            }

            // Rename table
            $sql = "RENAME TABLE `$database`.`$oldName` TO `$database`.`$newName`";

            if (!$conn->query($sql)) {
                throw new Exception("Failed to rename table: " . $conn->error);
            }

            ErrorHandler::logError('Table renamed', [
                'database' => $database,
                'old_name' => $oldName,
                'new_name' => $newName
            ]);

            return [
                'success' => true,
                'message' => "Table renamed from `$oldName` to `$newName`"
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Rename failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Delete a table
     */
    public static function deleteTable($database, $tableName) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        try {
            $database = $conn->real_escape_string($database);
            $tableName = $conn->real_escape_string($tableName);

            // Check if table exists
            $result = $conn->query("SHOW TABLES FROM `$database` LIKE '$tableName'");
            if ($result->num_rows === 0) {
                throw new Exception("Table `$database`.`$tableName` does not exist");
            }

            // Drop table
            $sql = "DROP TABLE `$database`.`$tableName`";

            if (!$conn->query($sql)) {
                throw new Exception("Failed to delete table: " . $conn->error);
            }

            ErrorHandler::logError('Table deleted', [
                'database' => $database,
                'table' => $tableName
            ]);

            return [
                'success' => true,
                'message' => "Table `$database`.`$tableName` deleted successfully"
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Delete failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Export table structure as SQL file
     */
    public static function exportStructure($database, $tableName, $options = []) {
        $defaults = [
            'drop_table' => true,
            'add_auto_increment' => true
        ];
        $options = array_merge($defaults, $options);

        try {
            $createSQL = self::getCreateTableSQL($database, $tableName);

            if (!$createSQL) {
                throw new Exception("Failed to get table structure");
            }

            $sql = "-- Table structure for `$database`.`$tableName`\n";
            $sql .= "-- Generated on " . date('Y-m-d H:i:s') . "\n\n";

            if ($options['drop_table']) {
                $sql .= "DROP TABLE IF EXISTS `$tableName`;\n\n";
            }

            if (!$options['add_auto_increment']) {
                $createSQL = preg_replace('/AUTO_INCREMENT=\d+\s*/', '', $createSQL);
            }

            $sql .= $createSQL . ";\n";

            return [
                'success' => true,
                'sql' => $sql,
                'filename' => $tableName . '_structure.sql'
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Export failed: ' . $e->getMessage()
            ];
        }
    }
}
