<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>Gate 6: The Mock Archive</title>
<style>
    /* Copy base styles from dashboard.php */
    * { box-sizing: border-box; }
    body {
        background: radial-gradient(circle at top, #020d18 0%, #000000 70%);
        font-family: 'Consolas', 'Monaco', monospace;
        color: #e0f7ff;
        margin: 0;
        padding: 0;
        min-height: 100vh;
    }
    .panel {
        background: rgba(0,0,0,0.35);
        border: 1px solid #00eaff55;
        box-shadow: 0 0 20px #00eaff44;
        margin: 20px auto;
        padding: 25px;
        border-radius: 12px;
        width: 95%;
        max-width: 1400px;
        animation: fadein 0.6s ease;
    }
    @keyframes fadein {
        from { opacity:0; transform: translateY(20px); }
        to { opacity:1; transform: translateY(0); }
    }
    h2 { text-shadow: 0 0 12px #00eaff; margin: 0 0 10px 0; color: #e0f7ff; }
    
    /* Buttons */
    button {
        background: #00eaff;
        color: #000;
        padding: 8px 16px;
        border-radius: 4px;
        border: none;
        cursor: pointer;
        font-weight: bold;
        box-shadow: 0 0 10px #00eaff88;
        transition: all 0.2s ease;
        font-family: 'Consolas', monospace;
    }
    button:hover { transform: scale(1.05); box-shadow: 0 0 15px #00eaff; }
    button.danger { background: #ff3366; color: #fff; box-shadow: 0 0 10px #ff336688; }
    button.danger:hover { box-shadow: 0 0 15px #ff3366; }
    
    /* Table */
    .table-wrap { overflow-x: auto; margin-top: 15px; }
    table { width: 100%; border-collapse: collapse; }
    th, td { border: 1px solid #00eaff33; padding: 10px; text-align: left; white-space: nowrap; }
    th { background: rgba(0,234,255,0.15); text-shadow: 0 0 6px #00eaff; }
    tr:hover td { background: rgba(0,234,255,0.08); }
    
    /* Modals */
    .modal {
        display: none; position: fixed; z-index: 100; left: 0; top: 0; width: 100%; height: 100%;
        background-color: rgba(0,0,0,0.8); backdrop-filter: blur(2px);
    }
    .modal-content {
        background: #020d18; border: 2px solid #00eaff;
        box-shadow: 0 0 40px #00eaff66;
        margin: 5% auto; padding: 25px; width: 80%; max-width: 800px;
        border-radius: 8px; max-height: 90vh; overflow-y: auto;
    }
    .close { color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer; }
    .close:hover { color: #fff; }
    
    /* Forms */
    .form-group { margin-bottom: 15px; }
    label { display: block; color: #00ffaa; margin-bottom: 5px; }
    input, select, textarea {
        width: 100%; background: #000; color: #00eaff; border: 1px solid #00eaff55;
        padding: 8px; border-radius: 4px; font-family: 'Consolas', monospace;
    }
    input:focus, textarea:focus { outline: none; border-color: #00eaff; box-shadow: 0 0 10px #00eaff44; }
    
    .back-link {
        display: inline-block; margin-bottom: 20px; color: #00eaff; text-decoration: none; font-weight: bold;
    }
    .back-link:hover { text-shadow: 0 0 8px #00eaff; }
</style>
</head>
<body>

<div class="panel">
    <a href="dashboard.php" class="back-link">← Return to Observatory</a>
    <div style="display:flex; justify-content:space-between; align-items:center;">
        <h2 id="pageTitle">Mock CRUD: Loading...</h2>
        <button onclick="openModal('createModal')">+ Add New Row</button>
    </div>
    <div id="status"></div>
</div>

<div class="panel">
    <div class="table-wrap">
        <table id="crudTable">
            <thead><tr id="tableHead"></tr></thead>
            <tbody id="tableBody"></tbody>
        </table>
    </div>
</div>

<!-- Create/Edit Modal -->
<div id="createModal" class="modal">
    <div class="modal-content">
        <span class="close" onclick="closeModal('createModal')">&times;</span>
        <h2 id="modalTitle">Add/Edit Row</h2>
        <form id="crudForm" onsubmit="handleSave(event)">
            <input type="hidden" id="editId" value="">
            <div id="formFields"></div>
            <div style="margin-top:20px; text-align:right;">
                <button type="button" onclick="closeModal('createModal')" style="background:#333; color:#fff;">Cancel</button>
                <button type="submit">Save Changes</button>
            </div>
        </form>
    </div>
</div>

<script>
// State
const urlParams = new URLSearchParams(window.location.search);
const db = urlParams.get('db');
const table = urlParams.get('table');
let currentRows = [];
let currentSchema = []; // keys from first row

function init() {
    if(!db || !table) {
        document.body.innerHTML = '<h1 style="text-align:center; margin-top:50px;">Missing DB or Table</h1>';
        return;
    }
    document.getElementById('pageTitle').textContent = `Mock CRUD: [${table}] @ ${db}`;
    loadData();
}

function loadData() {
    fetch(`api/mock_load.php?db=${db}&table=${table}`)
        .then(r => r.json())
        .then(data => {
            if(data.status === 'ok') {
                currentRows = data.rows;
                renderTable();
            } else {
                alert('Error: ' + data.message);
            }
        });
}

function renderTable() {
    const thead = document.getElementById('tableHead');
    const tbody = document.getElementById('tableBody');
    thead.innerHTML = '';
    tbody.innerHTML = '';

    if(currentRows.length === 0) {
        tbody.innerHTML = '<tr><td colspan="100">No data found. Add a row!</td></tr>';
        return;
    }

    // Infer schema from first row (or merge keys from all rows to be safe? First row is usually enough for consistent schema)
    // Ideally we scan all to find all potential keys
    const keys = new Set();
    currentRows.forEach(r => Object.keys(r).forEach(k => keys.add(k)));
    currentSchema = Array.from(keys);

    // Header
    let hHtml = '<th>Actions</th>';
    currentSchema.forEach(k => hHtml += `<th>${k}</th>`);
    thead.innerHTML = hHtml;

    // Body
    currentRows.forEach(row => {
        let tr = document.createElement('tr');
        let html = `
            <td>
                <button onclick='openEdit("${escapeHtml(row.id || '')}")'>Edit</button>
                <button class="danger" onclick='deleteRow("${escapeHtml(row.id || '')}")'>Del</button>
            </td>
        `;
        currentSchema.forEach(k => {
            html += `<td>${escapeHtml(row[k])}</td>`;
        });
        tr.innerHTML = html;
        tbody.appendChild(tr);
    });
}

function openEdit(id) {
    const row = currentRows.find(r => String(r.id) === String(id));
    if(!row) return;
    
    document.getElementById('editId').value = id;
    document.getElementById('modalTitle').textContent = 'Edit Row';
    generateForm(row);
    openModal('createModal');
}

function openModal(id) {
    if(id === 'createModal' && !document.getElementById('editId').value) {
        document.getElementById('modalTitle').textContent = 'Add New Row';
        // Empty object for new row, but use schema
        const emptyRow = {};
        currentSchema.forEach(k => emptyRow[k] = '');
        generateForm(emptyRow);
    }
    document.getElementById(id).style.display = 'block';
}

function closeModal(id) {
    document.getElementById(id).style.display = 'none';
    document.getElementById('editId').value = '';
}

function generateForm(data) {
    const container = document.getElementById('formFields');
    container.innerHTML = '';
    
    currentSchema.forEach(key => {
        // Skip ID if it's auto-generated/hidden usually, but for mock let's show read-only or editable? 
        // Let's make ID readonly
        const isId = (key.toLowerCase() === 'id');
        
        const div = document.createElement('div');
        div.className = 'form-group';
        
        let inputType = 'text';
        // Simple heuristics
        if(key.includes('date')) inputType = 'date';
        if(key.includes('email')) inputType = 'email';
        
        // Value
        const val = data[key] !== undefined ? data[key] : '';

        let inputHtml = '';
        if (val && val.length > 100) {
             inputHtml = `<textarea name="${key}">${escapeHtml(val)}</textarea>`;
        } else {
             inputHtml = `<input type="${inputType}" name="${key}" value="${escapeHtml(val)}" ${isId ? 'readonly style="opacity:0.5"' : ''}>`;
        }

        div.innerHTML = `
            <label>${key}</label>
            ${inputHtml}
        `;
        container.appendChild(div);
    });
}

function handleSave(e) {
    e.preventDefault();
    const formData = new FormData(e.target);
    const data = {};
    formData.forEach((value, key) => data[key] = value);
    
    const id = document.getElementById('editId').value;
    const endpoint = id ? 'api/mock_save.php' : 'api/mock_insert.php';
    
    const payload = {
        db: db,
        table: table,
        data: data
    };
    if(id) payload.id = id;

    fetch(endpoint, {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify(payload)
    })
    .then(r => r.json())
    .then(res => {
        if(res.status === 'ok') {
            currentRows = res.rows;
            renderTable();
            closeModal('createModal');
        } else {
            alert('Error: ' + res.message);
        }
    });
}

function deleteRow(id) {
    if(!confirm('Are you sure? This only deletes from the Mock Archive.')) return;
    
    fetch('api/mock_delete.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({ db, table, id })
    })
    .then(r => r.json())
    .then(res => {
         if(res.status === 'ok') {
            currentRows = res.rows;
            renderTable();
        } else {
            alert('Error: ' + res.message);
        }
    });
}

function escapeHtml(text) {
    if(text === null || text === undefined) return '';
    return String(text).replace(/&/g, "&")
        .replace(/</g, "<")
        .replace(/>/g, ">")
        .replace(/"/g, '"')
        .replace(/'/g, "&#039;");
}

window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = "none";
    }
}

init();
</script>
</body>
</html>
