<?php
/**
 * tabler_ui.php - The Ledger Whisperer Web Interface
 * 
 * Web UI for converting bank statement PDFs to TSV using tabler.php
 * 
 * Protocol: tabler.mother.tongue.protocol.v1
 * Doctrine: Miserable-First → Zoom-Out → Reconcile → Export
 * 
 * @version 1.0.0
 */

// ============================================================================
// BACKEND PROCESSING (Handle form submission)
// ============================================================================

$conversion_result = null;
$conversion_error = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'convert') {
    // Validate uploaded file
    if (!isset($_FILES['pdf_file']) || $_FILES['pdf_file']['error'] !== UPLOAD_ERR_OK) {
        $conversion_error = "No file uploaded or upload error";
    } else {
        $uploaded_file = $_FILES['pdf_file'];
        
        // Validate PDF
        $allowed_extensions = ['pdf'];
        $extension = strtolower(pathinfo($uploaded_file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($extension, $allowed_extensions)) {
            $conversion_error = "Only PDF files are allowed";
        } elseif ($uploaded_file['size'] > 50 * 1024 * 1024) {
            $conversion_error = "File too large (max 50MB)";
        } else {
            // Move uploaded file to temp location
            $temp_dir = sys_get_temp_dir() . '/tabler_ui_' . getmypid();
            if (!is_dir($temp_dir)) {
                mkdir($temp_dir, 0755, true);
            }
            
            $temp_pdf = $temp_dir . '/input.pdf';
            move_uploaded_file($uploaded_file['tmp_name'], $temp_pdf);
            
            // Determine output path
            $output_filename = !empty($_POST['output_filename']) 
                ? $_POST['output_filename'] 
                : pathinfo($uploaded_file['name'], PATHINFO_FILENAME) . '.txt';
            
            $output_path = $temp_dir . '/' . $output_filename;
            $audit_path = !empty($_POST['generate_audit']) ? $temp_dir . '/audit.json' : null;
            
            // Build command
            $cmd = "php " . escapeshellarg(__DIR__ . '/tabler.php');
            $cmd .= " " . escapeshellarg($temp_pdf);
            $cmd .= " " . escapeshellarg($output_path);
            
            if (!empty($_POST['language'])) {
                $cmd .= " --lang=" . escapeshellarg($_POST['language']);
            }
            
            if (!empty($_POST['generate_debug'])) {
                $cmd .= " --debug";
            }
            
            if ($audit_path) {
                $cmd .= " --audit=" . escapeshellarg($audit_path);
            }
            
            // Execute conversion
            $descriptors = [
                0 => ["pipe", "r"],
                1 => ["pipe", "w"],
                2 => ["pipe", "w"],
            ];
            
            $process = proc_open($cmd, $descriptors, $pipes);
            
            if (is_resource($process)) {
                // Close input pipe
                fclose($pipes[0]);
                
                // Read output
                $stdout = stream_get_contents($pipes[1]);
                fclose($pipes[1]);
                
                // Read stderr
                $stderr = stream_get_contents($pipes[2]);
                fclose($pipes[2]);
                
                // Wait for process
                $return_value = proc_close($process);
                
                if ($return_value === 0 && file_exists($output_path)) {
                    // Read output file
                    $output_content = file_get_contents($output_path);
                    $lines = explode("\n", trim($output_content));
                    
                    // Parse TSV
                    $headers = [];
                    $rows = [];
                    
                    foreach ($lines as $index => $line) {
                        $fields = explode("\t", $line);
                        
                        if ($index === 0) {
                            $headers = $fields;
                        } else {
                            $rows[] = $fields;
                        }
                    }
                    
                    $conversion_result = [
                        'input_filename' => $uploaded_file['name'],
                        'output_filename' => $output_filename,
                        'output_path' => $output_path,
                        'row_count' => count($rows),
                        'headers' => $headers,
                        'rows' => $rows,
                        'audit' => ($audit_path && file_exists($audit_path)) 
                            ? json_decode(file_get_contents($audit_path), true) 
                            : null,
                        'stdout' => $stdout,
                        'stderr' => $stderr,
                    ];
                } else {
                    $conversion_error = "Conversion failed (exit code: $return_value)\n\nSTDERR:\n$stderr";
                }
            } else {
                $conversion_error = "Could not execute tabler.php";
            }
            
            // Cleanup temp files
            @unlink($temp_pdf);
            @unlink($output_path);
            if ($audit_path) {
                @unlink($audit_path);
            }
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>⚡ tabler_ui - The Ledger Whisperer</title>

<style>
    * { box-sizing: border-box; }

    body {
        background: radial-gradient(circle at top, #020d18 0%, #000000 70%);
        font-family: 'Consolas', 'Monospace', monospace;
        color: #e0f7ff;
        margin: 0;
        padding: 0;
        min-height: 100vh;
    }

    .panel {
        background: rgba(0,0,0,0.35);
        border: 1px solid #00eaff55;
        box-shadow: 0 0 20px #00eaff44;
        margin: 20px auto;
        padding: 25px;
        border-radius: 12px;
        width: 90%;
        max-width: 1200px;
        animation: fadein 0.6s ease;
    }

    .panel.cathedral {
        border: 2px solid #00ff88;
        box-shadow: 0 0 40px #00ff88aa;
        animation: cathedral-pulse 2s ease-in-out infinite;
    }

    @keyframes fadein {
        from { opacity:0; transform: translateY(20px); }
        to { opacity:1; transform: translateY(0); }
    }

    @keyframes cathedral-pulse {
        0%, 100% { box-shadow: 0 0 40px #00ff88aa; }
        50% { box-shadow: 0 0 60px #00ff88dd, 0 0 80px #00ff8855; }
    }

    h2 {
        text-shadow: 0 0 12px #00eaff;
        font-weight: bold;
        margin-bottom: 12px;
        margin-top: 0;
    }

    h3 {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
        margin-top: 20px;
    }

    /* Input Styling */
    input[type="text"], input[type="number"], input[type="file"], select {
        background: #000;
        color: #00eaff;
        border: 1px solid #00eaff55;
        padding: 8px 12px;
        border-radius: 4px;
        font-family: 'Consolas', monospace;
        transition: all 0.3s ease;
    }

    input[type="file"] {
        padding: 6px;
    }

    input[type="checkbox"] {
        cursor: pointer;
        width: 18px;
        height: 18px;
        accent-color: #00eaff;
    }

    label {
        color: #00ffaa;
        font-weight: bold;
        margin-right: 10px;
    }

    /* Button Styling */
    button {
        background: #00eaff;
        font-size: 16px;
        border-radius: 8px;
        border: none;
        cursor: pointer;
        font-weight: bold;
        box-shadow: 0 0 14px #00eaffcc;
        transition: all 0.2s ease;
        font-family: 'Consolas', monospace;
    }

    button:hover {
        transform: scale(1.05);
        box-shadow: 0 0 22px #00eaff;
    }

    button:active {
        transform: scale(0.98);
    }

    button.convert-btn {
        background: linear-gradient(135deg, #00eaff 0%, #00ff88 100%);
        box-shadow: 0 0 20px #00ff88cc;
    }

    button.convert-btn:hover {
        box-shadow: 0 0 30px #00ff88;
    }

    /* Table Styling */
    .table-wrap {
        overflow-x: auto;
        margin-top: 15px;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 12px;
    }

    th, td {
        border: 1px solid #00eaff33;
        padding: 10px 12px;
        text-align: left;
    }

    th {
        background: rgba(0,234,255,0.15);
        text-shadow: 0 0 6px #00eaff;
        font-weight: bold;
    }

    tr:hover td {
        background: rgba(0,234,255,0.08);
        transition: background 0.2s ease;
    }

    /* Status Messages */
    .status {
        margin-top: 15px;
        padding: 10px;
        border-radius: 6px;
        animation: fadein 0.4s ease;
    }

    .status.success {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
    }

    .status.error {
        color: #ff3366;
        text-shadow: 0 0 12px #ff3366;
        font-weight: bold;
    }

    .status.info {
        color: #00eaff;
        text-shadow: 0 0 8px #00eaff;
    }

    /* Scrollbar Styling */
    ::-webkit-scrollbar {
        width: 10px;
        height: 10px;
    }

    ::-webkit-scrollbar-track {
        background: #000;
    }

    ::-webkit-scrollbar-thumb {
        background: #00eaff55;
        border-radius: 5px;
    }

    .back-link {
        display: inline-block;
        margin-bottom: 20px;
        color: #00eaff;
        text-decoration: none;
        font-weight: bold;
        transition: all 0.3s;
    }
    .back-link:hover {
        text-shadow: 0 0 8px #00eaff;
        transform: translateX(-5px);
    }

    /* Form groups */
    .form-group {
        margin-bottom: 20px;
    }

    .form-row {
        display: flex;
        gap: 20px;
        margin-bottom: 20px;
    }

    .form-col {
        flex: 1;
    }

    /* Options panel */
    .options-panel {
        background: rgba(0,0,0,0.5);
        border: 1px solid #00eaff22;
        border-radius: 8px;
        padding: 15px;
        margin-top: 20px;
    }

    .options-panel h4 {
        color: #00eaff;
        margin-top: 0;
        margin-bottom: 15px;
        font-size: 14px;
    }

    /* Audit panel */
    .audit-panel {
        background: rgba(0,0,0,0.5);
        border: 1px solid #ffaa0022;
        border-radius: 8px;
        padding: 15px;
        margin-top: 20px;
    }

    .audit-panel h4 {
        color: #ffaa00;
        margin-top: 0;
        margin-bottom: 15px;
        font-size: 14px;
    }

    .audit-json {
        background: #000;
        color: #ffaa00;
        border: 1px solid #ffaa0055;
        padding: 15px;
        border-radius: 8px;
        font-family: 'Consolas', monospace;
        font-size: 12px;
        white-space: pre-wrap;
        overflow-x: auto;
        max-height: 400px;
        overflow-y: auto;
    }

    /* Reconciliation status */
    .recon-badge {
        display: inline-block;
        padding: 4px 10px;
        border-radius: 4px;
        font-size: 12px;
        font-weight: bold;
    }

    .recon-badge.valid {
        background: rgba(0,255,136,0.2);
        color: #00ff88;
        border: 1px solid #00ff88;
    }

    .recon-badge.invalid {
        background: rgba(255,51,102,0.2);
        color: #ff3366;
        border: 1px solid #ff3366;
    }

    /* Loading spinner */
    .spinner {
        display: inline-block;
        width: 20px;
        height: 20px;
        border: 2px solid #00eaff;
        border-top-color: transparent;
        border-radius: 50%;
        animation: spin 1s linear infinite;
        vertical-align: middle;
        margin-right: 10px;
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }
</style>

</head>
<body>

<!-- GATE 1: THE ARRIVAL -->
<div class="panel">
    <a href="index.php" class="back-link">&#8592; Return to Threshold</a>
    <h2>&#128666; Gate 1: The Arrival &mdash; Upload Bank Statement PDF</h2>
    <p style="color: #00ffaa; font-size: 14px; margin-bottom: 15px;">
        Upload a bank statement PDF (text-based or scanned) to convert to TSV.<br>
        <span style="color: #00eaff; font-size: 12px;">
            Protocol: <strong>tabler.mother.tongue.protocol.v1</strong> &bull; 
            Doctrine: <strong>Miserable-First &#8594; Zoom-Out &#8594; Reconcile &#8594; Export</strong>
        </span>
    </p>

    <?php if ($conversion_error): ?>
    <div class="status error">
        &#9888; <?php echo htmlspecialchars($conversion_error); ?>
    </div>
    <?php endif; ?>

    <form method="POST" enctype="multipart/form-data" id="uploadForm">
        <input type="hidden" name="action" value="convert" />

        <div class="form-row">
            <div class="form-col">
                <div class="form-group">
                    <label for="pdf_file">Bank Statement (PDF):</label><br>
                    <input id="pdf_file" name="pdf_file" type="file" accept=".pdf" required />
                </div>
            </div>
            <div class="form-col">
                <div class="form-group">
                    <label for="output_filename">Output Filename:</label><br>
                    <input id="output_filename" name="output_filename" type="text" placeholder="statement.txt" />
                    <p style="color: #00ffaa; font-size: 12px; margin-top: 5px;">
                        (defaults to input basename + .txt)
                    </p>
                </div>
            </div>
        </div>

        <div class="options-panel">
            <h4>&#9894; Conversion Options</h4>
            
            <div class="form-row">
                <div class="form-col">
                    <label for="language">Language:</label>
                    <select id="language" name="language">
                        <option value="es">Spanish (es)</option>
                        <option value="en">English (en)</option>
                        <option value="fr">French (fr)</option>
                        <option value="pt">Portuguese (pt)</option>
                    </select>
                </div>
                <div class="form-col">
                    <label style="cursor: pointer;">
                        <input type="checkbox" id="generate_debug" name="generate_debug" value="1" />
                        Generate Debug Overlay
                    </label>
                </div>
                <div class="form-col">
                    <label style="cursor: pointer;">
                        <input type="checkbox" id="generate_audit" name="generate_audit" value="1" checked />
                        Generate Audit Log
                    </label>
                </div>
            </div>
        </div>

        <button type="submit" class="convert-btn" onclick="showConverting()">
            &#9886; Convert to TSV
        </button>

        <div id="converting" class="status info" style="display: none;">
            <span class="spinner"></span>
            Converting PDF to TSV... (this may take a moment for scanned PDFs)
        </div>
    </form>
</div>

<!-- PREVIEW PANEL -->
<?php if ($conversion_result): ?>
<div class="panel cathedral">
    <h2>&#10024; Gate 4: The Cathedral Moment &mdash; Conversion Complete</h2>
    
    <div class="status success">
        &#10004; Converted <?php echo htmlspecialchars($conversion_result['input_filename']); ?>
        &#8592; <?php echo htmlspecialchars($conversion_result['output_filename']); ?>
        (<?php echo $conversion_result['row_count']; ?> rows)
    </div>

    <h3>Converted Data (Preview)</h3>
    <div class="table-wrap">
        <table>
            <thead>
                <tr>
                    <?php foreach ($conversion_result['headers'] as $header): ?>
                    <th><?php echo htmlspecialchars($header); ?></th>
                    <?php endforeach; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach (array_slice($conversion_result['rows'], 0, 20) as $row): ?>
                <tr>
                    <?php foreach ($row as $cell): ?>
                    <td><?php echo htmlspecialchars($cell); ?></td>
                    <?php endforeach; ?>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php if (count($conversion_result['rows']) > 20): ?>
    <p style="color: #00eaff; margin-top: 10px;">
        ... and <?php echo count($conversion_result['rows']) - 20; ?> more rows
    </p>
    <?php endif; ?>

    <h3>Reconciliation Status</h3>
    <?php
    $is_valid = empty($conversion_result['audit']['reconciliation_failures']);
    ?>
    <div class="status <?php echo $is_valid ? 'success' : 'error'; ?>">
        <span class="recon-badge <?php echo $is_valid ? 'valid' : 'invalid'; ?>">
            <?php echo $is_valid ? '&#10004; LEDGER VALID' : '&#9888; RECONCILIATION ISSUES'; ?>
        </span>
        <?php if (!$is_valid && !empty($conversion_result['audit']['reconciliation_failures'])): ?>
        <br>
        <span style="font-size: 12px; margin-top: 10px; display: block;">
            First failure at row <?php echo $conversion_result['audit']['reconciliation_failures'][0]['row'] + 1; ?>
        </span>
        <?php endif; ?>
    </div>

    <?php if (!empty($conversion_result['audit'])): ?>
    <div class="audit-panel">
        <h4>&#9886; Audit Log (PII Redacted)</h4>
        <div class="audit-json"><?php echo htmlspecialchars(json_encode($conversion_result['audit'], JSON_PRETTY_PRINT)); ?></div>
    </div>
    <?php endif; ?>

    <h3>Download</h3>
    <p style="color: #00ffaa;">
        <a href="#" onclick="downloadOutput(); return false;" style="color: #00eaff;">
            &#8595; Download <?php echo htmlspecialchars($conversion_result['output_filename']); ?>
        </a>
        (<?php echo number_format(strlen(implode("\n", array_map(function($r) { return implode("\t", $r); }, $conversion_result['rows'])))); ?> bytes)
    </p>
</div>
<?php endif; ?>

<script>
    /**
     * Show converting status
     */
    function showConverting() {
        var converting = document.getElementById('converting');
        if (converting) {
            converting.style.display = 'block';
        }
    }

    /**
     * Download output as file
     */
    function downloadOutput() {
        <?php if ($conversion_result): ?>
        var rows = <?php echo json_encode($conversion_result['rows']); ?>;
        var headers = <?php echo json_encode($conversion_result['headers']); ?>;
        var filename = '<?php echo htmlspecialchars($conversion_result['output_filename']); ?>';
        
        // Build TSV content
        var tsv = headers.join('\t') + '\n';
        tsv += rows.map(function(row) {
            return row.join('\t');
        }).join('\n');
        
        // Create blob and download
        var blob = new Blob([tsv], { type: 'text/tab-separated-values' });
        var url = URL.createObjectURL(blob);
        var a = document.createElement('a');
        a.href = url;
        a.download = filename;
        a.click();
        URL.revokeObjectURL(url);
        <?php endif; ?>
    }

    /**
     * Escape HTML
     */
    function escapeHtml(text) {
        if (text === null || text === undefined) return '';
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
</script>

</body>
</html>
