<?php
/**
 * Test: Table Name Autocomplete System
 * Tests the API and verifies functionality
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/ErrorHandler.php';
require_once __DIR__ . '/lib/ImportLogger.php';

echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║         TABLE NAME AUTOCOMPLETE SYSTEM - TEST SUITE                          ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n\n";

// Test 1: getTableNames API returns valid JSON
echo "TEST 1: API Endpoint Response\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$context = stream_context_create(['http' => ['timeout' => 5]]);
$response = @file_get_contents('http://localhost/api/getTableNames.php', false, $context);

if ($response === false) {
    // Try via PHP directly
    $_GET = [];
    ob_start();
    include __DIR__ . '/api/getTableNames.php';
    $response = ob_get_clean();
}

$data = json_decode($response, true);
$test1 = isset($data['status']) && $data['status'] === 'ok';
echo sprintf("  Valid JSON response: %s\n", $test1 ? '✅ PASS' : '❌ FAIL');
echo sprintf("  Has 'table_names' key: %s\n", isset($data['table_names']) ? '✅ PASS' : '❌ FAIL');
echo sprintf("  Has 'count' key: %s\n", isset($data['count']) ? '✅ PASS' : '❌ FAIL');
echo "  Response: " . substr($response, 0, 100) . (strlen($response) > 100 ? '...' : '') . "\n\n";

// Test 2: ImportLogger::getRecentTableNames
echo "TEST 2: ImportLogger::getRecentTableNames()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$tableNames = ImportLogger::getRecentTableNames(null, 10);
$test2 = is_array($tableNames);
echo sprintf("  Returns array: %s\n", $test2 ? '✅ PASS' : '❌ FAIL');
echo sprintf("  Array contents: " . (empty($tableNames) ? 'empty (expected for new install)' : 'has data') . "\n\n");

// Test 3: Search filter
echo "TEST 3: Search Filter\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$filtered = ImportLogger::getRecentTableNames('test', 10);
$test3 = is_array($filtered);
echo sprintf("  Returns array: %s\n", $test3 ? '✅ PASS' : '❌ FAIL');
echo "  Search for 'test': " . count($filtered) . " results\n\n";

// Test 4: TableNameMemory.js exists
echo "TEST 4: Client-side Files\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$jsExists = file_exists(__DIR__ . '/lib/TableNameMemory.js');
echo sprintf("  TableNameMemory.js exists: %s\n", $jsExists ? '✅ PASS' : '❌ FAIL');

// Test 5: arrival.php has autocomplete UI
echo "\nTEST 5: arrival.php Integration\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$arrivalContent = file_get_contents(__DIR__ . '/arrival.php');
$hasDropdown = strpos($arrivalContent, 'tableNameDropdown') !== false;
$hasScript = strpos($arrivalContent, 'TableNameMemory.js') !== false;
$hasController = strpos($arrivalContent, 'TableName Autocomplete') !== false || strpos($arrivalContent, 'function onKeydown') !== false;

echo sprintf("  Has dropdown UI: %s\n", $hasDropdown ? '✅ PASS' : '❌ FAIL');
echo sprintf("  Includes TableNameMemory.js: %s\n", $hasScript ? '✅ PASS' : '❌ FAIL');
echo sprintf("  Has autocomplete controller: %s\n", $hasController ? '✅ PASS' : '❌ FAIL');

// Summary
echo "\n╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                              SUMMARY                                         ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";

$passed = ($test1 ? 1 : 0) + ($test2 ? 1 : 0) + ($test3 ? 1 : 0) + ($jsExists ? 1 : 0) + ($hasDropdown ? 1 : 0) + ($hasScript ? 1 : 0);
$total = 7;
echo sprintf("  Passed: %d/%d\n", $passed, $total);

if ($passed >= 5) {
    echo "\n  🎉 Autocomplete system is ready!\n";
    echo "  - Type in the Destination Table Name field\n";
    echo "  - Press ↓ or click ▼ to open dropdown\n";
    echo "  - Use ↑↓ to navigate, Enter to select\n";
    echo "  - Most-used tables appear first\n";
} else {
    echo "\n  ⚠️  Some tests failed. Check the output above.\n";
}

echo "\n";
