<?php
/**
 * Test UUID Format in Real Database
 * Creates a temporary table, inserts UUIDs, and verifies they work correctly
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/DatabaseHelper.php';
require_once __DIR__ . '/lib/ErrorHandler.php';

echo "========================================\n";
echo "UUID Database Integration Test\n";
echo "========================================\n\n";

try {
    $db = DatabaseHelper::getInstance();
    $db->createAndSelectDatabase('importer_db');
    $conn = $db->getConnection();

    // Create temporary test table
    echo "1. Creating test table...\n";
    $createTable = "
        CREATE TEMPORARY TABLE test_uuid_format (
            test_id CHAR(36) PRIMARY KEY COMMENT 'UUID primary key',
            test_name VARCHAR(100) NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ";

    if (!$conn->query($createTable)) {
        throw new Exception("Failed to create test table: " . $conn->error);
    }
    echo "   ✅ Test table created\n\n";

    // Generate and insert 5 test records
    echo "2. Inserting test records with ia_guid format UUIDs...\n";
    $insertedUUIDs = [];

    for ($i = 1; $i <= 5; $i++) {
        $uuid = $db->generateUUID();
        $name = "Test Record #$i";

        $stmt = $conn->prepare("INSERT INTO test_uuid_format (test_id, test_name) VALUES (?, ?)");
        if (!$stmt) {
            throw new Exception("Failed to prepare statement: " . $conn->error);
        }

        $stmt->bind_param('ss', $uuid, $name);

        if (!$stmt->execute()) {
            throw new Exception("Failed to insert record: " . $stmt->error);
        }

        $insertedUUIDs[] = $uuid;
        echo "   ✅ Inserted: $uuid | $name\n";
    }

    echo "\n3. Verifying records can be retrieved...\n";
    $result = $conn->query("SELECT test_id, test_name, LENGTH(test_id) as id_length FROM test_uuid_format ORDER BY created_at");

    if (!$result) {
        throw new Exception("Failed to select records: " . $conn->error);
    }

    $rowCount = 0;
    while ($row = $result->fetch_assoc()) {
        $rowCount++;
        $retrievedUUID = trim($row['test_id']); // Trim trailing spaces
        $name = $row['test_name'];
        $storedLength = $row['id_length'];

        echo "   ✅ Retrieved: $retrievedUUID | $name\n";
        echo "      Stored length: $storedLength chars (CHAR(36) with padding)\n";
        echo "      Trimmed length: " . strlen($retrievedUUID) . " chars\n";
    }

    if ($rowCount !== 5) {
        throw new Exception("Expected 5 records, got $rowCount");
    }

    echo "\n4. Testing WHERE clause with specific UUID...\n";
    $testUUID = $insertedUUIDs[0];
    $stmt = $conn->prepare("SELECT test_name FROM test_uuid_format WHERE test_id = ?");
    $stmt->bind_param('s', $testUUID);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 1) {
        $row = $result->fetch_assoc();
        echo "   ✅ WHERE clause works: Found '{$row['test_name']}' for UUID: $testUUID\n";
    } else {
        throw new Exception("WHERE clause failed: Expected 1 row, got " . $result->num_rows);
    }

    echo "\n5. Testing JOIN with ia_guid format UUIDs...\n";

    // Create second table for join test
    $createTable2 = "
        CREATE TEMPORARY TABLE test_uuid_details (
            detail_id CHAR(36) PRIMARY KEY,
            test_id CHAR(36) NOT NULL,
            detail_text VARCHAR(200)
        )
    ";
    $conn->query($createTable2);

    // Insert detail record linked to first test record
    $detailUUID = $db->generateUUID();
    $linkedTestUUID = $insertedUUIDs[0];
    $detailText = "Detail for first test record";

    $stmt = $conn->prepare("INSERT INTO test_uuid_details (detail_id, test_id, detail_text) VALUES (?, ?, ?)");
    $stmt->bind_param('sss', $detailUUID, $linkedTestUUID, $detailText);
    $stmt->execute();

    // Test JOIN
    $joinQuery = "
        SELECT t.test_name, d.detail_text
        FROM test_uuid_format t
        JOIN test_uuid_details d ON t.test_id = d.test_id
        WHERE t.test_id = ?
    ";
    $stmt = $conn->prepare($joinQuery);
    $stmt->bind_param('s', $linkedTestUUID);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 1) {
        $row = $result->fetch_assoc();
        echo "   ✅ JOIN works: '{$row['test_name']}' → '{$row['detail_text']}'\n";
    } else {
        throw new Exception("JOIN failed: Expected 1 row, got " . $result->num_rows);
    }

    echo "\n6. Testing UPDATE with ia_guid format UUID...\n";
    $updateUUID = $insertedUUIDs[1];
    $newName = "Updated Test Record #2";

    $stmt = $conn->prepare("UPDATE test_uuid_format SET test_name = ? WHERE test_id = ?");
    $stmt->bind_param('ss', $newName, $updateUUID);
    $stmt->execute();

    if ($stmt->affected_rows === 1) {
        echo "   ✅ UPDATE works: Changed name for UUID: $updateUUID\n";
    } else {
        throw new Exception("UPDATE failed: Expected 1 affected row, got " . $stmt->affected_rows);
    }

    echo "\n7. Testing DELETE with ia_guid format UUID...\n";
    $deleteUUID = $insertedUUIDs[2];

    $stmt = $conn->prepare("DELETE FROM test_uuid_format WHERE test_id = ?");
    $stmt->bind_param('s', $deleteUUID);
    $stmt->execute();

    if ($stmt->affected_rows === 1) {
        echo "   ✅ DELETE works: Removed record for UUID: $deleteUUID\n";
    } else {
        throw new Exception("DELETE failed: Expected 1 affected row, got " . $stmt->affected_rows);
    }

    // Verify count
    $result = $conn->query("SELECT COUNT(*) as count FROM test_uuid_format");
    $row = $result->fetch_assoc();
    if ($row['count'] == 4) {
        echo "   ✅ Verified: 4 records remain (1 deleted)\n";
    }

    echo "\n========================================\n";
    echo "Test Summary\n";
    echo "========================================\n";
    echo "✅ Table creation: PASS\n";
    echo "✅ INSERT operations: PASS (5 records)\n";
    echo "✅ SELECT operations: PASS\n";
    echo "✅ WHERE clause: PASS\n";
    echo "✅ JOIN operations: PASS\n";
    echo "✅ UPDATE operations: PASS\n";
    echo "✅ DELETE operations: PASS\n";
    echo "\n🎉 All database integration tests passed!\n";
    echo "========================================\n\n";

    echo "UUID Format Verification:\n";
    echo "  • Format: ia_guid (reversed segments, no dashes)\n";
    echo "  • Length: 32 characters (hex)\n";
    echo "  • Storage: CHAR(36) with 4 trailing spaces\n";
    echo "  • Comparison: MySQL auto-trims spaces ✅\n";
    echo "  • CRUD Operations: All working correctly ✅\n\n";

} catch (Exception $e) {
    echo "\n❌ Test failed: " . $e->getMessage() . "\n";
    echo "Trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}
