<?php
/**
 * Test UUID Generation - ia_guid() Format
 * Verifies that generateUUID() produces the correct reversed format
 */

require_once __DIR__ . '/lib/DatabaseHelper.php';

echo "========================================\n";
echo "UUID Format Test (ia_guid style)\n";
echo "========================================\n\n";

// Create DatabaseHelper instance
$db = DatabaseHelper::getInstance();

echo "Generating 5 test UUIDs:\n\n";

for ($i = 1; $i <= 5; $i++) {
    $uuid = $db->generateUUID();

    echo "UUID #$i:\n";
    echo "  Value: $uuid\n";
    echo "  Length: " . strlen($uuid) . " characters\n";

    // Validate format
    $isValid = true;
    $errors = [];

    // Check length (should be 32 hex chars)
    if (strlen($uuid) !== 32) {
        $isValid = false;
        $errors[] = "Length should be 32, got " . strlen($uuid);
    }

    // Check if all characters are hexadecimal
    if (!ctype_xdigit($uuid)) {
        $isValid = false;
        $errors[] = "Contains non-hexadecimal characters";
    }

    // Show validation result
    if ($isValid) {
        echo "  ✅ Valid ia_guid format\n";
    } else {
        echo "  ❌ Invalid format:\n";
        foreach ($errors as $error) {
            echo "     - $error\n";
        }
    }

    // Show segment breakdown (reverse order visualization)
    echo "  Segments (reversed): ";
    echo substr($uuid, 0, 12) . " | ";  // Last segment (eeeeeeeeeeee)
    echo substr($uuid, 12, 4) . " | ";  // dddd
    echo substr($uuid, 16, 4) . " | ";  // cccc
    echo substr($uuid, 20, 4) . " | ";  // bbbb
    echo substr($uuid, 24, 8) . "\n";   // aaaaaaaa

    echo "\n";
}

echo "========================================\n";
echo "Format Comparison\n";
echo "========================================\n\n";

// Generate one UUID and show both formats
$uuid = $db->generateUUID();

// Reconstruct what the standard format would have been
$seg5 = substr($uuid, 0, 12);   // eeeeeeeeeeee
$seg4 = substr($uuid, 12, 4);   // dddd
$seg3 = substr($uuid, 16, 4);   // cccc
$seg2 = substr($uuid, 20, 4);   // bbbb
$seg1 = substr($uuid, 24, 8);   // aaaaaaaa

$standardFormat = substr($seg1, 0, 8) . '-' .
                  substr($seg2, 0, 4) . '-' .
                  substr($seg3, 0, 4) . '-' .
                  substr($seg4, 0, 4) . '-' .
                  substr($seg5, 0, 12);

echo "Standard UUID format (hypothetical):\n";
echo "  $standardFormat\n";
echo "  aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee\n\n";

echo "ia_guid format (actual output):\n";
echo "  $uuid\n";
echo "  eeeeeeeeeeeeddddccccbbbbaaaaaaaa\n\n";

echo "Transformation:\n";
echo "  segments[4] . segments[3] . segments[2] . segments[1] . segments[0]\n";
echo "  ✅ Reversed order, no dashes\n\n";

echo "========================================\n";
echo "Database Compatibility Test\n";
echo "========================================\n\n";

echo "Column Type: CHAR(36)\n";
echo "UUID Length: " . strlen($uuid) . " characters\n";
echo "Fits in CHAR(36)? " . (strlen($uuid) <= 36 ? "✅ Yes" : "❌ No") . "\n";
echo "Note: CHAR(36) will have 4 trailing spaces (MySQL auto-trims on comparison)\n\n";

echo "========================================\n";
echo "🎉 All tests completed!\n";
echo "========================================\n";
