<?php
/**
 * Real-World Test: agoda_transaction Schema Detection
 */

require_once __DIR__ . '/../lib/DataCleaner.php';
require_once __DIR__ . '/../lib/SchemaDetector.php';

echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║           REAL-WORLD TEST: agoda_transaction Schema Detection               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
echo "\n";

// Actual column names from agoda_transaction
$agodaColumns = [
    'bookingidexternal_reference_id' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'RULE 1: ID field'],
    'booking_status' => ['expected_type' => 'VARCHAR', 'expected_length' => 50, 'reason' => 'RULE 7: Status field'],
    'staydatefrom' => ['expected_type' => 'DATE', 'expected_length' => null, 'reason' => 'RULE 4: Date field (NEW!)'],
    'staydateto' => ['expected_type' => 'DATE', 'expected_length' => null, 'reason' => 'RULE 4: Date field (NEW!)'],
    'bookeddate' => ['expected_type' => 'DATE', 'expected_length' => null, 'reason' => 'RULE 4: Date field (NEW!)'],
    'customer_name' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'customer_nationality' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'roomtype' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'special_request' => ['expected_type' => 'TEXT', 'expected_length' => null, 'reason' => 'Fallback (long text)'],
    'no_of_night' => ['expected_type' => 'INT', 'expected_length' => null, 'reason' => 'RULE 10: Count field'],
    'no_of_adult' => ['expected_type' => 'INT', 'expected_length' => null, 'reason' => 'RULE 10: Count field'],
    'no_of_children' => ['expected_type' => 'INT', 'expected_length' => null, 'reason' => 'RULE 10: Count field'],
    'no_of_extra_bed' => ['expected_type' => 'INT', 'expected_length' => null, 'reason' => 'RULE 10: Count field'],
    'channel' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'rateplan' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (should NOT match date!)'],
    'promotiontype' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'cancellationpolicycode' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'cancellationpolicydescription' => ['expected_type' => 'TEXT', 'expected_length' => null, 'reason' => 'RULE 9: Description field'],
    'acknowledged' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'total_inclusive_rate' => ['expected_type' => 'DECIMAL', 'expected_length' => '10,2', 'reason' => 'RULE 3: Total field'],
    'currency' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'paymentmodel' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'referencesellinclusive' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (no pattern)'],
    'commission' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'Fallback (should NOT match date!)'],
];

$passed = 0;
$failed = 0;
$criticalPassed = 0;
$criticalFailed = 0;

foreach ($agodaColumns as $columnName => $expected) {
    // Create dummy rows with realistic values
    $headers = [$columnName];
    $rows = [
        [$columnName => 'sample_value_1'],
        [$columnName => 'sample_value_2'],
        [$columnName => 'sample_value_3']
    ];

    $schema = SchemaDetector::detectSchema($rows, $headers);
    $result = $schema[0];

    $actualType = $result['type'];
    $actualLength = $result['length'];

    $typeMatch = ($actualType === $expected['expected_type']);
    $lengthMatch = ($actualLength == $expected['expected_length']);

    $isCritical = in_array($columnName, ['staydatefrom', 'staydateto', 'bookeddate', 'rateplan', 'commission']);

    if ($typeMatch && $lengthMatch) {
        $status = "✅ PASS";
        $passed++;
        if ($isCritical) $criticalPassed++;
    } else {
        $status = "❌ FAIL";
        $failed++;
        if ($isCritical) $criticalFailed++;
    }

    $criticalMarker = $isCritical ? '★' : ' ';

    printf("%s %s  %-35s → %-15s (Expected: %-15s)\n",
        $criticalMarker,
        $status,
        $columnName,
        $actualType . ($actualLength ? "($actualLength)" : ""),
        $expected['expected_type'] . ($expected['expected_length'] ? "({$expected['expected_length']})" : "")
    );

    if (!$typeMatch || !$lengthMatch) {
        echo "      Reason: " . $expected['reason'] . "\n";
    }
}

echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                              TEST SUMMARY                                    ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
echo "\n";
printf("Total Tests: %d\n", $passed + $failed);
printf("✅ Passed: %d\n", $passed);
printf("❌ Failed: %d\n", $failed);
echo "\n";
printf("★ Critical Tests (date fields): %d/%d PASSED\n", $criticalPassed, $criticalPassed + $criticalFailed);
echo "\n";

if ($criticalFailed === 0) {
    echo "🎉 ALL CRITICAL DATE FIELDS DETECTED CORRECTLY!\n";
    echo "\n";
    echo "   BEFORE (value-based detection):\n";
    echo "   - staydatefrom: VARCHAR(50) ❌ → Can't use SQL date functions!\n";
    echo "   - staydateto: VARCHAR(50) ❌ → Can't use SQL date functions!\n";
    echo "   - bookeddate: VARCHAR(50) ❌ → Can't use SQL date functions!\n";
    echo "\n";
    echo "   AFTER (name-based detection):\n";
    echo "   - staydatefrom: DATE ✅ → Enables DATEDIFF, DATE_ADD, etc.!\n";
    echo "   - staydateto: DATE ✅ → Enables DATEDIFF, DATE_ADD, etc.!\n";
    echo "   - bookeddate: DATE ✅ → Enables DATEDIFF, DATE_ADD, etc.!\n";
    echo "\n";
    exit(0);
} else {
    echo "⚠️  CRITICAL TESTS FAILED - Please review the results above.\n";
    echo "\n";
    exit(1);
}
