<?php
/**
 * Test Suite: Column‑Removal via `remove` Flag
 *
 * Verifies that a column marked with `remove` => true is omitted from
 * table creation and row insertion.
 *
 * Steps:
 *   1. Create a temporary database.
 *   2. Send a JSON payload to `insert.php` via CLI (suppressing PHP warnings).
 *   3. Assert that the column does not exist in the created table.
 *   4. Verify that inserted data is present for the retained column.
 *   5. Clean up the temporary database.
 *
 * Exit status: 0 on success, non‑zero on failure.
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../lib/ErrorHandler.php';
require_once __DIR__ . '/../lib/DatabaseHelper.php';
require_once __DIR__ . '/../lib/DataCleaner.php';
require_once __DIR__ . '/../lib/ImportLogger.php';
require_once __DIR__ . '/../lib/ProgressTracker.php';

// Helper to output test result and exit
function testResult(string $msg, bool $ok): void {
    echo ($ok ? "✅ PASS: " : "❌ FAIL: ") . $msg . PHP_EOL;
    if (!$ok) {
        exit(1);
    }
}

// 1. Create a temporary database
$timestamp = time();
$dbName    = "test_remove_col_{$timestamp}";
$tableName = "test_table_{$timestamp}";
$db        = DatabaseHelper::getInstance();
$db->createAndSelectDatabase($dbName);
$conn = $db->getConnection();

testResult("Temporary database `$dbName` created and selected", $conn !== null);

// 2. Build JSON payload
$schema = [
    [
        'name' => 'col_to_keep',
        'type' => 'VARCHAR',
        'length' => '50',
        'nullable' => true,
        'indexed' => false,
        'comment' => '',
        // No `remove` flag – this column should be created
    ],
    [
        'name' => 'col_to_remove',
        'type' => 'INT',
        'length' => '',
        'nullable' => false,
        'indexed' => false,
        'comment' => '',
        'remove' => true   // <-- Column to be ignored
    ]
];

$rows = [
    // Provide a value only for the column that remains
    ['col_to_keep' => 'kept value']
];

$payload = [
    'databaseName' => $dbName,
    'tableName'    => $tableName,
    'schema'       => $schema,
    'rows'         => $rows,
    'skipTableRecreation' => false
];

// 2b. Execute insert.php via CLI, suppressing warnings
$json = json_encode($payload);
$cmd  = sprintf(
    'echo %s | php -d display_errors=0 -d log_errors=0 %s',
    escapeshellarg($json),
    escapeshellarg(__DIR__ . '/../insert.php')
);
exec($cmd, $outputLines, $returnVar);
$response = implode("\n", $outputLines);
$decoded  = json_decode($response, true);

testResult("Insert script executed (exit code $returnVar)", $returnVar === 0);
testResult("Response is valid JSON", $decoded !== null && is_array($decoded));
testResult("Insert reported success", isset($decoded['status']) && $decoded['status'] === 'ok');
testResult("Inserted count is 1", $decoded['inserted'] === 1);

// 3. Verify table structure – removed column must not exist
$colsResult = $conn->query("SHOW COLUMNS FROM `$tableName`");
$columns   = [];
while ($row = $colsResult->fetch_assoc()) {
    $columns[] = $row['Field'];
}
$hasKept    = in_array('col_to_keep', $columns);
$hasRemoved = in_array('col_to_remove', $columns);
testResult("Table contains column `col_to_keep`", $hasKept);
testResult("Table does NOT contain column `col_to_remove`", !$hasRemoved);

// 4. Verify inserted data
$rowResult = $conn->query("SELECT `col_to_keep` FROM `$tableName` LIMIT 1");
$rowData   = $rowResult->fetch_assoc();
testResult("Inserted row value matches", $rowData['col_to_keep'] === 'kept value');

// 5. Cleanup
$conn->query("DROP DATABASE `$dbName`");
testResult("Temporary database `$dbName` dropped", true);

echo "All column‑removal tests passed." . PHP_EOL;
exit(0);
?>