<?php
/**
 * Test Suite: Intelligent Schema Detector
 *
 * Tests name-based pattern detection for the SchemaDetector class
 * Verifies that column names are correctly mapped to appropriate data types
 */

require_once __DIR__ . '/../lib/DataCleaner.php';
require_once __DIR__ . '/../lib/SchemaDetector.php';

class IntelligentSchemaDetectorTest {
    private $passed = 0;
    private $failed = 0;
    private $tests = [];

    /**
     * Test a single column name detection
     */
    private function testColumn($columnName, $expectedType, $expectedLength, $testDescription) {
        // Create dummy data sized appropriately for the expected field type
        // This prevents the hybrid detector from overriding due to data compatibility issues

        // For phone fields (VARCHAR(20)), use short values
        if (preg_match('/phone|telefono|celular/i', $columnName)) {
            $dummyValues = ['+1234567890', '+9876543210', '+5551234567'];
        }
        // For status/boolean fields (VARCHAR(50)), use short values
        elseif (preg_match('/is_|has_|status|estado|activo/i', $columnName)) {
            $dummyValues = ['Active', 'Inactive', 'Pending'];
        }
        // For other fields, use longer realistic values
        else {
            $dummyValues = [
                'This is a realistic sample value that is long enough to trigger VARCHAR(255)',
                'Another realistic sample value with sufficient length for testing',
                'Third realistic sample value to ensure proper length detection'
            ];
        }

        // Call SchemaDetector with minimal data
        $headers = [$columnName];
        $rows = [
            [$columnName => $dummyValues[0]],
            [$columnName => $dummyValues[1]],
            [$columnName => isset($dummyValues[2]) ? $dummyValues[2] : $dummyValues[0]]
        ];

        $schema = SchemaDetector::detectSchema($rows, $headers);

        $result = $schema[0];
        $actualType = $result['type'];
        $actualLength = $result['length'];

        // Compare results
        $typeMatch = ($actualType === $expectedType);
        $lengthMatch = ($actualLength == $expectedLength); // Use == for loose comparison (null == null, '10,2' == '10,2')

        $passed = $typeMatch && $lengthMatch;

        if ($passed) {
            $this->passed++;
            $status = "✅ PASS";
        } else {
            $this->failed++;
            $status = "❌ FAIL";
        }

        $this->tests[] = [
            'status' => $status,
            'description' => $testDescription,
            'column' => $columnName,
            'expected' => "$expectedType" . ($expectedLength ? "($expectedLength)" : ""),
            'actual' => "$actualType" . ($actualLength ? "($actualLength)" : ""),
            'passed' => $passed
        ];

        return $passed;
    }

    /**
     * Run all tests
     */
    public function runTests() {
        echo "\n";
        echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
        echo "║         INTELLIGENT SCHEMA DETECTOR TEST SUITE                              ║\n";
        echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
        echo "\n";

        // RULE 1: ID Fields → VARCHAR(100)
        echo "🧪 RULE 1: ID Fields → VARCHAR(100)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('property_id', 'VARCHAR', 100, 'Property ID field');
        $this->testColumn('user_id', 'VARCHAR', 100, 'User ID field');
        $this->testColumn('codigo', 'VARCHAR', 100, 'Spanish code field');
        $this->testColumn('transaction_code', 'VARCHAR', 100, 'Transaction code field');
        $this->testColumn('folio', 'VARCHAR', 100, 'Folio number field');
        echo "\n";

        // RULE 2: High-Precision Money (Exchange Rates) → DECIMAL(10,4)
        echo "🧪 RULE 2: High-Precision Money (Exchange Rates) → DECIMAL(10,4)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('exchange_rate', 'DECIMAL', '10,4', 'Exchange rate field');
        $this->testColumn('interchange', 'DECIMAL', '10,4', 'Interchange rate field');
        $this->testColumn('tipo_cambio', 'DECIMAL', '10,4', 'Spanish exchange rate field');
        $this->testColumn('conversion_rate', 'DECIMAL', '10,4', 'Conversion rate field');
        echo "\n";

        // RULE 3: Standard Money Fields → DECIMAL(10,2)
        echo "🧪 RULE 3: Standard Money Fields → DECIMAL(10,2)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('total_amount', 'DECIMAL', '10,2', 'Total amount field');
        $this->testColumn('commission_fee', 'DECIMAL', '10,2', 'Commission fee field');
        $this->testColumn('guest_payment', 'DECIMAL', '10,2', 'Guest payment field');
        $this->testColumn('platform_commission', 'DECIMAL', '10,2', 'Platform commission field');
        $this->testColumn('mxn_convert', 'DECIMAL', '10,2', 'MXN convert field');
        $this->testColumn('usd_amount', 'DECIMAL', '10,2', 'USD amount field');
        $this->testColumn('precio', 'DECIMAL', '10,2', 'Spanish price field');
        echo "\n";

        // RULE 4: Date Fields → DATE
        echo "🧪 RULE 4: Date Fields → DATE\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('check_in', 'DATE', null, 'Check-in date field');
        $this->testColumn('check_out', 'DATE', null, 'Check-out date field');
        $this->testColumn('created_date', 'DATE', null, 'Created date field');
        $this->testColumn('fecha', 'DATE', null, 'Spanish date field');
        $this->testColumn('created_at', 'DATE', null, 'Created timestamp field');
        $this->testColumn('updated_at', 'DATE', null, 'Updated timestamp field');
        echo "\n";

        // RULE 5: Email Fields → VARCHAR(255)
        echo "🧪 RULE 5: Email Fields → VARCHAR(255)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('email', 'VARCHAR', 255, 'Email field');
        $this->testColumn('guest_email', 'VARCHAR', 255, 'Guest email field');
        $this->testColumn('correo', 'VARCHAR', 255, 'Spanish email field');
        echo "\n";

        // RULE 6: Phone Fields → VARCHAR(20)
        echo "🧪 RULE 6: Phone Fields → VARCHAR(20)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('phone', 'VARCHAR', 20, 'Phone field');
        $this->testColumn('telefono', 'VARCHAR', 20, 'Spanish phone field');
        $this->testColumn('guest_phone', 'VARCHAR', 20, 'Guest phone field');
        $this->testColumn('celular', 'VARCHAR', 20, 'Spanish mobile field');
        echo "\n";

        // RULE 7: Boolean/Status Fields → VARCHAR(50)
        echo "🧪 RULE 7: Boolean/Status Fields → VARCHAR(50)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('is_active', 'VARCHAR', 50, 'Is active field');
        $this->testColumn('has_permission', 'VARCHAR', 50, 'Has permission field');
        $this->testColumn('booking_status', 'VARCHAR', 50, 'Booking status field');
        $this->testColumn('booking_estado', 'VARCHAR', 50, 'Spanish status field (with prefix)');
        $this->testColumn('activo', 'VARCHAR', 50, 'Spanish active field');
        echo "\n";

        // RULE 8: URL Fields → VARCHAR(500)
        echo "🧪 RULE 8: URL Fields → VARCHAR(500)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('listing_url', 'VARCHAR', 500, 'Listing URL field');
        $this->testColumn('website_link', 'VARCHAR', 500, 'Website link field');
        $this->testColumn('website', 'VARCHAR', 500, 'Website field');
        echo "\n";

        // RULE 9: Description/Notes Fields → TEXT
        echo "🧪 RULE 9: Description/Notes Fields → TEXT\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('property_description', 'TEXT', null, 'Property description field');
        $this->testColumn('guest_notes', 'TEXT', null, 'Guest notes field');
        $this->testColumn('comentarios', 'TEXT', null, 'Spanish comments field');
        $this->testColumn('descripcion', 'TEXT', null, 'Spanish description field');
        echo "\n";

        // RULE 10: Quantity/Count Fields → INT
        echo "🧪 RULE 10: Quantity/Count Fields → INT\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('guest_count', 'INT', null, 'Guest count field');
        $this->testColumn('booking_quantity', 'INT', null, 'Booking quantity field');
        $this->testColumn('nights_qty', 'INT', null, 'Nights quantity field');
        $this->testColumn('cantidad', 'INT', null, 'Spanish quantity field');
        echo "\n";

        // EDGE CASES: Fields that should NOT match patterns (fallback to value-based detection)
        echo "🧪 EDGE CASES: Fields with no pattern match (fallback to value-based)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testColumn('random_field', 'VARCHAR', 100, 'Random field with no pattern (should fallback to VARCHAR based on value length)');
        $this->testColumn('guest_name', 'VARCHAR', 100, 'Guest name (should fallback to VARCHAR based on value length)');
        echo "\n";

        // Print detailed results
        echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
        echo "║                              DETAILED RESULTS                                ║\n";
        echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
        echo "\n";

        foreach ($this->tests as $test) {
            if (!$test['passed']) {
                printf("%s %s\n", $test['status'], $test['description']);
                printf("   Column: %s\n", $test['column']);
                printf("   Expected: %s\n", $test['expected']);
                printf("   Actual: %s\n", $test['actual']);
                echo "\n";
            }
        }

        // Print summary
        echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
        echo "║                              TEST SUMMARY                                    ║\n";
        echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
        echo "\n";
        printf("Total Tests: %d\n", $this->passed + $this->failed);
        printf("✅ Passed: %d\n", $this->passed);
        printf("❌ Failed: %d\n", $this->failed);
        echo "\n";

        if ($this->failed === 0) {
            echo "🎉 ALL TESTS PASSED! 🎉\n";
            echo "\n";
            return true;
        } else {
            echo "⚠️  SOME TESTS FAILED - Please review the detailed results above.\n";
            echo "\n";
            return false;
        }
    }
}

// Run the tests
$tester = new IntelligentSchemaDetectorTest();
$allPassed = $tester->runTests();

exit($allPassed ? 0 : 1);
