<?php
/**
 * test_rag_context7.php - Test script for RAG Context7 Integration
 *
 * Tests the Context7 integration features:
 * - Library ID resolution
 * - Knowledge retrieval
 * - Merged context in RagQuery
 *
 * Usage:
 *   php tests/test_rag_context7.php
 */

require_once __DIR__ . '/../lib/RagQuery.php';
require_once __DIR__ . '/../lib/Context7LibraryResolver.php';
require_once __DIR__ . '/../lib/Context7KnowledgeRetriever.php';

class RagContext7Test
{
    private $passed = 0;
    private $failed = 0;
    private $tests = [];

    /**
     * Run all tests
     */
    public function runTests()
    {
        echo "\n";
        echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
        echo "║                    RAG CONTEXT7 INTEGRATION TEST SUITE                     ║\n";
        echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
        echo "\n";

        // Version tests
        $this->testVersionInfo();
        echo "\n";

        // Context7 Library Resolver tests
        $this->testLibraryResolution();
        echo "\n";

        // Context7 Knowledge Retriever tests
        $this->testKnowledgeRetrieval();
        echo "\n";

        // RagQuery Context7 integration tests
        $this->testRagQueryContext7Integration();
        echo "\n";

        // Print summary
        echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
        echo "║                              TEST SUMMARY                                   ║\n";
        echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
        echo "\n";
        printf("Total Tests: %d\n", $this->passed + $this->failed);
        printf("✅ Passed: %d\n", $this->passed);
        printf("❌ Failed: %d\n", $this->failed);
        echo "\n";

        if ($this->failed === 0) {
            echo "🎉 ALL CONTEXT7 INTEGRATION TESTS PASSED! 🎉\n";
            echo "\n";
            return true;
        } else {
            echo "️  SOME TESTS FAILED - Please review the results above.\n";
            echo "\n";
            return false;
        }
    }

    /**
     * Test version info
     */
    private function testVersionInfo()
    {
        echo "🧪 VERSION INFO TESTS\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";

        $this->runTest(
            'RagQuery version info includes context7_enabled',
            function () {
                $version = RagQuery::getVersion();
                return isset($version['context7_enabled']) && $version['context7_enabled'] === true;
            }
        );

        $this->runTest(
            'Context7LibraryResolver has version',
            function () {
                $version = Context7LibraryResolver::getVersion();
                return isset($version['version']);
            }
        );

        $this->runTest(
            'Context7KnowledgeRetriever has version',
            function () {
                $version = Context7KnowledgeRetriever::getVersion();
                return isset($version['version']);
            }
        );
    }

    /**
     * Test library resolution
     */
    private function testLibraryResolution()
    {
        echo "🧪 LIBRARY RESOLUTION TESTS\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";

        $this->runTest(
            'Resolve vercel/next.js',
            function () {
                $id = Context7LibraryResolver::resolve('vercel/next.js');
                return $id === '/vercel/next.js';
            }
        );

        $this->runTest(
            'Resolve phpspreadsheet/phpspreadsheet',
            function () {
                $id = Context7LibraryResolver::resolve('phpspreadsheet/phpspreadsheet');
                return stripos($id, 'phpspreadsheet') !== false;
            }
        );

        $this->runTest(
            'Resolve php/php',
            function () {
                $id = Context7LibraryResolver::resolve('php/php');
                return $id === '/php/php';
            }
        );

        $this->runTest(
            'Resolve unknown library returns constructed ID',
            function () {
                $id = Context7LibraryResolver::resolve('unknown/library');
                return $id === '/unknown/library';
            }
        );
    }

    /**
     * Test knowledge retrieval
     */
    private function testKnowledgeRetrieval()
    {
        echo "🧪 KNOWLEDGE RETRIEVAL TESTS\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";

        $this->runTest(
            'Query PHP documentation',
            function () {
                $docs = Context7KnowledgeRetriever::queryDocs('php/php', 'How to create arrays in PHP?');
                return !empty($docs) && strpos($docs, 'Context7 Documentation') !== false;
            }
        );

        $this->runTest(
            'Query spreadsheet documentation',
            function () {
                $docs = Context7KnowledgeRetriever::queryDocs('phpspreadsheet/phpspreadsheet', 'How to read Excel files?');
                return !empty($docs) && strpos($docs, 'Context7 Documentation') !== false;
            }
        );

        $this->runTest(
            'Query with options',
            function () {
                $docs = Context7KnowledgeRetriever::queryDocs(
                    'mysqli/mysqli',
                    'How to connect to database?',
                    ['maxResults' => 5]
                );
                return !empty($docs);
            }
        );
    }

    /**
     * Test RagQuery Context7 integration
     */
    private function testRagQueryContext7Integration()
    {
        echo "🧪 RAGQUERY CONTEXT7 INTEGRATION TESTS\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";

        $this->runTest(
            'getContext returns local context by default',
            function () {
                $context = RagQuery::getContext('What does the DatabaseHelper class do?', ['useContext7' => false]);
                return !empty($context) && strpos($context, 'Section: Codebase') !== false;
            }
        );

        $this->runTest(
            'getContext includes Context7 when enabled',
            function () {
                $context = RagQuery::getContext('How do I read Excel files in PHP?', ['useContext7' => true]);
                return !empty($context) && strpos($context, 'Context7 Documentation') !== false;
            }
        );

        $this->runTest(
            'getContext merges local and remote context',
            function () {
                $context = RagQuery::getContext('database query sql', ['useContext7' => true]);
                return !empty($context);
            }
        );

        $this->runTest(
            'getContext7Status returns correct structure',
            function () {
                $status = RagQuery::getContext7Status();
                return isset($status['enabled']) && isset($status['library_map']);
            }
        );

        $this->runTest(
            'Context7 library map contains expected libraries',
            function () {
                $status = RagQuery::getContext7Status();
                $libraries = $status['library_map'];
                return isset($libraries['excel']) && isset($libraries['database']);
            }
        );
    }

    /**
     * Run a single test
     */
    private function runTest($description, $testFn)
    {
        try {
            $result = $testFn();
            if ($result) {
                $this->passed++;
                echo "  ✅ $description\n";
            } else {
                $this->failed++;
                echo "  ❌ $description\n";
            }
        } catch (Exception $e) {
            $this->failed++;
            echo "  ❌ $description (Exception: " . $e->getMessage() . ")\n";
        }
    }
}

// Run the tests
$tester = new RagContext7Test();
$allPassed = $tester->runTests();

exit($allPassed ? 0 : 1);
