<?php
/**
 * Real-World Test: casitamx_transaction Schema Detection
 *
 * Tests the intelligent schema detector with actual column names
 * from the user's casitamx_transaction table
 */

require_once __DIR__ . '/../lib/DataCleaner.php';
require_once __DIR__ . '/../lib/SchemaDetector.php';

echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║           REAL-WORLD TEST: casitamx_transaction Schema Detection            ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
echo "\n";

// Simulate real-world column names from casitamx_transaction
$realWorldColumns = [
    'property_id' => ['expected_type' => 'VARCHAR', 'expected_length' => 100, 'reason' => 'RULE 1: ID field'],
    'check_in' => ['expected_type' => 'DATE', 'expected_length' => null, 'reason' => 'RULE 4: Date field'],
    'check_out' => ['expected_type' => 'DATE', 'expected_length' => null, 'reason' => 'RULE 4: Date field'],
    'interchange' => ['expected_type' => 'DECIMAL', 'expected_length' => '10,4', 'reason' => 'RULE 2: High-precision money (exchange rate)'],
    'mxn_convert' => ['expected_type' => 'DECIMAL', 'expected_length' => '10,2', 'reason' => 'RULE 3: Standard money field'],
    'guest_payment' => ['expected_type' => 'DECIMAL', 'expected_length' => '10,2', 'reason' => 'RULE 3: Standard money field'],
    'total_amount' => ['expected_type' => 'DECIMAL', 'expected_length' => '10,2', 'reason' => 'RULE 3: Standard money field'],
    'commission_fee' => ['expected_type' => 'DECIMAL', 'expected_length' => '10,2', 'reason' => 'RULE 3: Standard money field'],
    'platform_commission' => ['expected_type' => 'DECIMAL', 'expected_length' => '10,2', 'reason' => 'RULE 3: Standard money field'],
    'exchange_rate' => ['expected_type' => 'DECIMAL', 'expected_length' => '10,4', 'reason' => 'RULE 2: High-precision money (exchange rate)'],
    'guest_email' => ['expected_type' => 'VARCHAR', 'expected_length' => 255, 'reason' => 'RULE 5: Email field'],
    'guest_phone' => ['expected_type' => 'VARCHAR', 'expected_length' => 20, 'reason' => 'RULE 6: Phone field'],
    'booking_status' => ['expected_type' => 'VARCHAR', 'expected_length' => 50, 'reason' => 'RULE 7: Status field'],
    'listing_url' => ['expected_type' => 'VARCHAR', 'expected_length' => 500, 'reason' => 'RULE 8: URL field'],
    'guest_notes' => ['expected_type' => 'TEXT', 'expected_length' => null, 'reason' => 'RULE 9: Notes field'],
    'guest_count' => ['expected_type' => 'INT', 'expected_length' => null, 'reason' => 'RULE 10: Count field'],
];

$passed = 0;
$failed = 0;

foreach ($realWorldColumns as $columnName => $expected) {
    // Create dummy rows with realistic values
    $headers = [$columnName];
    $rows = [
        [$columnName => 'sample_value_1'],
        [$columnName => 'sample_value_2'],
        [$columnName => 'sample_value_3']
    ];

    $schema = SchemaDetector::detectSchema($rows, $headers);
    $result = $schema[0];

    $actualType = $result['type'];
    $actualLength = $result['length'];

    $typeMatch = ($actualType === $expected['expected_type']);
    $lengthMatch = ($actualLength == $expected['expected_length']);

    if ($typeMatch && $lengthMatch) {
        $status = "✅ PASS";
        $passed++;
    } else {
        $status = "❌ FAIL";
        $failed++;
    }

    printf("%s  %-25s → %-15s (Expected: %-15s) %s\n",
        $status,
        $columnName,
        $actualType . ($actualLength ? "($actualLength)" : ""),
        $expected['expected_type'] . ($expected['expected_length'] ? "({$expected['expected_length']})" : ""),
        $typeMatch && $lengthMatch ? "" : "← " . $expected['reason']
    );
}

echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                              TEST SUMMARY                                    ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
echo "\n";
printf("Total Tests: %d\n", $passed + $failed);
printf("✅ Passed: %d\n", $passed);
printf("❌ Failed: %d\n", $failed);
echo "\n";

if ($failed === 0) {
    echo "🎉 ALL TESTS PASSED! The intelligent schema detector correctly identifies\n";
    echo "   all casitamx_transaction column types based on their names!\n";
    echo "\n";
    echo "   BEFORE (value-based detection):\n";
    echo "   - interchange: DECIMAL(10,2) ❌ → Lost precision!\n";
    echo "   - property_id: INT ❌ → Would truncate alphanumeric IDs!\n";
    echo "\n";
    echo "   AFTER (name-based detection):\n";
    echo "   - interchange: DECIMAL(10,4) ✅ → Preserves precision for exchange rates!\n";
    echo "   - property_id: VARCHAR(100) ✅ → Handles alphanumeric IDs correctly!\n";
    echo "\n";
    exit(0);
} else {
    echo "⚠️  SOME TESTS FAILED - Please review the results above.\n";
    echo "\n";
    exit(1);
}
