<?php
/**
 * Exodus Download API
 *
 * Handles secure download of backup files
 */

require_once __DIR__ . '/../config.php';
require_once 'lib/ExodusEngine.php';

try {
    // Get filename from query parameter
    $filename = $_GET['file'] ?? '';

    if (empty($filename)) {
        throw new Exception('Filename parameter required');
    }

    // Sanitize filename (prevent directory traversal)
    $filename = basename($filename);

    // Verify filename matches exodus pattern
    if (!preg_match('/^exodus_.*\.(tar\.gz|zip)$/', $filename)) {
        throw new Exception('Invalid backup filename format');
    }

    // Build file path
    $backupDir = __DIR__ . '/../backups/';
    $filepath = $backupDir . $filename;

    // Verify file exists
    if (!file_exists($filepath)) {
        throw new Exception('Backup file not found');
    }

    // Determine MIME type
    $ext = pathinfo($filename, PATHINFO_EXTENSION);
    $mimeType = 'application/octet-stream';

    if ($ext === 'gz') {
        $mimeType = 'application/gzip';
    } elseif ($ext === 'zip') {
        $mimeType = 'application/zip';
    }

    // Send headers for download
    header('Content-Type: ' . $mimeType);
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Content-Length: ' . filesize($filepath));
    header('Cache-Control: no-cache, must-revalidate');
    header('Pragma: public');

    // Output file
    readfile($filepath);

    exit;

} catch (Exception $e) {
    // Return error as JSON
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
    exit;
}
