<?php
/**
 * API: Get Last Import Schema
 * Retrieves the most recent successful schema for a given table name
 * Used by arrival.php to pre-populate schema settings from memory
 */

require_once 'config.php';
require_once 'ErrorHandler.php';
require_once 'DatabaseHelper.php';
require_once 'ImportLogger.php';

try {
    // Get input parameters
    $tableName = $_GET['table_name'] ?? null;
    $databaseName = $_GET['database_name'] ?? null;

    if (!$tableName) {
        ErrorHandler::jsonError('Missing required parameter: table_name', 'MISSING_PARAMETER', 400);
    }

    // Initialize database connection
    $db = DatabaseHelper::getInstance();

    // If database name is provided, select it first
    // Otherwise, try to work with the current database
    if ($databaseName) {
        try {
            $db->createAndSelectDatabase($databaseName);
        } catch (Exception $e) {
            // If database doesn't exist yet, return no schema found
            ErrorHandler::jsonSuccess([
                'status' => 'ok',
                'found' => false,
                'message' => 'Database does not exist yet'
            ]);
            exit;
        }
    }

    // Retrieve last successful schema
    $schemaData = ImportLogger::getLastSuccessfulSchema($tableName, $databaseName);

    if ($schemaData === null) {
        ErrorHandler::jsonSuccess([
            'status' => 'ok',
            'found' => false,
            'message' => 'No previous import found for this table'
        ]);
    } else {
        ErrorHandler::jsonSuccess([
            'status' => 'ok',
            'found' => true,
            'schema' => $schemaData['schema'],
            'import_date' => $schemaData['import_date'],
            'import_log_id' => $schemaData['import_log_id'],
            'message' => 'Schema loaded from previous import on ' . $schemaData['import_date']
        ]);
    }

} catch (Exception $e) {
    ErrorHandler::handleException($e);
}
