<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>🌊 Cyberpunk Data Importer</title>

<style>
    * { box-sizing: border-box; }

    body {
        background: radial-gradient(circle at top, #020d18 0%, #000000 70%);
        font-family: 'Consolas', 'Monaco', monospace;
        color: #e0f7ff;
        margin: 0;
        padding: 0;
        min-height: 100vh;
    }

    .panel {
        background: rgba(0,0,0,0.35);
        border: 1px solid #00eaff55;
        box-shadow: 0 0 20px #00eaff44;
        margin: 20px auto;
        padding: 25px;
        border-radius: 12px;
        width: 90%;
        max-width: 1200px;
        animation: fadein 0.6s ease;
    }

    .panel.cathedral {
        border: 2px solid #00eaff;
        box-shadow: 0 0 40px #00eaffaa;
        animation: cathedral-pulse 2s ease-in-out infinite;
    }

    @keyframes fadein {
        from { opacity:0; transform: translateY(20px); }
        to { opacity:1; transform: translateY(0); }
    }

    @keyframes cathedral-pulse {
        0%, 100% { box-shadow: 0 0 40px #00eaffaa; }
        50% { box-shadow: 0 0 60px #00eaffdd, 0 0 80px #00eaff88; }
    }

    h2 {
        text-shadow: 0 0 12px #00eaff;
        font-weight: bold;
        margin-bottom: 12px;
        margin-top: 0;
    }

    h3 {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
        margin-top: 20px;
    }

    /* Input Styling */
    input[type="text"], input[type="number"], input[type="file"], select {
        background: #000;
        color: #00eaff;
        border: 1px solid #00eaff55;
        padding: 8px 12px;
        border-radius: 4px;
        font-family: 'Consolas', monospace;
        transition: all 0.3s ease;
    }

    input[type="text"]:focus, input[type="number"]:focus, select:focus {
        outline: none;
        border-color: #00eaff;
        box-shadow: 0 0 12px #00eaff88;
    }

    input[type="file"] {
        padding: 6px;
    }

    input[type="checkbox"] {
        cursor: pointer;
        width: 18px;
        height: 18px;
        accent-color: #00eaff;
    }

    label {
        color: #00ffaa;
        font-weight: bold;
        margin-right: 10px;
    }

    /* Button Styling */
    button {
        background: #00eaff;
        color: #000;
        padding: 12px 28px;
        font-size: 16px;
        border-radius: 8px;
        border: none;
        cursor: pointer;
        font-weight: bold;
        box-shadow: 0 0 14px #00eaffcc;
        transition: all 0.2s ease;
        font-family: 'Consolas', monospace;
    }

    button:hover {
        transform: scale(1.05);
        box-shadow: 0 0 22px #00eaff;
    }

    button:active {
        transform: scale(0.98);
    }

    /* Table Styling */
    .table-wrap {
        overflow-x: auto;
        margin-top: 15px;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 12px;
    }

    th, td {
        border: 1px solid #00eaff33;
        padding: 10px 12px;
        text-align: left;
    }

    th {
        background: rgba(0,234,255,0.15);
        text-shadow: 0 0 6px #00eaff;
        font-weight: bold;
    }

    tr:hover td {
        background: rgba(0,234,255,0.08);
        transition: background 0.2s ease;
    }

    tbody tr {
        animation: row-pulse 0.3s ease;
    }

    @keyframes row-pulse {
        from { opacity: 0.5; }
        to { opacity: 1; }
    }

    /* SQL Preview */
    #sqlPreview {
        background: #000;
        color: #00ff88;
        border: 1px solid #00eaff77;
        padding: 15px;
        border-radius: 8px;
        font-family: 'Courier New', monospace;
        font-size: 13px;
        white-space: pre-wrap;
        overflow-x: auto;
        box-shadow: inset 0 0 12px #00eaff44;
        max-height: 400px;
        overflow-y: auto;
    }

    /* Status Messages */
    .status {
        margin-top: 15px;
        padding: 10px;
        border-radius: 6px;
        animation: fadein 0.4s ease;
    }

    .status.success {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
    }

    .status.error {
        color: #ff3366;
        text-shadow: 0 0 12px #ff3366;
        font-weight: bold;
    }

    /* Schema Table Specific */
    #schemaTable input[type="text"],
    #schemaTable input[type="number"],
    #schemaTable select {
        width: 100%;
        padding: 6px 8px;
        font-size: 13px;
    }

    .db-table-inputs {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
        margin-bottom: 20px;
    }

    .input-group {
        display: flex;
        flex-direction: column;
        gap: 8px;
    }

    /* Scrollbar Styling */
    ::-webkit-scrollbar {
        width: 10px;
        height: 10px;
    }

    ::-webkit-scrollbar-track {
        background: #000;
    }

    ::-webkit-scrollbar-thumb {
        background: #00eaff55;
        border-radius: 5px;
    }

    ::-webkit-scrollbar-thumb:hover {
        background: #00eaff88;
    }

    .back-link {
        display: inline-block;
        margin-bottom: 20px;
        color: #00eaff;
        text-decoration: none;
        font-weight: bold;
        transition: all 0.3s;
    }
    .back-link:hover {
        text-shadow: 0 0 8px #00eaff;
        transform: translateX(-5px);
    }

    /* Floating Back to Top Button */
    #backToTopBtn {
        position: fixed;
        bottom: 30px;
        right: 30px;
        width: 55px;
        height: 55px;
        background: rgba(0, 234, 255, 0.9);
        color: #000;
        border: 2px solid #00eaff;
        border-radius: 50%;
        font-size: 24px;
        font-weight: bold;
        cursor: pointer;
        box-shadow: 0 0 20px #00eaffcc, 0 0 40px #00eaff66;
        transition: all 0.3s ease;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transform: scale(0.8);
        display: flex;
        align-items: center;
        justify-content: center;
        font-family: 'Consolas', monospace;
        animation: none;
    }

    #backToTopBtn.show {
        opacity: 1;
        visibility: visible;
        transform: scale(1);
    }

    #backToTopBtn:hover {
        background: #00eaff;
        box-shadow: 0 0 30px #00eaff, 0 0 60px #00eaff88;
        transform: scale(1.1) translateY(-3px);
    }

    #backToTopBtn:active {
        transform: scale(0.95);
    }

    /* Cyberpunk pulse animation for the button */
    @keyframes cyber-pulse {
        0%, 100% {
            box-shadow: 0 0 20px #00eaffcc, 0 0 40px #00eaff66;
        }
        50% {
            box-shadow: 0 0 30px #00eaff, 0 0 50px #00eaff88, 0 0 70px #00eaff44;
        }
    }

    #backToTopBtn.show {
        animation: cyber-pulse 2s ease-in-out infinite;
    }
</style>

</head>
<body>

<!-- GATE 1: THE ARRIVAL -->
<div class="panel">
    <a href="index.php" class="back-link">← Return to Threshold</a>
    <h2>🌊 Gate 1: The Arrival — Upload Data File</h2>
    <p style="color: #00ffaa; font-size: 14px; margin-bottom: 15px;">
        Accepted formats: <strong>XLSX</strong> (Excel) or <strong>CSV</strong> (Comma-Separated Values)
        <br>CSV files will be auto-detected for encoding and delimiter type.
    </p>

    <div style="margin-bottom: 15px;">
        <label for="destinationTableName" style="display: block; margin-bottom: 8px;">Destination Table Name:</label>
        <input id="destinationTableName" type="text" placeholder="e.g., productos, clientes, empleados..."
               style="width: 100%; max-width: 400px;" />
        <p style="color: #00ffaa; font-size: 12px; margin-top: 5px; margin-bottom: 0;">
            If this table was previously imported, its schema will be restored automatically.
        </p>
    </div>

    <input id="file" type="file" accept=".xlsx,.csv" />
    <button onclick="uploadXLSX()" style="margin-left: 15px;">Upload & Parse</button>
    <div id="status" class="status"></div>
</div>

<!-- PREVIEW PANEL -->
<div id="preview" class="panel" style="display:none;"></div>

<!-- GATE 3: THE SCHEMA GATE -->
<div id="schemaPanel" class="panel" style="display:none;">
    <h2>🌌 Gate 3: The Imagination — Schema Builder</h2>
    <p style="color: #00ffaa; margin-bottom: 20px;">Shape the structure before birth. Edit column types, lengths, and constraints.</p>

    <!-- Schema Memory Indicator -->
    <div id="schemaMemoryIndicator" style="display:none; background: rgba(0,234,255,0.15); border: 1px solid #00eaff; border-radius: 8px; padding: 12px 16px; margin-bottom: 20px;">
        <strong style="color: #00eaff;">🧠 Schema Restored from Memory</strong><br>
        <span id="schemaMemoryMessage" style="color: #00ffaa; font-size: 13px;"></span><br>
        <div style="margin-top: 12px;">
            <label style="color: #00ffaa; font-size: 13px; cursor: pointer;">
                <input type="checkbox" id="skipTableRecreation" style="margin-right: 8px;">
                Skip table recreation — insert records only (table already exists)
            </label>
        </div>
        <button onclick="resetToAutoDetectedSchema()" style="margin-top: 10px; background: #ff6600; padding: 8px 16px; font-size: 13px;">Reset to Auto-Detected</button>
    </div>

    <div class="db-table-inputs">
        <div class="input-group">
            <label>Database Name:</label>
            <select id="databaseNameSelect">
                <option value="">Select existing database...</option>
                <!-- Existing databases will be populated -->
            </select>
            <input id="databaseNameInput" type="text" placeholder="Or enter new database name" style="display:none;" />
            <button id="toggleInput" onclick="toggleDatabaseInput()" style="margin-left: 10px;">Toggle</button>
        </div>
        <div class="input-group">
            <label>Table Name:</label>
            <input id="tableName" type="text" placeholder="Table name" readonly
                   style="background: rgba(0,234,255,0.08); cursor: not-allowed;" />
            <p style="color: #00ffaa; font-size: 12px; margin-top: 5px;">
                (Set in Gate 1 - cannot be changed here)
            </p>
        </div>
    </div>

    <h3>Columns (Standard fields {table}_id, alta_db, alta_por, ultimo_cambio, ultimo_cambio_por will be added automatically)</h3>

    <div class="table-wrap">
        <table id="schemaTable">
            <thead>
                <tr>
                    <th>Column Name</th>
                    <th>Data Type</th>
                    <th>Length</th>
                    <th>Nullable</th>
                    <th>Indexed</th>
                    <th>Comment</th>
                </tr>
            </thead>
            <tbody id="schemaTableBody">
                <!-- Populated dynamically -->
            </tbody>
        </table>
    </div>

    <h3>SQL Preview</h3>
    <pre id="sqlPreview">-- Waiting for schema...</pre>

    <button onclick="createAndInsert()" style="margin-top: 20px;">⚡ Create Table & Insert Data</button>
    <div id="insertStatus" class="status"></div>
</div>

<!-- GATE 4: THE BLESSING - RESULTS -->
<div id="resultsPanel" class="panel" style="display:none;">
    <h2>✨ Gate 4: The Cathedral Moment — Success</h2>
    <div id="insertMessage" class="status success"></div>

    <h3 style="margin-top: 32px;">Inserted Rows (Preview)</h3>
    <div id="results"></div>
</div>

<script>
// Global state
window.parsedData = null;
window.currentSchema = null;
window.autoDetectedSchema = null; // Store auto-detected schema for reset functionality

/**
 * Gate 1: Upload and Parse XLSX
 */
function uploadXLSX() {
    const fileInput = document.getElementById("file");
    const tableNameInput = document.getElementById("destinationTableName");
    const statusDiv = document.getElementById("status");

    if (!fileInput.files[0]) {
        showError(statusDiv, "Please select a file.");
        return;
    }

    const tableName = tableNameInput.value.trim();
    if (!tableName) {
        showError(statusDiv, "Please enter a destination table name.");
        return;
    }

    const formData = new FormData();
    formData.append("xlsx", fileInput.files[0]);
    formData.append("tableName", tableName);

    statusDiv.className = "status";
    statusDiv.innerHTML = "🌊 Uploading and parsing XLSX...";

    fetch("upload.php", {
        method: "POST",
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.status !== "ok") {
            showError(statusDiv, data.message);
            return;
        }

        // Store parsed data
        window.parsedData = data;

        // Show success
        statusDiv.className = "status success";
        statusDiv.innerHTML = `✓ ${data.message}`;

        // Show preview
        showPreview(data.rows);

        // Show schema builder with memory indicator
        showSchemaBuilder(data.tableName, data.schema, data.schema_source, data.schema_import_date);

        // Load daemon memory
        loadDaemonMemory();
    })
    .catch(err => {
        showError(statusDiv, "Network error: " + err.message);
    });
}

/**
 * Show data preview
 */
function showPreview(rows) {
    const panel = document.getElementById("preview");

    if (!rows || rows.length === 0) {
        panel.style.display = "none";
        return;
    }

    let html = `<h2>📊 Parsed Data Preview (First ${Math.min(rows.length, 10)} rows)</h2>`;
    
    // Show CSV metadata if available
    if (window.parsedData && window.parsedData.stats) {
        const stats = window.parsedData.stats;
        if (stats.file_type === 'CSV') {
            html += `<p style="color: #00ffaa; font-size: 13px; margin-bottom: 10px;">`;
            html += `📋 CSV File • Delimiter: <strong>${escapeHtml(stats.delimiter)}</strong> • `;
            html += `Total Rows: <strong>${stats.total_rows}</strong>`;
            if (stats.inconsistent_rows && stats.inconsistent_rows > 0) {
                html += ` • <span style="color: #ffaa00;">⚠ ${stats.inconsistent_rows} rows normalized</span>`;
            }
            html += `</p>`;
        }
    }
    
    html += `<div class='table-wrap'><table><tr>`;

    const cols = Object.keys(rows[0]);
    cols.forEach(c => html += `<th>${escapeHtml(c)}</th>`);
    html += `</tr>`;

    const previewRows = rows.slice(0, 10);
    previewRows.forEach(r => {
        html += "<tr>";
        cols.forEach(c => html += `<td>${escapeHtml(r[c] || '')}</td>`);
        html += "</tr>";
    });

    html += "</table></div>";

    panel.innerHTML = html;
    panel.style.display = "block";
}

/**
 * Gate 3: Show Schema Builder
 */
function showSchemaBuilder(tableName, schema, schemaSource, schemaImportDate) {
    window.currentSchema = schema;

    // Store auto-detected schema if not already stored (for reset functionality)
    if (window.autoDetectedSchema === null) {
        // Deep copy the schema
        window.autoDetectedSchema = JSON.parse(JSON.stringify(schema));
    }

    // Show memory indicator if schema came from memory
    if (schemaSource === 'memory' && schemaImportDate) {
        const indicator = document.getElementById('schemaMemoryIndicator');
        const message = document.getElementById('schemaMemoryMessage');

        // Get match score and column counts
        const matchScore = window.parsedData.schema_match_score || 100;
        const missingColumns = schema.filter(col => col.missing_in_file === true);
        const newColumns = schema.filter(col => col.is_new_column === true);

        // Determine score color
        let scoreColor = matchScore >= 90 ? '#00ffaa' : matchScore >= 70 ? '#ffaa00' : '#ff6600';

        // Build message with match score and column info
        let messageHTML = `Schema for table "${tableName}" loaded from previous import on ${schemaImportDate}.<br>`;
        messageHTML += `<strong style="color: ${scoreColor};">Match Score: ${matchScore}%</strong>`;

        if (matchScore < 100) {
            messageHTML += ` <span style="color: #888; font-size: 12px;">(fuzzy match - 70% threshold)</span>`;
        }

        if (newColumns.length > 0) {
            messageHTML += `<br><strong style="color: #00ffaa;">✨ ${newColumns.length} new column(s) detected (not in saved schema):</strong>`;
            messageHTML += `<br><span style="font-size: 12px; color: #00ffaa;">` + newColumns.map(c => c.name).join(', ') + `</span>`;
        }

        if (missingColumns.length > 0) {
            messageHTML += `<br><strong style="color: #ffaa00;">⚠ ${missingColumns.length} column(s) missing in file (will use defaults):</strong>`;
            messageHTML += `<br><span style="font-size: 12px; color: #ffaa00;">` + missingColumns.map(c => c.name).join(', ') + `</span>`;
        }

        messageHTML += `<br><span style="color: #00ffaa;">You can review and modify as needed.</span>`;

        message.innerHTML = messageHTML;
        indicator.style.display = 'block';
    } else {
        document.getElementById('schemaMemoryIndicator').style.display = 'none';
    }

    // Set table name
    document.getElementById("tableName").value = tableName;

    // Intelligent database name recall: check table-specific mapping first, then fall back to last used
    let suggestedDbName = 'importer_db';
    try {
        const tableMappings = JSON.parse(localStorage.getItem('tableToDatabaseMap') || '{}');
        if (tableMappings[tableName]) {
            // This table was previously imported to a specific database
            suggestedDbName = tableMappings[tableName];
        } else {
            // Fall back to last used database
            suggestedDbName = localStorage.getItem('lastDatabaseName') || 'importer_db';
        }
    } catch (e) {
        suggestedDbName = localStorage.getItem('lastDatabaseName') || 'importer_db';
    }

    // Populate dropdown with available databases
    fetchDatabaseOptions();

    // Set the dropdown value or switch to text input if needed
    const selectElement = document.getElementById("databaseNameSelect");
    if (selectElement.options.length > 1 && selectElement.options.namedItem(suggestedDbName)) {
        selectElement.value = suggestedDbName;
    } else {
        // Switch to text input mode for new database name
        selectElement.style.display = "none";
        document.getElementById("databaseNameInput").value = suggestedDbName;
        document.getElementById("databaseNameInput").style.display = "inline-block";
        document.getElementById("toggleInput").textContent = "Switch to dropdown";
    }

    // Populate schema table
    const tbody = document.getElementById("schemaTableBody");
    tbody.innerHTML = "";

    schema.forEach((col, idx) => {
        const row = tbody.insertRow();

        // Highlight missing columns (orange) and new columns (green)
        const isMissing = col.missing_in_file === true;
        const isNew = col.is_new_column === true;

        let rowStyle = '';
        let tag = '';

        if (isMissing) {
            rowStyle = 'background: rgba(255,170,0,0.1); border-left: 3px solid #ffaa00;';
            tag = '<br><small style="color: #ffaa00;">⚠ Missing in file</small>';
        } else if (isNew) {
            rowStyle = 'background: rgba(0,255,170,0.1); border-left: 3px solid #00ffaa;';
            tag = '<br><small style="color: #00ffaa;">✨ New column</small>';
        }

        row.innerHTML = `
            <td style="${rowStyle}">
                <input type="text" value="${escapeHtml(col.name)}"
                    onchange="updateSchema(${idx}, 'name', this.value)" />
                ${tag}
            </td>
            <td style="${rowStyle}">
                <select onchange="updateSchema(${idx}, 'type', this.value)">
                    <option ${col.type==='VARCHAR'?'selected':''}>VARCHAR</option>
                    <option ${col.type==='TEXT'?'selected':''}>TEXT</option>
                    <option ${col.type==='INT'?'selected':''}>INT</option>
                    <option ${col.type==='DECIMAL'?'selected':''}>DECIMAL</option>
                    <option ${col.type==='DATE'?'selected':''}>DATE</option>
                    <option ${col.type==='DATETIME'?'selected':''}>DATETIME</option>
                    <option ${col.type==='BOOLEAN'?'selected':''}>BOOLEAN</option>
                    <option ${col.type==='ENUM'?'selected':''}>ENUM</option>
                </select>
            </td>
            <td style="${rowStyle}"><input type="text" value="${escapeHtml(col.length || '')}"
                onchange="updateSchema(${idx}, 'length', this.value)"
                ${['TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'INT'].includes(col.type) ? 'disabled' : ''}
                placeholder="precision,scale" /></td>
            <td style="text-align: center; ${rowStyle}"><input type="checkbox" ${col.nullable?'checked':''}
                onchange="updateSchema(${idx}, 'nullable', this.checked)"/></td>
            <td style="text-align: center; ${rowStyle}"><input type="checkbox" ${col.indexed?'checked':''}
                onchange="updateSchema(${idx}, 'indexed', this.checked)"/></td>
            <td style="${rowStyle}"><input type="text" value="${escapeHtml(col.comment || '')}"
                onchange="updateSchema(${idx}, 'comment', this.value)" placeholder="Description"/></td>
        `;
    });

    // Update SQL preview
    updateSQLPreview();

    // Show panel
    document.getElementById("schemaPanel").style.display = "block";
}

/**
 * Update schema when user edits
 */
function updateSchema(idx, field, value) {
        if (field === 'nullable' || field === 'indexed') {
            window.currentSchema[idx][field] = value;
        } else if (field === 'length') {
            // Special handling for different types
            const columnType = window.currentSchema[idx].type;

            if (columnType === 'DECIMAL') {
                // DECIMAL: Parse comma-separated precision,scale (e.g., "10,2")
                const parts = value.toString().split(',');
                if (parts.length === 2) {
                    const precision = parseInt(parts[0], 10);
                    const scale = parseInt(parts[1], 10);
                    if (!isNaN(precision) && !isNaN(scale)) {
                        window.currentSchema[idx].precision = precision;
                        window.currentSchema[idx].scale = scale;
                        // Store the raw string for display purposes
                        window.currentSchema[idx].length = value;
                    } else {
                        // Fall back to numeric parsing if comma parsing fails
                        const cleanedValue = value.toString().replace(/,/g, '.').replace(/[^0-9.]/g, '');
                        const parsedValue = cleanedValue ? parseFloat(cleanedValue) : null;
                        window.currentSchema[idx][field] = isNaN(parsedValue) ? null : parsedValue;
                    }
                } else {
                    // Fall back to numeric parsing for invalid DECIMAL format
                    const cleanedValue = value.toString().replace(/,/g, '.').replace(/[^0-9.]/g, '');
                    const parsedValue = cleanedValue ? parseFloat(cleanedValue) : null;
                    window.currentSchema[idx][field] = isNaN(parsedValue) ? null : parsedValue;
                }
            } else if (columnType === 'ENUM') {
                // ENUM: Store raw string value (e.g., "('Si','No')")
                // Do NOT parse as number - preserve the entire string including parentheses and quotes
                window.currentSchema[idx].length = value || null;
            } else if (columnType === 'VARCHAR') {
                // VARCHAR: Store as string (e.g., "100")
                // Keep as string for consistency, even though it's numeric
                window.currentSchema[idx].length = value || null;
            } else {
                // Other types: Parse as numeric (for future numeric types)
                let stringValue = '';
                if (typeof value === 'string') {
                    stringValue = value;
                } else if (typeof value === 'number') {
                    stringValue = String(value);
                } else {
                    stringValue = '';
                }
                const cleanedValue = stringValue.replace(/,/g, '.').replace(/[^0-9.]/g, '');
                const parsedValue = cleanedValue ? parseFloat(cleanedValue) : null;
                window.currentSchema[idx][field] = isNaN(parsedValue) ? null : parsedValue;
            }
        } else {
            window.currentSchema[idx][field] = value;
        }

    // Handle length input for different types
    if (field === 'type') {
        const row = document.getElementById("schemaTableBody").rows[idx];
        const lengthInput = row.cells[2].querySelector('input');
        const noLengthTypes = ['TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'INT'];

        if (noLengthTypes.includes(value)) {
            // Type doesn't use length - clear both UI and data
            lengthInput.disabled = true;
            lengthInput.value = '';  // Clear the visible value
            window.currentSchema[idx].length = null;  // Clear from schema data
        } else {
            // Type uses length - enable the input
            lengthInput.disabled = false;
        }
    }

    updateSQLPreview();
}

/**
 * Singularize table name (mirrors PHP SchemaConventions logic)
 * Converts plural table names to singular for primary key naming
 */
function singularizeTableName(tableName) {
    // Explicit whitelist of known plural forms (mirrors PHP SchemaConventions::IRREGULAR_PLURALS)
    const irregulars = {
        'propietarios': 'propietario',
        'departamentos': 'departamento',
        'alumnos_becados': 'alumno_becado',
        'eleyeme_cfdi_emitidos': 'eleyeme_cfdi_emitido',
        'productos': 'producto',
        'servicios': 'servicio',
        'empleados': 'empleado',
        'clientes': 'cliente',
        'proveedores': 'proveedor',
        'contratos': 'contrato',
        'facturas': 'factura',
        'pagos': 'pago',
        'usuarios': 'usuario',
        'roles': 'rol',
        'permisos': 'permiso',
        'catalogos': 'catalogo',
        'categorias': 'categoria',
        'sucursales': 'sucursal',
        'almacenes': 'almacen',
        'inventarios': 'inventario',
        'movimientos': 'movimiento',
        'documentos': 'documento',
        'archivos': 'archivo',
        'reportes': 'reporte',
        'configuraciones': 'configuracion',
        'bitacoras': 'bitacora',
        'auditorias': 'auditoria',
        'notificaciones': 'notificacion',
        'mensajes': 'mensaje',
        'tareas': 'tarea',
        'proyectos': 'proyecto',
        'eventos': 'evento',
        'registros': 'registro'
    };

    // Check explicit whitelist first
    if (irregulars[tableName]) {
        return irregulars[tableName];
    }

    // Fallback rule: strip trailing 's'
    if (tableName.endsWith('s')) {
        return tableName.slice(0, -1);
    }

    // Return as-is if already singular or unknown pattern
    return tableName;
}

/**
 * Generate SQL preview
 */
function updateSQLPreview() {
    const dbNameElement = document.getElementById("databaseNameSelect");
    const dbNameInput = document.getElementById("databaseNameInput");
    
    let dbName;
    if (dbNameElement.style.display !== "none") {
        dbName = dbNameElement.value;
    } else {
        dbName = dbNameInput.value;
    }
    const tableName = document.getElementById("tableName").value;

    // Get singularized table name for primary key
    const singularForm = singularizeTableName(tableName);
    const pkFieldName = `${singularForm}_id`;

    let sql = `-- Database: ${dbName}\n`;
    sql += `DROP TABLE IF EXISTS \`${tableName}\`;\n\n`;
    sql += `CREATE TABLE \`${tableName}\` (\n`;

    // Dynamic primary key field (table-specific, e.g., producto_id for productos table)
    sql += `  \`${pkFieldName}\` VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',\n`;

    // User columns
    window.currentSchema.forEach(col => {
        let colType = col.type;
        if (colType === 'VARCHAR' && col.length) {
            colType += `(${col.length})`;
        } else if (colType === 'ENUM' && col.length) {
            colType += col.length;  // Already formatted as ('value1','value2')
        } else if (colType === 'DECIMAL') {
            if (col.precision !== undefined && col.scale !== undefined) {
                colType += `(${col.precision},${col.scale})`;
            } else if (col.length) {
                // Legacy compatibility - use length as comma-separated string
                const parts = col.length.toString().split(',');
                if (parts.length === 2) {
                    colType += `(${parts.join(',')})`;
                } else {
                    colType += `(${col.length})`;
                }
            }
        } else if (colType === 'DECIMAL' && col.length) {
            colType += `(${col.length})`;
        }

        sql += `  \`${col.name}\` ${colType}`;
        sql += col.nullable ? ' NULL' : ' NOT NULL';
        if (col.comment) {
            // Escape single quotes in comment
            const safeComment = col.comment.replace(/'/g, "''");
            sql += ` COMMENT '${safeComment}'`;
        }
        sql += `,\n`;
    });

    // Standard audit fields (Spanish canonical schema)
    sql += `  \`alta_db\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'Timestamp del momento de inserción del registro',\n`;
    sql += `  \`alta_por\` VARCHAR(32) DEFAULT 'system' COMMENT 'Usuario que insertó el registro',\n`;
    sql += `  \`ultimo_cambio\` TIMESTAMP NULL DEFAULT NULL COMMENT 'Timestamp del último cambio al registro',\n`;
    sql += `  \`ultimo_cambio_por\` VARCHAR(32) NULL DEFAULT NULL COMMENT 'Usuario que hizo el último cambio',\n`;

    // Indexes
    window.currentSchema.forEach(col => {
        if (col.indexed) {
            sql += `  INDEX \`idx_${col.name}\` (\`${col.name}\`),\n`;
        }
    });

    // Remove trailing comma
    sql = sql.replace(/,\n$/, '\n');

    sql += `) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;`;

    document.getElementById("sqlPreview").textContent = sql;
}

/**
 * Gate 4: Create Table and Insert Data
 */
function createAndInsert() {
    const dbNameElement = document.getElementById("databaseNameSelect");
    const dbNameInput = document.getElementById("databaseNameInput");
    
    let dbName;
    if (dbNameElement.style.display !== "none") {
        dbName = dbNameElement.value.trim();
    } else {
        dbName = dbNameInput.value.trim();
    }
    const tableName = document.getElementById("tableName").value.trim();
    const statusDiv = document.getElementById("insertStatus");

    if (!dbName || !tableName) {
        showError(statusDiv, "Database name and table name are required");
        return;
    }

    if (!window.parsedData || !window.currentSchema) {
        showError(statusDiv, "No data to insert. Please upload a file first.");
        return;
    }

    // Check if user wants to skip table recreation
    const skipTableRecreationCheckbox = document.getElementById("skipTableRecreation");
    const skipTableRecreation = skipTableRecreationCheckbox ? skipTableRecreationCheckbox.checked : false;

    statusDiv.className = "status";
    statusDiv.innerHTML = skipTableRecreation
        ? "⚡ Inserting data into existing table..."
        : "⚡ Creating table and inserting data...";

    // Save to daemon memory
    saveDaemonMemory(dbName, tableName);

    fetch("insert.php", {
        method: "POST",
        headers: {"Content-Type":"application/json"},
        body: JSON.stringify({
            databaseName: dbName,
            tableName: tableName,
            schema: window.currentSchema,
            rows: window.parsedData.rows,
            skipTableRecreation: skipTableRecreation
        })
    })
    .then(r => r.json())
    .then(data => {
        if (data.status !== "ok") {
            showError(statusDiv, data.message);
            return;
        }

        // Show success
        statusDiv.className = "status success";
        statusDiv.innerHTML = `✓ ${data.message}`;

        // Show cathedral moment
        showCathedralMoment(data);
    })
    .catch(err => {
        showError(statusDiv, "Network error: " + err.message);
    });
}

/**
 * Show cathedral moment - the final success display
 */
function showCathedralMoment(data) {
    const panel = document.getElementById("resultsPanel");
    const messageDiv = document.getElementById("insertMessage");
    const resultsDiv = document.getElementById("results");

    // Success message
    let correctedInfo = '';
    if (data.corrected && data.corrected > 0) {
        correctedInfo = `<br><span style="color: #ffaa00;">⚙ Auto-corrected: ${data.corrected} row${data.corrected > 1 ? 's' : ''}</span>`;
    }

    messageDiv.innerHTML = `
        <strong>✨ The Blessing is Complete</strong><br>
        ${data.message}<br>
        Rows inserted: ${data.inserted}${data.failed > 0 ? `, Failed: ${data.failed}` : ''}${correctedInfo}
    `;

    // Show results table
    if (data.rows && data.rows.length > 0) {
        let html = `<div class='table-wrap'><table><tr>`;
        const cols = Object.keys(data.rows[0]);
        cols.forEach(c => html += `<th>${escapeHtml(c)}</th>`);
        html += "</tr>";

        data.rows.forEach(r => {
            html += "<tr>";
            cols.forEach(c => html += `<td>${escapeHtml(r[c] || '')}</td>`);
            html += "</tr>";
        });

        html += "</table></div>";
        resultsDiv.innerHTML = html;
    }

    // Add cathedral pulse effect
    panel.className = "panel cathedral";
    panel.style.display = "block";

    // Scroll to results
    panel.scrollIntoView({ behavior: 'smooth' });
}

/**
 * Daemon Memory: Save preferences
 * Stores table-to-database mappings for future imports
 */
function saveDaemonMemory(dbName, tableName) {
    localStorage.setItem('lastDatabaseName', dbName);
    localStorage.setItem('lastTableName', tableName);

    // Store table-to-database mapping for intelligent defaults
    try {
        let tableMappings = JSON.parse(localStorage.getItem('tableToDatabaseMap') || '{}');
        tableMappings[tableName] = dbName;
        localStorage.setItem('tableToDatabaseMap', JSON.stringify(tableMappings));
    } catch (e) {
        console.error('Failed to save table-database mapping:', e);
    }
}

/**
 * Toggle between dropdown and text input modes for database name
 */
function toggleDatabaseInput() {
    const selectElement = document.getElementById("databaseNameSelect");
    const inputElement = document.getElementById("databaseNameInput");
    const toggleBtn = document.getElementById("toggleInput");
    
    if (selectElement.style.display !== "none") {
        // Switch to text input mode
        selectElement.style.display = "none";
        inputElement.style.display = "inline-block";
        toggleBtn.textContent = "Switch to dropdown";
        inputElement.focus();
    } else {
        // Switch to dropdown mode
        selectElement.style.display = "inline-block";
        inputElement.style.display = "none";
        toggleBtn.textContent = "Toggle";
        
        // Fetch databases when switching to dropdown mode
        fetchDatabaseOptions();
    }
}

/**
 * Fetch available databases and populate dropdown
 */
function fetchDatabaseOptions() {
    fetch("api/getDatabases.php")
        .then(response => response.json())
        .then(data => {
            if (data.status === "ok") {
                const selectElement = document.getElementById("databaseNameSelect");
                const existingOptions = Array.from(selectElement.options).map(option => option.value);
                
                // Clear existing non-default options
                for (let i = selectElement.options.length - 1; i >= 1; i--) {
                    selectElement.remove(i);
                }
                
                // Add new available databases
                data.databases.forEach(dbName => {
                    if (!existingOptions.includes(dbName)) {
                        const option = document.createElement("option");
                        option.value = dbName;
                        option.textContent = dbName;
                        selectElement.appendChild(option);
                    }
                });
                
                // Set the default value if it exists
                const lastDbName = localStorage.getItem('lastDatabaseName') || 'importer_db';
                if (data.databases.includes(lastDbName)) {
                    selectElement.value = lastDbName;
                }
            }
        })
        .catch(error => {
            console.error("Error fetching databases:", error);
        });
}

/**
 * Daemon Memory: Load preferences
 */
function loadDaemonMemory() {
    // This function is now handled by the schema builder logic
    // The dropdown will automatically load the last selected value on page load
}

/**
 * Reset schema to auto-detected values (ignoring memory)
 */
function resetToAutoDetectedSchema() {
    if (!window.autoDetectedSchema) {
        alert('No auto-detected schema available');
        return;
    }

    // Restore auto-detected schema
    window.currentSchema = JSON.parse(JSON.stringify(window.autoDetectedSchema));

    // Hide memory indicator
    document.getElementById('schemaMemoryIndicator').style.display = 'none';

    // Re-populate schema table
    const tbody = document.getElementById("schemaTableBody");
    tbody.innerHTML = "";

    window.currentSchema.forEach((col, idx) => {
        const row = tbody.insertRow();

        // Highlight missing columns (orange) and new columns (green)
        const isMissing = col.missing_in_file === true;
        const isNew = col.is_new_column === true;

        let rowStyle = '';
        let tag = '';

        if (isMissing) {
            rowStyle = 'background: rgba(255,170,0,0.1); border-left: 3px solid #ffaa00;';
            tag = '<br><small style="color: #ffaa00;">⚠ Missing in file</small>';
        } else if (isNew) {
            rowStyle = 'background: rgba(0,255,170,0.1); border-left: 3px solid #00ffaa;';
            tag = '<br><small style="color: #00ffaa;">✨ New column</small>';
        }

        row.innerHTML = `
            <td style="${rowStyle}">
                <input type="text" value="${escapeHtml(col.name)}"
                    onchange="updateSchema(${idx}, 'name', this.value)" />
                ${tag}
            </td>
            <td style="${rowStyle}">
                <select onchange="updateSchema(${idx}, 'type', this.value)">
                    <option ${col.type==='VARCHAR'?'selected':''}>VARCHAR</option>
                    <option ${col.type==='TEXT'?'selected':''}>TEXT</option>
                    <option ${col.type==='INT'?'selected':''}>INT</option>
                    <option ${col.type==='DECIMAL'?'selected':''}>DECIMAL</option>
                    <option ${col.type==='DATE'?'selected':''}>DATE</option>
                    <option ${col.type==='DATETIME'?'selected':''}>DATETIME</option>
                    <option ${col.type==='BOOLEAN'?'selected':''}>BOOLEAN</option>
                    <option ${col.type==='ENUM'?'selected':''}>ENUM</option>
                </select>
            </td>
            <td style="${rowStyle}"><input type="text" value="${escapeHtml(col.length || '')}"
                onchange="updateSchema(${idx}, 'length', this.value)"
                ${['TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'INT'].includes(col.type) ? 'disabled' : ''}
                placeholder="precision,scale" /></td>
            <td style="text-align: center; ${rowStyle}"><input type="checkbox" ${col.nullable?'checked':''}
                onchange="updateSchema(${idx}, 'nullable', this.checked)"/></td>
            <td style="text-align: center; ${rowStyle}"><input type="checkbox" ${col.indexed?'checked':''}
                onchange="updateSchema(${idx}, 'indexed', this.checked)"/></td>
            <td style="${rowStyle}"><input type="text" value="${escapeHtml(col.comment || '')}"
                onchange="updateSchema(${idx}, 'comment', this.value)" placeholder="Description"/></td>
        `;
    });

    // Update SQL preview
    updateSQLPreview();

    alert('✓ Schema reset to auto-detected values');
}

/**
 * Show error (red rune)
 */
function showError(element, message) {
    element.className = "status error";
    element.innerHTML = `⚠ ${escapeHtml(message)}`;
}

/**
 * Escape HTML to prevent XSS
 */
function escapeHtml(text) {
    if (text === null || text === undefined) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
    fetchDatabaseOptions();

    // Pre-populate destination table name with last used table name (as a helpful hint)
    const lastTableName = localStorage.getItem('lastTableName');
    if (lastTableName) {
        const tableNameInput = document.getElementById('destinationTableName');
        if (tableNameInput) {
            tableNameInput.placeholder = `e.g., ${lastTableName}`;
        }
    }

    // Back to Top Button - Show/Hide on scroll
    const backToTopBtn = document.getElementById('backToTopBtn');

    window.addEventListener('scroll', function() {
        if (window.scrollY > 300) {
            backToTopBtn.classList.add('show');
        } else {
            backToTopBtn.classList.remove('show');
        }
    });

    // Smooth scroll to top when button is clicked
    backToTopBtn.addEventListener('click', function() {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });
});
</script>

<!-- Floating Back to Top Button -->
<button id="backToTopBtn" title="Back to Top">↑</button>

</body>
</html>
