<?php
/**
 * insert.php - Gate 4: The Blessing
 * Creates tables, plants UUID seeds, inserts data with transactions
 */

session_start(); // Enable session to retrieve import_log_id

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/ErrorHandler.php';
require_once __DIR__ . '/lib/DatabaseHelper.php';
require_once __DIR__ . '/lib/DataCleaner.php';
require_once __DIR__ . '/lib/DataValidator.php';
require_once __DIR__ . '/lib/SchemaConventions.php';
require_once __DIR__ . '/lib/ImportLogger.php';

/**
 * Get default value for a given data type
 * Used to prevent NULL constraint violations
 */
function getDefaultValueForType($type) {
    switch (strtoupper($type)) {
        case 'VARCHAR':
        case 'TEXT':
        case 'ENUM':
            return ''; // Empty string for text and enum
        case 'INT':
        case 'INTEGER':
        case 'BIGINT':
        case 'SMALLINT':
        case 'TINYINT':
            return 0;
        case 'DECIMAL':
        case 'NUMERIC':
        case 'FLOAT':
        case 'DOUBLE':
            return '0.00';
        case 'DATE':
            return date('Y-m-d');
        case 'DATETIME':
        case 'TIMESTAMP':
            return date('Y-m-d H:i:s');
        case 'BOOLEAN':
        case 'BOOL':
            return 0;
        default:
            return null;
    }
}

/**
 * Get existing columns from a table
 * Returns array of column names
 */
function getTableColumns($conn, $tableName) {
    $safeTableName = $conn->real_escape_string($tableName);
    $result = $conn->query("DESCRIBE `$safeTableName`");

    if (!$result) {
        return [];
    }

    $columns = [];
    while ($row = $result->fetch_assoc()) {
        $columns[] = $row['Field'];
    }
    return $columns;
}

/**
 * Format value for SQL DEFAULT clause
 */
function sqlValue($value, $conn) {
    if ($value === null) {
        return 'NULL';
    }
    if (is_numeric($value)) {
        return $value;
    }
    return "'" . $conn->real_escape_string($value) . "'";
}

/**
 * Generate ALTER TABLE ADD COLUMN statement
 */
function generateAlterTableSQL($tableName, $col, $conn) {
    $colName = $conn->real_escape_string($col['name']);
    $baseType = strtoupper($col['type']);
    $colType = $baseType;

    // Add length for VARCHAR/DECIMAL/ENUM
    if ($baseType === 'VARCHAR' && !empty($col['length'])) {
        $colType .= '(' . $col['length'] . ')';
    } elseif ($baseType === 'DECIMAL' && !empty($col['length'])) {
        $colType .= '(' . $col['length'] . ')';
    } elseif ($baseType === 'ENUM' && !empty($col['length'])) {
        $colType .= $col['length']; // Already formatted as ('value1','value2')
    }

    // Add character set for string types
    $charset = '';
    if (in_array($baseType, ['VARCHAR', 'TEXT', 'CHAR', 'TINYTEXT', 'MEDIUMTEXT', 'LONGTEXT', 'ENUM'])) {
        $charset = ' CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci';
    }

    // Add NULL/NOT NULL
    $nullable = $col['nullable'] ? 'NULL' : 'NOT NULL';

    // Add default value if specified
    $default = '';
    if (isset($col['default_value'])) {
        $default = ' DEFAULT ' . sqlValue($col['default_value'], $conn);
    }

    // Add comment
    $comment = '';
    if (!empty($col['comment'])) {
        $safeComment = $conn->real_escape_string($col['comment']);
        $comment = " COMMENT '$safeComment'";
    }

    return "ALTER TABLE `$tableName` ADD COLUMN `$colName` $colType$charset $nullable$default$comment";
}

try {
    // Read JSON input
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (!$data) {
        ErrorHandler::jsonError('Invalid JSON input', 'INVALID_INPUT', 400);
    }

    // Validate required fields
    if (!isset($data['databaseName']) || !isset($data['tableName']) || !isset($data['schema']) || !isset($data['rows'])) {
        ErrorHandler::jsonError('Missing required fields: databaseName, tableName, schema, rows', 'MISSING_FIELDS', 400);
    }

    $databaseName = DataCleaner::sanitizeTableName($data['databaseName']);
    $tableName = DataCleaner::sanitizeTableName($data['tableName']);
    $schema = $data['schema'];
    $rows = $data['rows'];
    $skipTableRecreation = isset($data['skipTableRecreation']) ? (bool)$data['skipTableRecreation'] : false;

    // Sanitize schema column names
    foreach ($schema as &$col) {
        $col['name'] = DataCleaner::sanitizeColumnName($col['name']);
    }
    unset($col); // Break reference

    // ===== INITIALIZE DATABASE FIRST =====
    // IMPORTANT: Must select database BEFORE starting import log
    // so that ImportLogger can create log tables in the correct database
    
    $db = DatabaseHelper::getInstance();
    $db->createAndSelectDatabase($databaseName);
    $conn = $db->getConnection();

    // ===== START IMPORT LOGGING =====
    // Now that database is selected, we can start the import log
    $importLogId = null;
    
    try {
        // Retrieve file metadata from session (set by upload.php)
        $fileMetadata = $_SESSION['file_metadata'] ?? [];
        
        // Merge with database information
        $logMetadata = array_merge($fileMetadata, [
            'database_name' => $databaseName,
            'table_name' => $tableName,
            'alta_por' => 'system'
        ]);
        
        // Start import log (database is now selected, so log tables can be created)
        $importLogId = ImportLogger::startImport($logMetadata);
        ErrorHandler::logError('Import log started', ['import_log_id' => $importLogId, 'database' => $databaseName, 'table' => $tableName]);
        
        // Clear file metadata from session as it's now been used
        unset($_SESSION['file_metadata']);
        
    } catch (Exception $logEx) {
        // Don't fail the insert if logging fails
        ErrorHandler::logError('Failed to start import log: ' . $logEx->getMessage());
        $importLogId = null;
    }

    $safeTableName = $db->escapeIdentifier($tableName);

    // ===== DROP AND CREATE TABLE (SKIPPABLE) =====
    if (!$skipTableRecreation) {
        // ===== DROP EXISTING TABLE =====
        $dropSQL = "DROP TABLE IF EXISTS $safeTableName";

        if (!$conn->query($dropSQL)) {
            ErrorHandler::jsonError('Failed to drop existing table: ' . $conn->error, 'DROP_ERROR', 500);
        }

        // ===== BUILD CREATE TABLE STATEMENT =====

        $createSQL = "CREATE TABLE $safeTableName (\n";

        // Add dynamic primary key first (table-specific)
        $pkFieldName = SchemaConventions::getPrimaryKeyName($tableName);
        $createSQL .= "  `{$pkFieldName}` VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',\n";

        // Add user-defined columns from schema
        foreach ($schema as $col) {
            $colName = $db->escapeIdentifier($col['name']);
            $baseType = strtoupper($col['type']);
            $colType = $baseType;

            // Add length for VARCHAR/DECIMAL/ENUM
            if ($baseType === 'VARCHAR' && !empty($col['length'])) {
                $colType .= '(' . $col['length'] . ')';
            } elseif ($baseType === 'DECIMAL' && !empty($col['length'])) {
                $colType .= '(' . $col['length'] . ')';
            } elseif ($baseType === 'ENUM' && !empty($col['length'])) {
                $colType .= $col['length']; // Already formatted as ('value1','value2')
            }

            $createSQL .= "  $colName $colType";

            // Add character set for string types only (BEFORE NULL constraint)
            if (in_array($baseType, ['VARCHAR', 'TEXT', 'CHAR', 'TINYTEXT', 'MEDIUMTEXT', 'LONGTEXT', 'ENUM'])) {
                $createSQL .= " CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci";
            }

            // Add NULL/NOT NULL (AFTER character set)
            if (!$col['nullable']) {
                $createSQL .= " NOT NULL";
            } else {
                $createSQL .= " NULL";
            }

            // Add Comment
            if (!empty($col['comment'])) {
                $safeComment = $conn->real_escape_string($col['comment']);
                $createSQL .= " COMMENT '$safeComment'";
            }

            $createSQL .= ",\n";
        }

        // Add standard audit fields (Spanish canonical)
        $standardFields = SchemaConventions::getStandardFields($tableName);
        foreach ($standardFields as $fieldName => $fieldConfig) {
            $createSQL .= "  `$fieldName` {$fieldConfig['type']} {$fieldConfig['constraint']} COMMENT '{$fieldConfig['comment']}',\n";
        }

        // Add indexes for marked columns
        foreach ($schema as $col) {
            if ($col['indexed']) {
                $colName = $col['name'];
                $createSQL .= "  INDEX `idx_$colName` (`$colName`),\n";
            }
        }

        // Remove trailing comma and newline
        $createSQL = rtrim($createSQL, ",\n") . "\n";

        // Close CREATE TABLE statement
        $createSQL .= ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";

        // Log the SQL for debugging (with hex dump to see hidden chars)
        ErrorHandler::logError('Generated SQL: ' . $createSQL, ['action' => 'CREATE_TABLE', 'hex' => bin2hex($createSQL)]);

        // Execute CREATE TABLE
        if (!$conn->query($createSQL)) {
            ErrorHandler::logError('SQL that failed: ' . $createSQL, ['error' => $conn->error]);

            // Log CREATE TABLE failure
            ImportLogger::logCreateTable($importLogId, [
                'database_name' => $databaseName,
                'table_name' => $tableName,
                'create_sql' => $createSQL,
                'status' => 'error',
                'error' => $conn->error
            ]);

            ErrorHandler::jsonError('Failed to create table: ' . $conn->error, 'CREATE_ERROR', 500);
        }

        // ===== LOG CREATE TABLE SUCCESS =====
        ImportLogger::logCreateTable($importLogId, [
            'database_name' => $databaseName,
            'table_name' => $tableName,
            'create_sql' => $createSQL,
            'status' => 'success'
        ]);

        // ===== LOG SCHEMA DETAILS =====
        ImportLogger::logSchema($importLogId, $schema);
    } else {
        // ===== TABLE RECREATION SKIPPED - ALTER TABLE IF NEEDED =====
        ErrorHandler::logError('Table recreation skipped (insert-only mode)', ['table' => $tableName]);

        // Verify table exists
        $tableCheckResult = $conn->query("SHOW TABLES LIKE '$tableName'");
        if (!$tableCheckResult || $tableCheckResult->num_rows === 0) {
            ErrorHandler::jsonError("Table '$tableName' does not exist. Cannot use skip mode on non-existent table.", 'TABLE_NOT_FOUND', 400);
        }

        // Get existing table columns
        $existingColumns = getTableColumns($conn, $tableName);

        // Find columns in schema that don't exist in table
        $missingColumns = [];
        foreach ($schema as $col) {
            if (!in_array($col['name'], $existingColumns)) {
                $missingColumns[] = $col;
            }
        }

        // ALTER TABLE to add missing columns
        if (!empty($missingColumns)) {
            ErrorHandler::logError('Adding missing columns via ALTER TABLE', [
                'table' => $tableName,
                'missing_count' => count($missingColumns),
                'columns' => array_map(fn($c) => $c['name'], $missingColumns)
            ]);

            foreach ($missingColumns as $col) {
                $alterSQL = generateAlterTableSQL($tableName, $col, $conn);

                ErrorHandler::logError('Executing ALTER TABLE', [
                    'table' => $tableName,
                    'column' => $col['name'],
                    'sql' => $alterSQL
                ]);

                if (!$conn->query($alterSQL)) {
                    ErrorHandler::logError('ALTER TABLE failed', [
                        'sql' => $alterSQL,
                        'error' => $conn->error
                    ]);
                    ErrorHandler::jsonError("Failed to add column '{$col['name']}': " . $conn->error, 'ALTER_TABLE_ERROR', 500);
                }

                ErrorHandler::logError('Column added successfully', [
                    'table' => $tableName,
                    'column' => $col['name'],
                    'type' => $col['type']
                ]);
            }

            ErrorHandler::logError('ALTER TABLE completed', [
                'table' => $tableName,
                'columns_added' => count($missingColumns)
            ]);
        } else {
            ErrorHandler::logError('No missing columns - table structure matches schema', ['table' => $tableName]);
        }
    }

    // ===== PREPARE INSERT STATEMENT =====

    // Build column list (primary key + standard audit fields + user columns)
    $pkFieldName = SchemaConventions::getPrimaryKeyName($tableName);
    $standardFields = SchemaConventions::getStandardFields($tableName);

    $columnNames = [$pkFieldName]; // Start with dynamic primary key
    $columnNames = array_merge($columnNames, array_keys($standardFields)); // Add audit fields

    foreach ($schema as $col) {
        $columnNames[] = $col['name'];
    }

    $columnList = '`' . implode('`, `', $columnNames) . '`';
    $placeholders = str_repeat('?, ', count($columnNames));
    $placeholders = rtrim($placeholders, ', ');

    $insertSQL = "INSERT INTO $safeTableName ($columnList) VALUES ($placeholders)";
    $stmt = $conn->prepare($insertSQL);
    file_put_contents('C:\wamp\logs\insert.txt', $insertSQL);
    if (!$stmt) {
        ErrorHandler::jsonError('Failed to prepare insert statement: ' . $conn->error, 'PREPARE_ERROR', 500);
    }

    // ===== INSERT ROWS IN BATCHES =====

    $insertedCount = 0;
    $failedCount = 0;
    $correctedCount = 0; // Track auto-corrected values
    $batchSize = 50;
    $currentBatch = 0;

    $db->beginTransaction();

    // Generate timestamp once for all inserts in this batch
    $now = date('Y-m-d H:i:s');

    foreach ($rows as $rowIndex => $row) {
        try {
            // Generate UUID for this row
            $uuid = $db->generateUUID();

            // Build values array: [uuid (dynamic PK), alta_db, alta_por, ultimo_cambio (NULL), ultimo_cambio_por (NULL), ...user data]
            // Order must match: primary key, then audit fields from SchemaConventions::getStandardFields()
            $values = [
                $uuid,        // Dynamic primary key (e.g., producto_id)
                $now,         // alta_db (explicit PHP timestamp)
                'system',     // alta_por
                null,         // ultimo_cambio (NULL on insert, populated on UPDATE)
                null          // ultimo_cambio_por (NULL on insert, populated on UPDATE)
            ];

            // Track if this row had any corrections
            $rowHadCorrections = false;

            // Add user data in schema order
            // Use 'original' key to fetch data (sanitized from XLSX), but 'name' is used for SQL column
            foreach ($schema as $col) {
                // Check if this column was missing from the uploaded file (subset matching)
                if (isset($col['missing_in_file']) && $col['missing_in_file'] === true) {
                    // Column is in schema but NOT in uploaded file
                    // Use the default value that was set in upload.php
                    $values[] = $col['default_value'] ?? null;
                    continue;
                }

                $originalKey = $col['original'] ?? $col['name']; // Fallback to 'name' if 'original' not set
                $rawValue = $row[$originalKey] ?? null;
                $colType = $col['type'];
                $colLength = $col['length'] ?? null;
                $isNullable = isset($col['nullable']) ? $col['nullable'] : true;

                // Extract enum values if ENUM type
                $enumValues = null;
                if (strtoupper($colType) === 'ENUM' && isset($col['length'])) {
                    // Parse ENUM values from length field (e.g., "('value1','value2')")
                    $enumString = trim($col['length'], "()");
                    $enumValues = array_map(function($v) {
                        return trim($v, " '\"");
                    }, explode(',', $enumString));
                }

                // VALIDATE AND SANITIZE VALUE
                $validationResult = DataValidator::validate($rawValue, $colType, $colLength, $isNullable, $enumValues);
                $validatedValue = $validationResult['value'];

                // Track corrections
                if ($validationResult['corrected']) {
                    $rowHadCorrections = true;

                    // Log the correction for audit trail
                    ErrorHandler::logError('Value auto-corrected', [
                        'row' => $rowIndex + 2,
                        'column' => $col['name'],
                        'original' => $validationResult['original'],
                        'corrected_to' => $validatedValue,
                        'reason' => $validationResult['reason']
                    ]);
                }

                $values[] = $validatedValue;
            }

            // Increment corrected count if row had any corrections
            if ($rowHadCorrections) {
                $correctedCount++;
            }

            // Bind parameters dynamically
            $types = str_repeat('s', count($values)); // All strings for simplicity
            $stmt->bind_param($types, ...$values);

            // Execute
            if ($stmt->execute()) {
                $insertedCount++;
            } else {
                $failedCount++;
                ErrorHandler::trackRowError($rowIndex + 2, $stmt->error);
            }

            // Commit in batches
            $currentBatch++;
            if ($currentBatch >= $batchSize) {
                $db->commit();
                $db->beginTransaction();
                $currentBatch = 0;
            }

        } catch (Exception $e) {
            $failedCount++;
            ErrorHandler::trackRowError($rowIndex + 2, $e->getMessage());
        }
    }

    // Final commit
    $db->commit();
    $stmt->close();

    // ===== RETRIEVE INSERTED DATA =====

    $result = $conn->query("SELECT * FROM $safeTableName LIMIT 100");
    $insertedRows = [];

    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $insertedRows[] = $row;
        }
    }

    // ===== LOG FINAL IMPORT STATUS =====
    ImportLogger::updateImportStatus($importLogId, [
        'rows_processed' => count($rows),
        'rows_inserted' => $insertedCount,
        'rows_failed' => $failedCount,
        'errors' => ErrorHandler::getErrors()
    ]);

    // ===== SUCCESS RESPONSE =====

    $successMessage = $skipTableRecreation
        ? "Inserted $insertedCount rows into existing table '$tableName'" . ($failedCount > 0 ? ", $failedCount failed" : "")
        : "Table '$tableName' created! Inserted $insertedCount rows" . ($failedCount > 0 ? ", $failedCount failed" : "");

    // Add corrected count to message if any corrections were made
    if ($correctedCount > 0) {
        $successMessage .= " ($correctedCount row" . ($correctedCount > 1 ? 's' : '') . " auto-corrected)";
    }

    ErrorHandler::jsonSuccess([
        'message' => $successMessage,
        'inserted' => $insertedCount,
        'failed' => $failedCount,
        'corrected' => $correctedCount,
        'errors' => ErrorHandler::getErrors(),
        'rows' => $insertedRows
    ]);

} catch (Exception $e) {
    // Rollback on error
    if (isset($db)) {
        $db->rollback();
    }

    // Log error to import logger
    if (isset($importLogId)) {
        ImportLogger::logError($importLogId, $e->getMessage());
    }

    ErrorHandler::handleException($e);
}
