<?php
/**
 * ErrorHandler - The Rune Scribe
 * Transforms errors into red runes, not failures
 */

class ErrorHandler {
    private static $errors = [];
    private static $logFile = null;

    /**
     * Initialize error handler
     */
    public static function init() {
        self::$logFile = __DIR__ . '/../logs/importer.log';
    }

    /**
     * Log an error
     */
    public static function logError($message, $context = []) {
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? json_encode($context, JSON_UNESCAPED_UNICODE) : '';

        $logEntry = "[$timestamp] ERROR: $message\n";
        if ($contextStr) {
            $logEntry .= "Context: $contextStr\n";
        }
        $logEntry .= str_repeat('-', 80) . "\n";

        // Write to log file
        if (self::$logFile) {
            file_put_contents(self::$logFile, $logEntry, FILE_APPEND);
        }

        // Also log to error_log if in debug mode
        if (DEBUG_MODE) {
            error_log($message);
        }
    }

    /**
     * Track a row-level error
     */
    public static function trackRowError($rowNumber, $reason) {
        self::$errors[] = [
            'row' => $rowNumber,
            'reason' => $reason
        ];

        // Persist to log file for debugging
        self::logError("Row $rowNumber failed: $reason", ['row' => $rowNumber]);
    }

    /**
     * Get all tracked errors
     */
    public static function getErrors() {
        return self::$errors;
    }

    /**
     * Clear errors
     */
    public static function clearErrors() {
        self::$errors = [];
    }

    /**
     * Return JSON error response
     */
    public static function jsonError($message, $code = 'ERROR', $httpCode = 500) {
        http_response_code($httpCode);
        header('Content-Type: application/json; charset=UTF-8');

        echo json_encode([
            'status' => 'error',
            'message' => $message,
            'code' => $code,
            'errors' => self::$errors
        ], JSON_UNESCAPED_UNICODE);

        self::logError($message, ['code' => $code, 'http' => $httpCode]);
        exit;
    }

    /**
     * Return JSON success response
     */
    public static function jsonSuccess($data = []) {
        header('Content-Type: application/json; charset=UTF-8');

        $response = array_merge(['status' => 'ok'], $data);
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
        exit;
    }

    /**
     * Handle exception
     */
    public static function handleException($e) {
        $message = $e->getMessage();
        $trace = DEBUG_MODE ? $e->getTraceAsString() : '';

        self::logError($message, [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $trace
        ]);

        self::jsonError($message, 'EXCEPTION', 500);
    }

    /**
     * Get error statistics
     */
    public static function getErrorStats() {
        return [
            'total' => count(self::$errors),
            'errors' => self::$errors
        ];
    }
}

// Initialize
ErrorHandler::init();
