<?php
include_once(__DIR__ . "/../config.php");
include_once(__DIR__ . "/../lib/DatabaseHelper.php");
include_once(__DIR__ . "/../lib/FormBuilder.php");

class MockCRUDEngine {
    private static $storagePath = '/tmp/mockdata/';

    public static function getMockPath($db, $table) {
        // Sanitize just in case
        $db = preg_replace('/[^a-zA-Z0-9_]/', '', $db);
        $table = preg_replace('/[^a-zA-Z0-9_]/', '', $table);
        return self::$storagePath . "{$db}.{$table}.json";
    }

    public static function loadMock($db, $table) {
        $path = self::getMockPath($db, $table);
        
        if (!file_exists($path)) {
            return self::seedMockFromRealDB($db, $table);
        }
        
        $json = file_get_contents($path);
        $data = json_decode($json, true);
        
        if (!is_array($data)) {
            // Corrupt file, re-seed
            return self::seedMockFromRealDB($db, $table);
        }
        
        return $data;
    }

    public static function seedMockFromRealDB($dbName, $tableName) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->connect();
        
        // Sanitize via helper (validation)
        $safeDb = $db->escapeIdentifier($dbName);
        $safeTable = $db->escapeIdentifier($tableName);
        $rawDb = trim($safeDb, '`');
        
        if (!$conn->select_db($rawDb)) {
            // If DB doesn't exist, we return empty array but don't crash
             return []; 
        }

        // Get 50 rows
        $sql = "SELECT * FROM $safeTable LIMIT 50";
        $result = $conn->query($sql);
        
        $rows = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $rows[] = $row;
            }
        }
        
        // Save to JSON
        self::saveMock($dbName, $tableName, $rows);
        
        return $rows;
    }

    public static function saveMock($db, $table, $data) {
        $path = self::getMockPath($db, $table);
        file_put_contents($path, json_encode($data, JSON_PRETTY_PRINT));
    }
    
    public static function deleteRow($db, $table, $id) {
        $rows = self::loadMock($db, $table);
        $newRows = array_filter($rows, function($r) use ($id) {
            // Check for various ID keys just in case
            $rowId = $r['id'] ?? $r['ID'] ?? $r['Id'] ?? null;
            return (string)$rowId !== (string)$id;
        });
        // Re-index array
        $newRows = array_values($newRows);
        self::saveMock($db, $table, $newRows);
        return $newRows;
    }
    
    public static function insertRow($db, $table, $row) {
        $rows = self::loadMock($db, $table);
        // Ensure ID
        if (!isset($row['id'])) {
            $row['id'] = DatabaseHelper::getInstance()->generateUUID();
        }
        // Add timestamp if columns exist and missing
        if (!isset($row['created_at']) && isset($rows[0]) && array_key_exists('created_at', $rows[0])) {
             $row['created_at'] = date('Y-m-d H:i:s');
        }
        
        $rows[] = $row;
        self::saveMock($db, $table, $rows);
        return $rows;
    }

    public static function updateRow($db, $table, $id, $newData) {
        $rows = self::loadMock($db, $table);
        foreach ($rows as &$r) {
            $rowId = $r['id'] ?? $r['ID'] ?? $r['Id'] ?? null;
            if ((string)$rowId === (string)$id) {
                // Merge updates
                $r = array_merge($r, $newData);
                break;
            }
        }
        self::saveMock($db, $table, $rows);
        return $rows;
    }
}
