<?php
/**
 * table_operations.php - Gate 6: The Replicator
 * Table management operations: copy, move, rename, delete
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/TableOperations.php';
require_once __DIR__ . '/lib/ErrorHandler.php';

// Get all databases for dropdown
$databases = TableOperations::getAllDatabases();
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>Gate 6: The Replicator</title>
<style>
    * { box-sizing: border-box; }

    body {
        background: radial-gradient(circle at top, #020d18 0%, #000000 70%);
        font-family: 'Consolas', 'Monaco', monospace;
        color: #e0f7ff;
        margin: 0;
        padding: 20px;
        min-height: 100vh;
    }

    .header {
        text-align: center;
        margin-bottom: 40px;
    }

    h1 {
        text-shadow: 0 0 12px #00eaff;
        font-weight: bold;
        margin: 0 0 10px 0;
        color: #e0f7ff;
        font-size: 2.5em;
    }

    .subtitle {
        color: #00ffaa;
        font-size: 1.2em;
        text-shadow: 0 0 8px #00ffaa;
    }

    .back-link {
        position: absolute;
        top: 20px;
        left: 20px;
        color: #00eaff;
        text-decoration: none;
        font-size: 1.2em;
        transition: all 0.2s;
    }

    .back-link:hover {
        text-shadow: 0 0 10px #00eaff;
        transform: translateX(-5px);
    }

    .container {
        max-width: 1400px;
        margin: 0 auto;
        display: grid;
        grid-template-columns: 350px 1fr;
        gap: 30px;
    }

    .panel {
        background: rgba(0, 10, 20, 0.6);
        border: 2px solid #00eaff33;
        border-radius: 12px;
        padding: 30px;
    }

    .panel-title {
        color: #00eaff;
        font-size: 1.5em;
        margin: 0 0 20px 0;
        text-shadow: 0 0 8px #00eaff;
        border-bottom: 1px solid #00eaff33;
        padding-bottom: 10px;
    }

    .form-group {
        margin-bottom: 20px;
    }

    label {
        display: block;
        color: #00ffaa;
        margin-bottom: 8px;
        font-weight: bold;
        text-shadow: 0 0 6px #00ffaa44;
    }

    select, input[type="text"] {
        width: 100%;
        padding: 12px;
        background: rgba(0, 30, 50, 0.8);
        border: 1px solid #00eaff44;
        border-radius: 6px;
        color: #e0f7ff;
        font-family: 'Consolas', monospace;
        font-size: 14px;
        transition: all 0.2s;
    }

    select:focus, input[type="text"]:focus {
        outline: none;
        border-color: #00eaff;
        box-shadow: 0 0 10px #00eaff44;
    }

    .table-info {
        background: rgba(0, 40, 60, 0.4);
        padding: 15px;
        border-radius: 6px;
        border: 1px solid #00eaff22;
        margin-top: 15px;
        font-size: 0.9em;
    }

    .table-info-row {
        display: flex;
        justify-content: space-between;
        padding: 5px 0;
        border-bottom: 1px solid #00eaff11;
    }

    .table-info-row:last-child {
        border-bottom: none;
    }

    .table-info-label {
        color: #8899aa;
    }

    .table-info-value {
        color: #00ffaa;
        font-weight: bold;
    }

    .operation-section {
        background: rgba(0, 40, 60, 0.3);
        border: 1px solid #00eaff22;
        border-radius: 8px;
        padding: 20px;
        margin-bottom: 20px;
    }

    .operation-section h3 {
        color: #00eaff;
        margin: 0 0 15px 0;
        font-size: 1.2em;
    }

    .checkbox-group {
        margin: 10px 0;
    }

    .checkbox-group label {
        display: flex;
        align-items: center;
        color: #8899aa;
        font-weight: normal;
        margin-bottom: 8px;
        cursor: pointer;
    }

    .checkbox-group input[type="checkbox"] {
        margin-right: 10px;
        width: auto;
        cursor: pointer;
    }

    button {
        background: transparent;
        color: #00eaff;
        border: 2px solid #00eaff;
        padding: 12px 24px;
        border-radius: 6px;
        font-weight: bold;
        font-family: 'Consolas', monospace;
        cursor: pointer;
        transition: all 0.2s;
        font-size: 14px;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    button:hover:not(:disabled) {
        background: #00eaff;
        color: #000;
        box-shadow: 0 0 15px #00eaff;
    }

    button:disabled {
        opacity: 0.3;
        cursor: not-allowed;
    }

    button.danger {
        border-color: #ff3366;
        color: #ff3366;
    }

    button.danger:hover:not(:disabled) {
        background: #ff3366;
        color: #fff;
        box-shadow: 0 0 15px #ff3366;
    }

    button.success {
        border-color: #00ffaa;
        color: #00ffaa;
    }

    button.success:hover:not(:disabled) {
        background: #00ffaa;
        color: #000;
        box-shadow: 0 0 15px #00ffaa;
    }

    .alert {
        padding: 15px;
        border-radius: 6px;
        margin-bottom: 20px;
        display: none;
    }

    .alert.success {
        background: rgba(0, 255, 170, 0.1);
        border: 1px solid #00ffaa;
        color: #00ffaa;
    }

    .alert.error {
        background: rgba(255, 51, 102, 0.1);
        border: 1px solid #ff3366;
        color: #ff3366;
    }

    .alert.show {
        display: block;
        animation: slideIn 0.3s ease;
    }

    @keyframes slideIn {
        from { opacity: 0; transform: translateY(-10px); }
        to { opacity: 1; transform: translateY(0); }
    }

    .loading {
        display: none;
        text-align: center;
        color: #00eaff;
        margin: 20px 0;
    }

    .loading.show {
        display: block;
    }

    .spinner {
        border: 3px solid rgba(0, 234, 255, 0.2);
        border-top: 3px solid #00eaff;
        border-radius: 50%;
        width: 40px;
        height: 40px;
        animation: spin 1s linear infinite;
        margin: 0 auto 10px;
    }

    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }

    @media (max-width: 900px) {
        .container {
            grid-template-columns: 1fr;
        }
    }
</style>
</head>
<body>

<a href="index.php" class="back-link">← Back to Gates</a>

<div class="header">
    <h1>🔄 Gate 6: The Replicator</h1>
    <div class="subtitle">Clone, migrate, and orchestrate table operations</div>
</div>

<div class="container">
    <!-- Left Panel: Table Selector -->
    <div class="panel">
        <h2 class="panel-title">Source Table</h2>

        <div class="form-group">
            <label for="sourceDatabase">Database</label>
            <select id="sourceDatabase">
                <option value="">-- Select Database --</option>
                <?php foreach ($databases as $db): ?>
                    <option value="<?= htmlspecialchars($db) ?>"><?= htmlspecialchars($db) ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="sourceTable">Table</label>
            <select id="sourceTable" disabled>
                <option value="">-- Select Database First --</option>
            </select>
        </div>

        <div id="tableInfo" class="table-info" style="display: none;">
            <div class="table-info-row">
                <span class="table-info-label">Rows:</span>
                <span class="table-info-value" id="infoRows">-</span>
            </div>
            <div class="table-info-row">
                <span class="table-info-label">Engine:</span>
                <span class="table-info-value" id="infoEngine">-</span>
            </div>
            <div class="table-info-row">
                <span class="table-info-label">Size:</span>
                <span class="table-info-value" id="infoSize">-</span>
            </div>
            <div class="table-info-row">
                <span class="table-info-label">Collation:</span>
                <span class="table-info-value" id="infoCollation">-</span>
            </div>
        </div>
    </div>

    <!-- Right Panel: Operations Dashboard -->
    <div class="panel">
        <h2 class="panel-title">Operations</h2>

        <div id="alertBox" class="alert"></div>
        <div id="loadingBox" class="loading">
            <div class="spinner"></div>
            <div>Processing operation...</div>
        </div>

        <!-- Copy Table Operation -->
        <div class="operation-section">
            <h3>📋 Copy Table</h3>

            <div class="form-group">
                <label for="copyTargetDb">Target Database</label>
                <select id="copyTargetDb">
                    <option value="">-- Select Database --</option>
                    <?php foreach ($databases as $db): ?>
                        <option value="<?= htmlspecialchars($db) ?>"><?= htmlspecialchars($db) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="form-group">
                <label for="copyTargetTable">Target Table Name</label>
                <input type="text" id="copyTargetTable" placeholder="new_table_name" />
            </div>

            <div class="checkbox-group">
                <label>
                    <input type="checkbox" id="copyStructure" checked /> Copy structure
                </label>
                <label>
                    <input type="checkbox" id="copyData" checked /> Copy data
                </label>
                <label>
                    <input type="checkbox" id="copyDropTable" /> Add DROP TABLE
                </label>
                <label>
                    <input type="checkbox" id="copyAutoIncrement" checked /> Add AUTO_INCREMENT value
                </label>
            </div>

            <button id="btnCopy" onclick="performCopy()">Copy Table</button>
        </div>

        <!-- Move Table Operation -->
        <div class="operation-section">
            <h3>🚚 Move Table</h3>

            <div class="form-group">
                <label for="moveTargetDb">Target Database</label>
                <select id="moveTargetDb">
                    <option value="">-- Select Database --</option>
                    <?php foreach ($databases as $db): ?>
                        <option value="<?= htmlspecialchars($db) ?>"><?= htmlspecialchars($db) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="form-group">
                <label for="moveTargetTable">Target Table Name</label>
                <input type="text" id="moveTargetTable" placeholder="new_table_name" />
            </div>

            <button id="btnMove" class="danger" onclick="performMove()">Move Table (Delete Source)</button>
        </div>

        <!-- Rename Table Operation -->
        <div class="operation-section">
            <h3>✏️ Rename Table</h3>

            <div class="form-group">
                <label for="renameNewName">New Table Name</label>
                <input type="text" id="renameNewName" placeholder="new_name" />
            </div>

            <button id="btnRename" onclick="performRename()">Rename Table</button>
        </div>

        <!-- Delete Table Operation -->
        <div class="operation-section">
            <h3>🗑️ Delete Table</h3>

            <p style="color: #ff3366; margin: 0 0 15px 0;">⚠️ This action cannot be undone!</p>

            <button id="btnDelete" class="danger" onclick="performDelete()">Delete Table</button>
        </div>

        <!-- Export Structure -->
        <div class="operation-section">
            <h3>💾 Export Structure</h3>

            <div class="checkbox-group">
                <label>
                    <input type="checkbox" id="exportDropTable" checked /> Add DROP TABLE
                </label>
                <label>
                    <input type="checkbox" id="exportAutoIncrement" /> Add AUTO_INCREMENT value
                </label>
            </div>

            <button id="btnExport" class="success" onclick="performExport()">Download SQL</button>
        </div>
    </div>
</div>

<script>
// Load tables when database is selected
document.getElementById('sourceDatabase').addEventListener('change', async function() {
    const database = this.value;
    const tableSelect = document.getElementById('sourceTable');
    const tableInfo = document.getElementById('tableInfo');

    tableInfo.style.display = 'none';
    tableSelect.disabled = true;
    tableSelect.innerHTML = '<option value="">Loading...</option>';

    if (!database) {
        tableSelect.innerHTML = '<option value="">-- Select Database First --</option>';
        return;
    }

    try {
        const response = await fetch(`api/tableOperations.php?action=getTables&database=${encodeURIComponent(database)}`);
        const data = await response.json();

        if (data.success) {
            tableSelect.innerHTML = '<option value="">-- Select Table --</option>';
            data.tables.forEach(table => {
                const option = document.createElement('option');
                option.value = table;
                option.textContent = table;
                tableSelect.appendChild(option);
            });
            tableSelect.disabled = false;
        } else {
            showAlert('Error loading tables: ' + data.message, 'error');
        }
    } catch (error) {
        showAlert('Failed to load tables: ' + error.message, 'error');
    }
});

// Load table info when table is selected
document.getElementById('sourceTable').addEventListener('change', async function() {
    const database = document.getElementById('sourceDatabase').value;
    const table = this.value;
    const tableInfo = document.getElementById('tableInfo');

    if (!table) {
        tableInfo.style.display = 'none';
        return;
    }

    try {
        const response = await fetch(`api/tableOperations.php?action=getTableInfo&database=${encodeURIComponent(database)}&table=${encodeURIComponent(table)}`);
        const data = await response.json();

        if (data.success) {
            document.getElementById('infoRows').textContent = data.info.row_count.toLocaleString();
            document.getElementById('infoEngine').textContent = data.info.engine;
            document.getElementById('infoSize').textContent = formatBytes(parseInt(data.info.data_length) + parseInt(data.info.index_length));
            document.getElementById('infoCollation').textContent = data.info.collation;
            tableInfo.style.display = 'block';

            // Auto-fill target table names
            document.getElementById('copyTargetTable').value = table;
            document.getElementById('moveTargetTable').value = table;
            document.getElementById('renameNewName').value = table + '_renamed';
        }
    } catch (error) {
        console.error('Failed to load table info:', error);
    }
});

function formatBytes(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

function showAlert(message, type) {
    const alertBox = document.getElementById('alertBox');
    alertBox.textContent = message;
    alertBox.className = 'alert ' + type + ' show';

    setTimeout(() => {
        alertBox.classList.remove('show');
    }, 5000);
}

function showLoading(show) {
    document.getElementById('loadingBox').className = show ? 'loading show' : 'loading';
}

async function performCopy() {
    const sourceDb = document.getElementById('sourceDatabase').value;
    const sourceTable = document.getElementById('sourceTable').value;
    const targetDb = document.getElementById('copyTargetDb').value;
    const targetTable = document.getElementById('copyTargetTable').value;

    if (!sourceDb || !sourceTable || !targetDb || !targetTable) {
        showAlert('Please fill in all required fields', 'error');
        return;
    }

    const options = {
        structure: document.getElementById('copyStructure').checked,
        data: document.getElementById('copyData').checked,
        drop_table: document.getElementById('copyDropTable').checked,
        add_auto_increment: document.getElementById('copyAutoIncrement').checked
    };

    if (!options.structure && !options.data) {
        showAlert('Please select at least structure or data', 'error');
        return;
    }

    showLoading(true);

    try {
        const response = await fetch('api/tableOperations.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                action: 'copy',
                sourceDb, sourceTable, targetDb, targetTable, options
            })
        });

        const data = await response.json();
        showLoading(false);

        if (data.success) {
            showAlert(data.message, 'success');
        } else {
            showAlert(data.message, 'error');
        }
    } catch (error) {
        showLoading(false);
        showAlert('Operation failed: ' + error.message, 'error');
    }
}

async function performMove() {
    const sourceDb = document.getElementById('sourceDatabase').value;
    const sourceTable = document.getElementById('sourceTable').value;
    const targetDb = document.getElementById('moveTargetDb').value;
    const targetTable = document.getElementById('moveTargetTable').value;

    if (!sourceDb || !sourceTable || !targetDb || !targetTable) {
        showAlert('Please fill in all required fields', 'error');
        return;
    }

    if (!confirm(`⚠️ This will DELETE the source table "${sourceDb}.${sourceTable}" after copying. Continue?`)) {
        return;
    }

    showLoading(true);

    try {
        const response = await fetch('api/tableOperations.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                action: 'move',
                sourceDb, sourceTable, targetDb, targetTable
            })
        });

        const data = await response.json();
        showLoading(false);

        if (data.success) {
            showAlert(data.message, 'success');
            // Reload tables list
            document.getElementById('sourceDatabase').dispatchEvent(new Event('change'));
        } else {
            showAlert(data.message, 'error');
        }
    } catch (error) {
        showLoading(false);
        showAlert('Operation failed: ' + error.message, 'error');
    }
}

async function performRename() {
    const database = document.getElementById('sourceDatabase').value;
    const oldName = document.getElementById('sourceTable').value;
    const newName = document.getElementById('renameNewName').value;

    if (!database || !oldName || !newName) {
        showAlert('Please fill in all required fields', 'error');
        return;
    }

    showLoading(true);

    try {
        const response = await fetch('api/tableOperations.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                action: 'rename',
                database, oldName, newName
            })
        });

        const data = await response.json();
        showLoading(false);

        if (data.success) {
            showAlert(data.message, 'success');
            // Reload tables list
            document.getElementById('sourceDatabase').dispatchEvent(new Event('change'));
        } else {
            showAlert(data.message, 'error');
        }
    } catch (error) {
        showLoading(false);
        showAlert('Operation failed: ' + error.message, 'error');
    }
}

async function performDelete() {
    const database = document.getElementById('sourceDatabase').value;
    const tableName = document.getElementById('sourceTable').value;

    if (!database || !tableName) {
        showAlert('Please select a table first', 'error');
        return;
    }

    if (!confirm(`⚠️ DANGER: This will permanently DELETE the table "${database}.${tableName}". This action CANNOT be undone!\n\nType the table name to confirm: ${tableName}`)) {
        return;
    }

    const confirmName = prompt(`Type "${tableName}" to confirm deletion:`);
    if (confirmName !== tableName) {
        showAlert('Table name does not match. Deletion cancelled.', 'error');
        return;
    }

    showLoading(true);

    try {
        const response = await fetch('api/tableOperations.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                action: 'delete',
                database, tableName
            })
        });

        const data = await response.json();
        showLoading(false);

        if (data.success) {
            showAlert(data.message, 'success');
            // Reload tables list
            document.getElementById('sourceTable').value = '';
            document.getElementById('tableInfo').style.display = 'none';
            document.getElementById('sourceDatabase').dispatchEvent(new Event('change'));
        } else {
            showAlert(data.message, 'error');
        }
    } catch (error) {
        showLoading(false);
        showAlert('Operation failed: ' + error.message, 'error');
    }
}

async function performExport() {
    const database = document.getElementById('sourceDatabase').value;
    const tableName = document.getElementById('sourceTable').value;

    if (!database || !tableName) {
        showAlert('Please select a table first', 'error');
        return;
    }

    const options = {
        drop_table: document.getElementById('exportDropTable').checked,
        add_auto_increment: document.getElementById('exportAutoIncrement').checked
    };

    showLoading(true);

    try {
        const response = await fetch('api/tableOperations.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                action: 'export',
                database, tableName, options
            })
        });

        const data = await response.json();
        showLoading(false);

        if (data.success) {
            // Download SQL file
            const blob = new Blob([data.sql], {type: 'text/plain'});
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = data.filename;
            a.click();
            window.URL.revokeObjectURL(url);

            showAlert('SQL file downloaded successfully!', 'success');
        } else {
            showAlert(data.message, 'error');
        }
    } catch (error) {
        showLoading(false);
        showAlert('Export failed: ' + error.message, 'error');
    }
}
</script>

</body>
</html>
