<?php
/**
 * Test European Number Format Detection and Conversion
 * Tests the fix for European decimal separator (comma) vs US (period)
 */

require_once __DIR__ . '/lib/DataValidator.php';

echo "========================================\n";
echo "European Number Format Detection Test\n";
echo "========================================\n\n";

// Test cases: [input, expected_output, description, precision]
$testCases = [
    // European format (comma as decimal separator)
    ['17,012', '17.0120', 'European: Simple decimal with comma (4 decimals like interchange)', '10,4'],
    ['1.234,56', '1234.56', 'European: Thousands with period, decimal with comma'],
    ['10.234.567,89', '10234567.89', 'European: Multiple thousand separators'],
    ['0,5', '0.5', 'European: Zero with comma decimal'],

    // US format (period as decimal separator)
    ['17.012', '17.012', 'US: Simple decimal with period'],
    ['1,234.56', '1234.56', 'US: Thousands with comma, decimal with period'],
    ['10,234,567.89', '10234567.89', 'US: Multiple thousand separators'],
    ['0.5', '0.5', 'US: Zero with period decimal'],

    // Edge cases
    ['17', '17', 'No decimal separator'],
    ['$1,234.56', '1234.56', 'US with currency symbol'],
    ['€1.234,56', '1234.56', 'European with currency symbol'],
    ['1082,09', '1082.09', 'European: Value from screenshot'],
    ['467,33', '467.33', 'European: Another screenshot value'],
    ['1633,38', '1633.38', 'European: Another screenshot value'],
];

$passedCount = 0;
$failedCount = 0;

foreach ($testCases as $index => $testCase) {
    // Handle optional precision parameter (default to 10,2)
    if (count($testCase) === 4) {
        list($input, $expected, $description, $precision) = $testCase;
    } else {
        list($input, $expected, $description) = $testCase;
        $precision = '10,2';
    }

    // Run validation
    $result = DataValidator::validate($input, 'DECIMAL', $precision, true);
    $actual = $result['value'];

    // Compare (normalize to float for comparison)
    $expectedFloat = (float)$expected;
    $actualFloat = (float)$actual;

    $passed = (abs($expectedFloat - $actualFloat) < 0.001); // Floating point tolerance

    if ($passed) {
        echo "✅ PASS: $description\n";
        echo "   Input: '$input' → Output: '$actual' (Expected: '$expected')\n";
        $passedCount++;
    } else {
        echo "❌ FAIL: $description\n";
        echo "   Input: '$input' → Output: '$actual' (Expected: '$expected')\n";
        $failedCount++;
    }

    if ($result['corrected']) {
        echo "   ℹ️  Correction: {$result['reason']}\n";
    }
    echo "\n";
}

echo "========================================\n";
echo "Test Results\n";
echo "========================================\n";
echo "Passed: $passedCount\n";
echo "Failed: $failedCount\n";
echo "Total:  " . ($passedCount + $failedCount) . "\n\n";

if ($failedCount === 0) {
    echo "🎉 All tests passed!\n";
    exit(0);
} else {
    echo "⚠️  Some tests failed. Please review.\n";
    exit(1);
}
