<?php
/**
 * Test Excel Cell Value Reading
 * Diagnose getValue() vs getFormattedValue() for European numbers
 */

require_once __DIR__ . '/vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

echo "========================================\n";
echo "Excel Cell Value Reading Test\n";
echo "========================================\n\n";

// Create a test spreadsheet with different number formats
$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();

// Test data
$sheet->setCellValue('A1', 'Description');
$sheet->setCellValue('B1', 'getValue()');
$sheet->setCellValue('C1', 'getFormattedValue()');
$sheet->setCellValue('D1', 'getDataType()');

$testCases = [
    ['Plain number', 17.012],
    ['Large number', 20230.0],
    ['Decimal', 178.3477],
    ['Small decimal', 8.8160],
];

$row = 2;
foreach ($testCases as $test) {
    [$desc, $value] = $test;

    // Set the value
    $sheet->setCellValue("A$row", $desc);
    $sheet->setCellValue("B$row", $value);

    // Apply European number format with comma as decimal separator
    $sheet->getStyle("B$row")->getNumberFormat()->setFormatCode('#.##0,00');

    $row++;
}

// Save to temp file
$tempFile = '/tmp/test_european_numbers.xlsx';
$writer = new Xlsx($spreadsheet);
$writer->save($tempFile);

echo "Created test Excel file: $tempFile\n\n";

// Now read it back
$reader = IOFactory::createReaderForFile($tempFile);
$loadedSpreadsheet = $reader->load($tempFile);
$loadedSheet = $loadedSpreadsheet->getActiveSheet();

echo "Reading back the Excel file:\n";
echo str_repeat("=", 80) . "\n";

for ($row = 2; $row <= 5; $row++) {
    $desc = $loadedSheet->getCell("A$row")->getValue();
    $cell = $loadedSheet->getCell("B$row");

    $rawValue = $cell->getValue();
    $formattedValue = $cell->getFormattedValue();
    $dataType = $cell->getDataType();
    $cellFormat = $cell->getStyle()->getNumberFormat()->getFormatCode();

    echo "\nRow $row: $desc\n";
    echo "  getValue():          " . var_export($rawValue, true) . " (type: " . gettype($rawValue) . ")\n";
    echo "  getFormattedValue(): " . var_export($formattedValue, true) . "\n";
    echo "  getDataType():       $dataType\n";
    echo "  Number Format:       $cellFormat\n";

    // Test our European detection
    require_once __DIR__ . '/lib/DataValidator.php';

    if (is_string($formattedValue)) {
        $result = DataValidator::validate($formattedValue, 'DECIMAL', '10,2', true);
        echo "  After validation:    " . $result['value'] . "\n";
        echo "  Corrected?:          " . ($result['corrected'] ? 'YES' : 'NO') . "\n";
        if ($result['reason']) {
            echo "  Reason:              " . $result['reason'] . "\n";
        }
    }
}

echo "\n" . str_repeat("=", 80) . "\n";
echo "\n🔍 Key Findings:\n";
echo "  • getValue() returns the RAW numeric value (no formatting)\n";
echo "  • getFormattedValue() returns the DISPLAYED string (with European comma)\n";
echo "  • For European number format cells, we NEED getFormattedValue()\n";

// Clean up
unlink($tempFile);

echo "\n✅ Test complete!\n";
