<?php
/**
 * Test Skip Mode with ALTER TABLE
 * Simulates uploading a file with more columns than existing table has
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/DatabaseHelper.php';
require_once __DIR__ . '/lib/ErrorHandler.php';

// Include the helper functions from insert.php
function getTableColumns($conn, $tableName) {
    $safeTableName = $conn->real_escape_string($tableName);
    $result = $conn->query("DESCRIBE `$safeTableName`");

    if (!$result) {
        return [];
    }

    $columns = [];
    while ($row = $result->fetch_assoc()) {
        $columns[] = $row['Field'];
    }
    return $columns;
}

function sqlValue($value, $conn) {
    if ($value === null) {
        return 'NULL';
    }
    if (is_numeric($value)) {
        return $value;
    }
    return "'" . $conn->real_escape_string($value) . "'";
}

function generateAlterTableSQL($tableName, $col, $conn) {
    $colName = $conn->real_escape_string($col['name']);
    $baseType = strtoupper($col['type']);
    $colType = $baseType;

    // Add length for VARCHAR/DECIMAL
    if ($baseType === 'VARCHAR' && !empty($col['length'])) {
        $colType .= '(' . $col['length'] . ')';
    } elseif ($baseType === 'DECIMAL' && !empty($col['length'])) {
        $colType .= '(' . $col['length'] . ')';
    }

    // Add character set for string types
    $charset = '';
    if (in_array($baseType, ['VARCHAR', 'TEXT', 'CHAR'])) {
        $charset = ' CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci';
    }

    // Add NULL/NOT NULL
    $nullable = $col['nullable'] ? 'NULL' : 'NOT NULL';

    // Add default value if specified
    $default = '';
    if (isset($col['default_value'])) {
        $default = ' DEFAULT ' . sqlValue($col['default_value'], $conn);
    }

    // Add comment
    $comment = '';
    if (!empty($col['comment'])) {
        $safeComment = $conn->real_escape_string($col['comment']);
        $comment = " COMMENT '$safeComment'";
    }

    return "ALTER TABLE `$tableName` ADD COLUMN `$colName` $colType$charset $nullable$default$comment";
}

echo "=== Testing Skip Mode with ALTER TABLE ===\n\n";

try {
    $db = DatabaseHelper::getInstance();
    $conn = $db->getConnection();
    $conn->select_db('importer_db');

    $tableName = 'airbnb_test_skip_mode';

    // Simulate full 21-column schema (saved from previous import)
    $fullSchema = [
        ['name' => 'fecha', 'type' => 'DATE', 'nullable' => false],
        ['name' => 'tipo', 'type' => 'VARCHAR', 'length' => '50', 'nullable' => false],
        ['name' => 'codigo_de_confirmacion', 'type' => 'VARCHAR', 'length' => '50', 'nullable' => true],
        ['name' => 'huesped', 'type' => 'VARCHAR', 'length' => '100', 'nullable' => true],
        ['name' => 'espacio', 'type' => 'VARCHAR', 'length' => '100', 'nullable' => true],
        ['name' => 'moneda', 'type' => 'VARCHAR', 'length' => '10', 'nullable' => false],
        ['name' => 'monto', 'type' => 'DECIMAL', 'length' => '10,2', 'nullable' => true],
        ['name' => 'ingresos_recibidos', 'type' => 'DECIMAL', 'length' => '10,2', 'nullable' => true],
        // Missing columns that need to be added:
        ['name' => 'fecha_de_llegada_estimada', 'type' => 'DATE', 'nullable' => true, 'default_value' => null],
        ['name' => 'fecha_de_la_reservacion', 'type' => 'DATE', 'nullable' => true, 'default_value' => null],
        ['name' => 'fecha_de_inicio', 'type' => 'DATE', 'nullable' => true, 'default_value' => null],
        ['name' => 'fecha_de_finalizacion', 'type' => 'DATE', 'nullable' => true, 'default_value' => null],
        ['name' => 'noches', 'type' => 'INT', 'nullable' => true, 'default_value' => 0],
        ['name' => 'detalles', 'type' => 'VARCHAR', 'length' => '100', 'nullable' => true, 'default_value' => ''],
        ['name' => 'codigo_de_referencia', 'type' => 'VARCHAR', 'length' => '50', 'nullable' => true, 'default_value' => ''],
        ['name' => 'tarifa_de_servicio', 'type' => 'DECIMAL', 'length' => '10,2', 'nullable' => true, 'default_value' => '0.00'],
        ['name' => 'tarifa_por_pago_rapido', 'type' => 'DECIMAL', 'length' => '10,2', 'nullable' => true, 'default_value' => '0.00'],
        ['name' => 'tarifa_de_limpieza', 'type' => 'DECIMAL', 'length' => '10,2', 'nullable' => true, 'default_value' => '0.00'],
        ['name' => 'ingresos_brutos', 'type' => 'DECIMAL', 'length' => '10,2', 'nullable' => true, 'default_value' => '0.00'],
        ['name' => 'impuestos_sobe_el_alojamiento', 'type' => 'DECIMAL', 'length' => '10,2', 'nullable' => true, 'default_value' => '0.00'],
        ['name' => 'ano_de_ingresos', 'type' => 'INT', 'nullable' => true, 'default_value' => 0]
    ];

    echo "1. Get existing table columns...\n";
    $existingColumns = getTableColumns($conn, $tableName);
    echo "   ✅ Table has " . count($existingColumns) . " columns\n\n";

    echo "2. Find missing columns...\n";
    $missingColumns = [];
    foreach ($fullSchema as $col) {
        if (!in_array($col['name'], $existingColumns)) {
            $missingColumns[] = $col;
        }
    }
    echo "   ✅ Found " . count($missingColumns) . " missing columns:\n";
    foreach ($missingColumns as $col) {
        echo "      - {$col['name']} ({$col['type']})\n";
    }
    echo "\n";

    echo "3. Generate and execute ALTER TABLE statements...\n";
    foreach ($missingColumns as $col) {
        $alterSQL = generateAlterTableSQL($tableName, $col, $conn);
        echo "   📝 {$alterSQL}\n";

        if ($conn->query($alterSQL)) {
            echo "   ✅ Column '{$col['name']}' added successfully\n\n";
        } else {
            echo "   ❌ Failed to add column '{$col['name']}': {$conn->error}\n\n";
        }
    }

    echo "4. Verify final table structure...\n";
    $finalColumns = getTableColumns($conn, $tableName);
    echo "   ✅ Table now has " . count($finalColumns) . " columns\n\n";

    echo "=== Test Complete ===\n";
    echo "Expected: 21 columns (8 original + 13 added)\n";
    echo "Actual: " . count($finalColumns) . " columns\n";

    if (count($finalColumns) === 21) {
        echo "\n🎉 SUCCESS! All columns added correctly!\n";
    } else {
        echo "\n❌ MISMATCH! Expected 21 but got " . count($finalColumns) . "\n";
    }

} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}
