<?php
/**
 * Test Script: Spanish Canonical Field CRUD Lifecycle
 * Tests INSERT, SELECT, UPDATE, DELETE on productos table with Spanish schema
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/DatabaseHelper.php';
require_once __DIR__ . '/lib/SchemaConventions.php';

echo "========================================\n";
echo "Spanish Canonical Field CRUD Test\n";
echo "========================================\n\n";

$tableName = 'productos';
$dbName = 'mrw';

try {
    // TEST 1: INSERT
    echo "[1] Testing INSERT...\n";
    $testData = [
        'nombre' => 'Laptop Dell XPS 15',
        'precio' => 1299.99,
        'descripcion' => 'High-performance laptop with 16GB RAM'
    ];

    $insertedId = DatabaseHelper::insertRow($tableName, $testData, $dbName);
    echo "✓ Record inserted with ID: {$insertedId}\n\n";

    // TEST 2: SELECT (getRow)
    echo "[2] Testing SELECT by ID...\n";
    $record = DatabaseHelper::getRow($tableName, $insertedId, $dbName);

    if ($record) {
        echo "✓ Record retrieved successfully\n";
        echo "  - producto_id: " . $record['producto_id'] . "\n";
        echo "  - nombre: " . $record['nombre'] . "\n";
        echo "  - precio: " . $record['precio'] . "\n";
        echo "  - alta_db: " . $record['alta_db'] . "\n";
        echo "  - alta_por: " . $record['alta_por'] . "\n";
        echo "  - ultimo_cambio: " . ($record['ultimo_cambio'] ?? 'NULL') . "\n";
        echo "  - ultimo_cambio_por: " . ($record['ultimo_cambio_por'] ?? 'NULL') . "\n\n";
    } else {
        throw new Exception("Failed to retrieve record");
    }

    // TEST 3: UPDATE
    echo "[3] Testing UPDATE...\n";
    $updateData = [
        'nombre' => 'Laptop Dell XPS 15 (Updated)',
        'precio' => 1199.99,
        'descripcion' => 'High-performance laptop with 16GB RAM - ON SALE!'
    ];

    DatabaseHelper::updateRow($tableName, $insertedId, $updateData, $dbName);
    echo "✓ Record updated successfully\n\n";

    // TEST 4: SELECT again to verify update and ultimo_cambio
    echo "[4] Verifying UPDATE (checking ultimo_cambio)...\n";
    $updatedRecord = DatabaseHelper::getRow($tableName, $insertedId, $dbName);

    if ($updatedRecord) {
        echo "✓ Updated record retrieved\n";
        echo "  - nombre: " . $updatedRecord['nombre'] . "\n";
        echo "  - precio: " . $updatedRecord['precio'] . "\n";
        echo "  - alta_db: " . $updatedRecord['alta_db'] . "\n";
        echo "  - ultimo_cambio: " . ($updatedRecord['ultimo_cambio'] ?? 'NULL') . "\n";
        echo "  - ultimo_cambio_por: " . ($updatedRecord['ultimo_cambio_por'] ?? 'NULL') . "\n";

        if ($updatedRecord['ultimo_cambio'] && $updatedRecord['ultimo_cambio_por']) {
            echo "✓ Audit fields (ultimo_cambio, ultimo_cambio_por) populated correctly\n\n";
        } else {
            echo "✗ WARNING: ultimo_cambio or ultimo_cambio_por not populated\n\n";
        }
    } else {
        throw new Exception("Failed to retrieve updated record");
    }

    // TEST 5: DELETE
    echo "[5] Testing DELETE...\n";
    DatabaseHelper::deleteRow($tableName, $insertedId, $dbName);
    echo "✓ Record deleted successfully\n\n";

    // TEST 6: Verify deletion
    echo "[6] Verifying DELETE...\n";
    $deletedRecord = DatabaseHelper::getRow($tableName, $insertedId, $dbName);

    if (!$deletedRecord) {
        echo "✓ Record no longer exists (deletion confirmed)\n\n";
    } else {
        throw new Exception("Record still exists after deletion!");
    }

    // TEST 7: Test SchemaConventions
    echo "[7] Testing SchemaConventions...\n";
    $pkField = SchemaConventions::getPrimaryKeyName($tableName);
    echo "✓ Primary key for 'productos': {$pkField}\n";

    $singularForm = SchemaConventions::singularize($tableName);
    echo "✓ Singular form of 'productos': {$singularForm}\n";

    $standardFields = SchemaConventions::getStandardFields($tableName, $dbName);
    echo "✓ Standard fields: " . implode(', ', array_keys($standardFields)) . "\n";

    $isStandard = SchemaConventions::isStandardField('alta_db');
    echo "✓ Is 'alta_db' a standard field? " . ($isStandard ? 'YES' : 'NO') . "\n";

    $mode = SchemaConventions::detectFieldMode($tableName, $dbName);
    echo "✓ Field mode for 'productos': {$mode}\n\n";

    echo "========================================\n";
    echo "✓ ALL TESTS PASSED!\n";
    echo "========================================\n\n";
    echo "Summary:\n";
    echo "- INSERT with auto-population of alta_db, alta_por ✓\n";
    echo "- SELECT by dynamic primary key (producto_id) ✓\n";
    echo "- UPDATE with auto-population of ultimo_cambio, ultimo_cambio_por ✓\n";
    echo "- DELETE using dynamic primary key ✓\n";
    echo "- SchemaConventions API working correctly ✓\n";

} catch (Exception $e) {
    echo "\n✗ TEST FAILED: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}
