# Agent Coding Guidelines

This document provides guidelines for agentic coding agents working in this repository.

## Build, Lint, and Test Commands

### PHP Execution
```bash
# Run a single PHP script (local)
php /lamp/www/importer/script_name.php

# Run with specific PHP binary
/lamp/php/bin/php /lamp/www/importer/script_name.php

# Web execution (remote testing)
curl -s https://dev-app.filemonprime.net/importer/script_name.php
```

### Database Commands
```bash
# MySQL CLI connection
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock

# Execute query directly
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock -e "SHOW DATABASES;"

# Connect to specific database
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock importer_db
```

### Running Tests
There is no formal test framework ( PHPUnit ). Tests are standalone PHP scripts located in:
- `/lamp/www/importer/tests/` - Demo and test files
- `/lamp/www/importer/test_*.php` - Test files in root

Run a test file directly:
```bash
php /lamp/www/importer/tests/test_intelligent_schema_detector.php
php /lamp/www/importer/test_import_flow.php
```

## Code Style Guidelines

### General Principles
- **Framework-less PHP 8**: No external frameworks; use raw PHP
- **Security-first**: Validate all inputs, escape all outputs for SQL
- **Defensive programming**: Always check return values and handle edge cases

### File Structure
- Use `<?php` opening tag (no closing `?>` at EOF to prevent whitespace issues)
- Every file should have a docblock at the top describing its purpose
- Class files: One class per file, filename matches class name
- Include paths: Use `require_once __DIR__ . '/relative/path.php';`

### Naming Conventions
| Element | Convention | Example |
|---------|------------|---------|
| Classes | PascalCase | `ImportLogger`, `DatabaseHelper` |
| Functions | snake_case | `getDefaultValueForType()` |
| Constants | UPPER_SNAKE_CASE | `DB_HOST`, `MAX_FILE_SIZE` |
| Variables | snake_case | `$import_log_id`, `$table_name` |
| Table names | snake_case | `import_logs`, `alumnos_becados` |
| Column names | snake_case | `alta_db`, `ultima_actualizacion` |
| Primary keys | Dynamic (table-based) | `propietario_id` for `propietarios` |

### Code Formatting
- Indentation: 4 spaces (no tabs)
- Line endings: Unix-style (LF)
- Opening brace for classes/functions on same line
- Space after control flow keywords: `if ($result) {` not `if($result){`
- Use parentheses in conditionals even when single condition

### Type Handling
- No type declarations (project uses PHP 7.x style)
- Manual type checking when needed
- Use `===` for strict comparisons
- Use `==` only when loose comparison is intentional

### Error Handling
- Use try-catch blocks for operations that may throw exceptions
- Always log errors using `ErrorHandler::logError($message, $context)`
- Check query results: `if (!$result) { handle error; }`
- Check connection: `if (!$conn) { handle error; }`
- Never suppress errors with `@` operator
- Use exceptions for unrecoverable errors, return false for recoverable

### Database Patterns
- Use prepared statements for all queries with user input
- Escape table/column names with backticks: `` `$tableName` ``
- Escape string values with `$conn->real_escape_string()`
- Use transactions for multi-row inserts: `$conn->begin_transaction()`, `$conn->commit()`
- Always close connections or use DatabaseHelper singleton pattern

### Session and State
- Call `session_start()` at the top of scripts that use sessions
- Store metadata in `$_SESSION` for multi-step operations
- Clear session data after use: `unset($_SESSION['key'])`

### Include Order
1. `<?php` declaration
2. File docblock
3. `session_start()` (if needed)
4. `require_once` statements (alphabetical within groups)
5. Constants definitions
6. Class definitions
7. Function definitions
8. Main execution code

### Comments and Documentation
- Use docblocks for classes and public methods
- Comment complex business logic
- Comment SQL queries that are non-obvious
- Use Spanish/English mixed comments where appropriate (project uses Spanish terminology like `alta_db`, `ultimo_cambio`)

### Common Patterns
- **Singleton for database**: Use `DatabaseHelper::getInstance()->getConnection()`
- **Static helper classes**: Classes like `ErrorHandler`, `ImportLogger` use static methods
- **Schema conventions**: Use `SchemaConventions::getStandardFields($tableName)` for canonical fields
- **Standard fields**: Every table gets `alta_db`, `alta_por`, `ultimo_cambio`, `ultimo_cambio_por`

### Database Schema Conventions
- Primary key: Dynamic, table-specific (e.g., `propietario_id` for `propietarios`)
- Standard timestamps: `TIMESTAMP` with `DEFAULT CURRENT_TIMESTAMP`
- User tracking: `VARCHAR(32)` for `alta_por`, `ultimo_cambio_por`
- Character set: `utf8mb4` with `utf8mb4_unicode_ci` collation
- Storage engine: `InnoDB` for transaction support

### Forbidden Patterns
- ❌ Never use `as any`, `@ts-ignore` (not applicable to PHP)
- ❌ Never commit directly without explicit request
- ❌ Never leave code in broken state after failures
- ❌ Never use empty catch blocks: `catch (Exception $e) {}`
- ❌ Never use variables for table/column names in SQL without escaping
- ❌ Never store passwords in plaintext in code (use config.php constants)
