# Schema Builder - Length Input Auto-Clear Fix

## Problem Summary

When changing a column type from one that uses length (e.g., `VARCHAR(50)`) to one that doesn't (e.g., `DATE`), the length input field would remain visible with the old value even though it was disabled.

### Example Issue

**Before Fix:**
1. Column is set to `VARCHAR` with length `50`
2. User changes type to `DATE`
3. Length input shows `50` (disabled and grayed out) ❌
4. Confusing because DATE doesn't use length

**After Fix:**
1. Column is set to `VARCHAR` with length `50`
2. User changes type to `DATE`
3. Length input is **cleared** and disabled ✅
4. Clean and intuitive UI

---

## Root Cause

The `updateSchema()` function was **disabling** the length input for types that don't use it, but **not clearing the value**.

**Original Code (Line 652-658):**
```javascript
// Disable length input for TEXT, DATE, etc.
if (field === 'type') {
    const row = document.getElementById("schemaTableBody").rows[idx];
    const lengthInput = row.cells[2].querySelector('input');
    const noLengthTypes = ['TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'INT'];
    lengthInput.disabled = noLengthTypes.includes(value);  // ❌ Only disables, doesn't clear
}
```

---

## Solution Implemented

Updated the logic to **clear both the UI and the data model** when switching to a type that doesn't use length.

**New Code:**
```javascript
// Handle length input for different types
if (field === 'type') {
    const row = document.getElementById("schemaTableBody").rows[idx];
    const lengthInput = row.cells[2].querySelector('input');
    const noLengthTypes = ['TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'INT'];

    if (noLengthTypes.includes(value)) {
        // Type doesn't use length - clear both UI and data
        lengthInput.disabled = true;
        lengthInput.value = '';  // ✅ Clear the visible value
        window.currentSchema[idx].length = null;  // ✅ Clear from schema data
    } else {
        // Type uses length - enable the input
        lengthInput.disabled = false;
    }
}
```

---

## Behavior Matrix

| Column Type | Uses Length? | Length Input State | Example |
|-------------|--------------|-------------------|---------|
| `VARCHAR` | ✅ Yes | Enabled, keeps value | `VARCHAR(100)` |
| `DECIMAL` | ✅ Yes | Enabled, keeps value | `DECIMAL(10,2)` |
| `ENUM` | ✅ Yes | Enabled, keeps value | `ENUM('active','inactive')` |
| `TEXT` | ❌ No | **Disabled, cleared** | `TEXT` |
| `DATE` | ❌ No | **Disabled, cleared** | `DATE` |
| `DATETIME` | ❌ No | **Disabled, cleared** | `DATETIME` |
| `INT` | ❌ No | **Disabled, cleared** | `INT` |
| `BOOLEAN` | ❌ No | **Disabled, cleared** | `BOOLEAN` |

---

## User Experience Improvements

### Before Fix
```
User Actions:
1. Set column to VARCHAR(50)
2. Change type to DATE
3. See: [DATE] [50] ← Confusing! Why is 50 still there?
```

### After Fix
```
User Actions:
1. Set column to VARCHAR(50)
2. Change type to DATE
3. See: [DATE] [  ] ← Clean! Length is cleared because DATE doesn't use it
```

### Switching Between Types
```
VARCHAR(100) → DATE → VARCHAR
         ↓         ↓        ↓
   Shows "100"    Shows ""  Shows "" (can enter new value)

VARCHAR(100) → DECIMAL → VARCHAR
         ↓         ↓         ↓
   Shows "100"  Shows "100" Shows "100" (preserved)
```

---

## Technical Details

### What Gets Cleared

When switching to a non-length type:

1. **DOM Input Value**: `lengthInput.value = ''`
   - Clears the visible text in the input field
   - User sees an empty, disabled field

2. **Schema Data Model**: `window.currentSchema[idx].length = null`
   - Removes the length property from the schema object
   - Ensures no stale data is sent to the backend

3. **Input State**: `lengthInput.disabled = true`
   - Disables the input (existing behavior)
   - Prevents user from entering values

### What Gets Preserved

When switching between types that both use length:

- `VARCHAR(100)` → `DECIMAL` keeps "100"
- Can be useful if user wants to reuse the value
- User can still manually edit/clear if desired

---

## Testing Steps

### Manual Testing

1. **Test Clearing:**
   - Go to [Gate 3 - Schema Builder](https://dev-app.filemonprime.net/importer/arrival.php)
   - Upload any Excel file
   - Set a column to `VARCHAR` with length `50`
   - Change type to `DATE`
   - **Expected**: Length input is empty and disabled ✅

2. **Test Preservation:**
   - Set a column to `VARCHAR` with length `100`
   - Change type to `DECIMAL`
   - **Expected**: Length input shows `100` and is enabled ✅

3. **Test Re-enabling:**
   - Set a column to `DATE` (length cleared)
   - Change type back to `VARCHAR`
   - **Expected**: Length input is empty and enabled (ready for new value) ✅

### Browser Console Test

```javascript
// After changing type to DATE, check schema:
console.log(window.currentSchema[0].length);
// Should output: null

// Length input should be empty:
document.getElementById("schemaTableBody").rows[0].cells[2].querySelector('input').value
// Should output: ""
```

---

## Files Modified

- ✅ `/lamp/www/importer/arrival.php` - Updated `updateSchema()` function (lines 652-667)

---

## Benefits

### 1. Clearer User Interface
No confusing disabled fields with stale values

### 2. Data Integrity
Schema data model stays clean (no `length: 50` on DATE columns)

### 3. Better UX
Users understand immediately which fields are relevant for each type

### 4. Consistent Behavior
All non-length types behave the same way (clear on switch)

---

## Related Data Types Reference

### Types Without Length
These types **automatically clear** length when selected:
- `TEXT` - Variable-length text (up to 65,535 chars)
- `DATE` - Date values (YYYY-MM-DD)
- `DATETIME` - Date and time values
- `INT` - Integer values (-2147483648 to 2147483647)
- `BOOLEAN` - True/False values (0 or 1)

### Types With Length
These types **preserve** length when selected:
- `VARCHAR(n)` - Variable-length string (max n chars)
- `DECIMAL(p,s)` - Fixed-point number (precision, scale)
- `ENUM('a','b',...)` - Enumerated values

---

**Date Implemented:** 2025-12-30
**Status:** ✅ Production Ready
**Location:** [arrival.php](https://dev-app.filemonprime.net/importer/arrival.php) - Gate 3: The Imagination
