# UUID Format Change - ia_guid() Style

## Summary

Modified `generateUUID()` to produce UUIDs in **ia_guid() format**: reversed segment order with no dashes.

### Format Comparison

| Format | Pattern | Example |
|--------|---------|---------|
| **Standard UUID** | `aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee` | `672630f3-f430-45b7-910c-fc0b89658594` |
| **ia_guid (NEW)** | `eeeeeeeeeeeeddddccccbbbbaaaaaaaa` | `fc0b89658594910c45b7f430672630f3` |

**Transformation:** `segments[4] . segments[3] . segments[2] . segments[1] . segments[0]` (reversed order, no dashes)

---

## Motivation

Align PHP `generateUUID()` with the existing `ia_guid()` function used elsewhere in the system for **consistency** across the codebase.

### ia_guid() Function

```php
/**
 * @param string $cmnt
 * @return string uuid
 */
function ia_guid($cmnt='guid'):string {
    $a = explode('-', ia_singleread("SELECT SQL_NO_CACHE /* $cmnt */ UUID()", '', false));
    return $a[4].$a[3].$a[2].$a[1].$a[0];
    // aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee → eeeeeeeeeeeeddddccccbbbbaaaaaaaa
}
```

**Logic:** Explode MySQL UUID by dashes, then concatenate segments in **reverse order**.

---

## Implementation

### Modified File

**Location:** `/lamp/www/importer/lib/DatabaseHelper.php` (lines 113-124)

### Before (Standard UUID v4)

```php
public function generateUUID() {
    $data = random_bytes(16);
    $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Version 4
    $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Variant
    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
}

// Output: "672630f3-f430-45b7-910c-fc0b89658594"
```

### After (ia_guid Format)

```php
public function generateUUID() {
    $data = random_bytes(16);
    $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Version 4
    $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Variant

    // Generate standard UUID format first
    $uuid = vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));

    // Reverse segment order to match ia_guid() format
    $segments = explode('-', $uuid);
    return $segments[4].$segments[3].$segments[2].$segments[1].$segments[0];
}

// Output: "fc0b89658594910c45b7f430672630f3"
```

### Key Changes

1. **Generate standard UUID** using `vsprintf()` (unchanged logic)
2. **Explode by dashes** to get 5 segments: `[aaaaaaaa, bbbb, cccc, dddd, eeeeeeeeeeee]`
3. **Reverse concatenate** segments: `$segments[4].$segments[3].$segments[2].$segments[1].$segments[0]`
4. **Return 32-character hex string** (no dashes)

---

## Technical Details

### UUID v4 Compliance

✅ **Maintained:** Version and variant bits are still set correctly before reordering:

```php
$data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Version 4 (0100 in bits 12-15)
$data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Variant (10 in bits 6-7)
```

These bits remain in the correct positions within the byte stream, even after segment reordering.

### Output Characteristics

| Property | Value |
|----------|-------|
| **Length** | 32 characters |
| **Character Set** | Hexadecimal (0-9, a-f) |
| **Dashes** | None |
| **Format** | `eeeeeeeeeeeeddddccccbbbbaaaaaaaa` |
| **Entropy** | 122 bits (same as UUID v4) |
| **Uniqueness** | Cryptographically random |

---

## Database Compatibility

### Column Type

**Current Schema:**
```sql
`table_id` CHAR(36) PRIMARY KEY
```

**UUID Length:**
- Standard format: 36 characters (`aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee`)
- ia_guid format: **32 characters** (`eeeeeeeeeeeeddddccccbbbbaaaaaaaa`)

### Fit Analysis

✅ **Compatible:** 32-character UUID fits in CHAR(36) column

**Behavior:**
- MySQL stores: `"fc0b89658594910c45b7f430672630f3    "` (4 trailing spaces)
- MySQL compares: Auto-trims spaces, so queries work correctly

### Example Queries

**Insert:**
```sql
INSERT INTO productos (producto_id, nombre)
VALUES ('fc0b89658594910c45b7f430672630f3', 'Product Name');
```

**Select:**
```sql
SELECT * FROM productos
WHERE producto_id = 'fc0b89658594910c45b7f430672630f3';
-- ✅ Works (MySQL trims trailing spaces in comparison)
```

**Join:**
```sql
SELECT * FROM productos p
JOIN inventarios i ON p.producto_id = i.producto_id;
-- ✅ Works (equality comparison trims spaces)
```

---

## Impact Analysis

### Files Using `generateUUID()`

**1. `/lamp/www/importer/insert.php` (line 395)**
```php
// Generate UUID for each row inserted
$uuid = $db->generateUUID();
```
- **Impact:** All new row primary keys use ia_guid format
- **Backward Compatible:** Yes, works with CHAR(36) columns

**2. `/lamp/www/importer/lib/ImportLogger.php` (line 208)**
```php
// Generate import log ID
$importLogId = $db->generateUUID();
```
- **Impact:** Import log IDs use ia_guid format
- **Backward Compatible:** Yes, works with existing log tables

**3. `/lamp/www/importer/lib/ImportLogger.php` (line 360)**
```php
// Generate schema log ID
$schemaLogId = $db->generateUUID();
```
- **Impact:** Schema log IDs use ia_guid format
- **Backward Compatible:** Yes, works with existing log tables

### Backward Compatibility

**Existing Data:**
- ✅ Old UUIDs (standard format with dashes) still stored in database
- ✅ New UUIDs (ia_guid format) inserted alongside old UUIDs
- ✅ MySQL CHAR(36) holds both formats (36 chars for old, 32 chars for new)
- ✅ No migration needed

**Mixed Format Example:**
```sql
-- Table may contain both formats:
SELECT producto_id FROM productos;

-- Old format (36 chars with dashes):
672630f3-f430-45b7-910c-fc0b89658594

-- New format (32 chars, no dashes):
fc0b89658594910c45b7f430672630f3
```

---

## Testing

### Test Suite

Created comprehensive test: `/lamp/www/importer/test_uuid_format.php`

**Run Tests:**
```bash
# CLI
/lamp/php/bin/php /lamp/www/importer/test_uuid_format.php

# Web
curl -s https://dev-app.filemonprime.net/importer/test_uuid_format.php
```

### Test Results

**5 Sample UUIDs Generated:**
```
UUID #1: 4998dd6eb349bd544b427a3c7627be33  ✅ Valid
UUID #2: 62302cb5618d8f9846f0471b275d8ee3  ✅ Valid
UUID #3: 1623acb20159a8814e955681bdad24e8  ✅ Valid
UUID #4: 0adef3775a49b3ea45145e7c979f7f19  ✅ Valid
UUID #5: 710557decf768c8f4c0cde8b2fe19dcb  ✅ Valid
```

**Validation Checks:**
- ✅ Length: 32 characters
- ✅ Character set: Hexadecimal only
- ✅ Format: Reversed segment order
- ✅ Database compatibility: Fits in CHAR(36)

---

## Example Transformation

### Step-by-Step Breakdown

**1. Generate Random Bytes:**
```php
$data = random_bytes(16);
// 16 bytes = 128 bits of randomness
```

**2. Set UUID v4 Version & Variant:**
```php
$data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Version 4
$data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Variant
```

**3. Convert to Standard UUID Format:**
```php
$uuid = vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
// Result: "672630f3-f430-45b7-910c-fc0b89658594"
```

**4. Explode into Segments:**
```php
$segments = explode('-', $uuid);
// $segments[0] = "672630f3"     (aaaaaaaa)
// $segments[1] = "f430"         (bbbb)
// $segments[2] = "45b7"         (cccc)
// $segments[3] = "910c"         (dddd)
// $segments[4] = "fc0b89658594" (eeeeeeeeeeee)
```

**5. Reverse Concatenate:**
```php
return $segments[4].$segments[3].$segments[2].$segments[1].$segments[0];
// Result: "fc0b89658594910c45b7f430672630f3"
```

### Visual Transformation

```
Standard UUID:
672630f3 - f430 - 45b7 - 910c - fc0b89658594
   (a)      (b)    (c)    (d)        (e)

                    ↓ Reverse ↓

ia_guid Format:
fc0b89658594 910c 45b7 f430 672630f3
    (e)      (d)  (c)  (b)     (a)

Final Output (no dashes):
fc0b89658594910c45b7f430672630f3
```

---

## Why This Format?

### Advantages of ia_guid Format

**1. Consistency Across Codebase**
- Matches existing `ia_guid()` function behavior
- Uniform UUID format throughout the system

**2. Shorter String Length**
- 32 characters vs 36 characters (4 fewer chars)
- Saves 4 bytes per UUID in storage (marginal but consistent)

**3. No Dash Parsing Needed**
- Pure hexadecimal string
- Easier to work with in some contexts (e.g., URL slugs, API keys)

**4. Still Cryptographically Random**
- Full 122 bits of entropy (UUID v4)
- Reordering doesn't reduce randomness
- Version/variant bits preserved

### Disadvantages (Trade-offs)

**1. Non-Standard Format**
- Not recognized as UUID by parsers
- Debugging tools may not recognize as UUID

**2. Mixed Format in Database**
- Old records: standard format with dashes
- New records: ia_guid format without dashes
- Acceptable but worth noting

**3. Requires Documentation**
- Developers need to understand the format difference
- This document serves that purpose

---

## Migration Considerations

### No Migration Required

**Reason:** CHAR(36) accommodates both formats:
- Old format: 36 characters (fills column exactly)
- New format: 32 characters (4 trailing spaces)

**Recommendation:**
- ✅ Leave existing data as-is
- ✅ New inserts use ia_guid format
- ✅ Both formats coexist peacefully

### Optional: Normalize to ia_guid Format

If you want **all UUIDs** in ia_guid format (optional):

```sql
-- WARNING: Only run if UUIDs are in standard format with dashes

UPDATE table_name
SET id_column = CONCAT(
    SUBSTRING_INDEX(SUBSTRING_INDEX(id_column, '-', -1), '-', 1),  -- segment 5 (e)
    SUBSTRING_INDEX(SUBSTRING_INDEX(id_column, '-', -2), '-', 1),  -- segment 4 (d)
    SUBSTRING_INDEX(SUBSTRING_INDEX(id_column, '-', 3), '-', -1),  -- segment 3 (c)
    SUBSTRING_INDEX(SUBSTRING_INDEX(id_column, '-', 2), '-', -1),  -- segment 2 (b)
    SUBSTRING_INDEX(id_column, '-', 1)                             -- segment 1 (a)
)
WHERE id_column LIKE '%-%';  -- Only convert UUIDs with dashes
```

**⚠️ Recommendation:** Not necessary unless you need uniform format across all records.

---

## Code Review Notes

### What Changed

✅ **Changed:**
- UUID output format (reversed segments, no dashes)
- Documentation (added comments explaining ia_guid format)

❌ **Not Changed:**
- UUID generation algorithm (still UUID v4)
- Version/variant bits (still compliant)
- Randomness source (`random_bytes(16)`)
- Database schema (CHAR(36) still works)

### Review Checklist

- ✅ Uses `random_bytes(16)` as specified
- ✅ Applies ia_guid reordering logic
- ✅ Maintains UUID v4 version/variant bits
- ✅ Returns 32-character hex string
- ✅ Compatible with CHAR(36) columns
- ✅ Tested via CLI and web interface
- ✅ Documented thoroughly

---

## Files Modified

- ✅ `/lamp/www/importer/lib/DatabaseHelper.php` - Updated `generateUUID()` (lines 113-124)

## Files Created

- ✅ `/lamp/www/importer/test_uuid_format.php` - Test suite
- ✅ `/lamp/www/importer/UUID_FORMAT_CHANGE.md` - This documentation

---

**Date Implemented:** 2025-12-30
**Status:** ✅ Production Ready
**Format:** ia_guid (reversed segments, no dashes)
**Test Results:** All tests passing ✅
