# UUID Column Migration: CHAR(36) → VARCHAR(32)

## Summary

Updated all UUID primary key columns from `CHAR(36)` to `VARCHAR(32)` to match the new ia_guid format (32-character UUIDs without dashes).

### Column Type Comparison

| Type | Storage | UUID Length | Wasted Space | Example |
|------|---------|-------------|--------------|---------|
| **CHAR(36) (OLD)** | 36 bytes | 32 chars | 4 bytes/row | `fc0b89658594910c45b7f430672630f3    ` (4 trailing spaces) |
| **VARCHAR(32) (NEW)** | 33 bytes | 32 chars | 0 bytes/row | `fc0b89658594910c45b7f430672630f3` (perfect fit) |

**Note:** VARCHAR(32) uses 33 bytes (32 + 1 length byte), but no wasted space for the actual data.

---

## Motivation

### Why Change?

**1. ia_guid Format Uses 32 Characters**
```
Standard UUID (old):  672630f3-f430-45b7-910c-fc0b89658594  (36 chars with dashes)
ia_guid format (new): fc0b89658594910c45b7f430672630f3     (32 chars, no dashes)
```

**2. CHAR(36) Wastes Space**
- Old format: 36 chars fit perfectly in CHAR(36)
- New format: 32 chars → 4 trailing spaces in CHAR(36)
- Wasted: 4 bytes per row

**3. VARCHAR(32) is Optimal**
- Exact fit for 32-character UUIDs
- Variable length storage (more flexible)
- Better index performance (smaller keys)

---

## Changes Made

### Files Modified

**1. `/lamp/www/importer/insert.php` (line 204)**
Backend PHP table creation

**Before:**
```php
$createSQL .= "  `{$pkFieldName}` CHAR(36) PRIMARY KEY COMMENT 'UUID primary key',\n";
```

**After:**
```php
$createSQL .= "  `{$pkFieldName}` VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',\n";
```

---

**2. `/lamp/www/importer/arrival.php` (line 820)**
Frontend JavaScript SQL preview

**Before:**
```javascript
sql += `  \`${pkFieldName}\` CHAR(36) PRIMARY KEY COMMENT 'UUID primary key',\n`;
```

**After:**
```javascript
sql += `  \`${pkFieldName}\` VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',\n`;
```

---

**3. `/lamp/www/importer/lib/ImportLogger.php` (lines 107, 152, 153)**
Import logging tables

**Before:**
```sql
import_log_id CHAR(36) PRIMARY KEY
import_schema_log_id CHAR(36) PRIMARY KEY
import_log_id CHAR(36) NOT NULL COMMENT 'FK to import_logs'
```

**After:**
```sql
import_log_id VARCHAR(32) PRIMARY KEY
import_schema_log_id VARCHAR(32) PRIMARY KEY
import_log_id VARCHAR(32) NOT NULL COMMENT 'FK to import_logs'
```

---

## Impact Analysis

### New Tables (Going Forward)

✅ **All new tables** created via the importer will use `VARCHAR(32)` for UUID primary keys

**Example:**
```sql
CREATE TABLE productos (
  producto_id VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',
  nombre VARCHAR(100) NOT NULL,
  precio DECIMAL(10,2) NOT NULL,
  -- ... other columns
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

### Existing Tables

**Status:** Unchanged (still have CHAR(36) columns)

**Behavior:**
- Old tables: `CHAR(36)` columns (contain both old and new UUID formats)
- New tables: `VARCHAR(32)` columns (will only contain ia_guid format)

**Compatibility:**
- ✅ Old UUIDs (36 chars with dashes) work in CHAR(36)
- ✅ New UUIDs (32 chars no dashes) work in both CHAR(36) and VARCHAR(32)

---

## Performance Benefits

### Storage Savings

**Per Row:**
- CHAR(36): 36 bytes (fixed)
- VARCHAR(32): 33 bytes (32 + 1 length byte)
- **Savings: 3 bytes per row**

**At Scale:**
| Rows | CHAR(36) Storage | VARCHAR(32) Storage | Savings |
|------|------------------|---------------------|---------|
| 1,000 | 36 KB | 33 KB | **3 KB** |
| 10,000 | 360 KB | 330 KB | **30 KB** |
| 100,000 | 3.6 MB | 3.3 MB | **300 KB** |
| 1,000,000 | 36 MB | 33 MB | **3 MB** |

### Index Performance

**Smaller Index Keys:**
- CHAR(36): 36 bytes per key
- VARCHAR(32): 32 bytes per key
- **Index size reduction: ~11%**

**Benefits:**
- ✅ More index entries fit in memory
- ✅ Faster index lookups
- ✅ Reduced disk I/O

---

## Migration Considerations

### No Migration Required

**Reason:** Existing tables still work fine with CHAR(36)

**Options:**

### Option 1: Leave Existing Tables As-Is (Recommended)
- ✅ No downtime
- ✅ No risk
- ✅ Old tables keep CHAR(36)
- ✅ New tables use VARCHAR(32)
- ⚠️ Mixed schema across tables

### Option 2: Migrate Existing Tables to VARCHAR(32) (Optional)

**Warning:** Only do this if:
1. All UUIDs are already in ia_guid format (32 chars, no dashes)
2. You want uniform schema across all tables
3. You can afford downtime for ALTER TABLE

**Migration Script:**
```sql
-- Check if all UUIDs are 32 characters (no dashes)
SELECT table_id, LENGTH(TRIM(table_id)) as len
FROM your_table
WHERE LENGTH(TRIM(table_id)) != 32;
-- If this returns any rows, migration is not safe!

-- Backup first!
CREATE TABLE your_table_backup AS SELECT * FROM your_table;

-- Alter column type
ALTER TABLE your_table
MODIFY COLUMN table_id VARCHAR(32) NOT NULL;

-- Verify
DESCRIBE your_table;

-- Test queries still work
SELECT * FROM your_table LIMIT 5;
```

**Time Estimate:**
- Small tables (<10k rows): Seconds
- Medium tables (100k rows): Minutes
- Large tables (1M+ rows): May require downtime

---

## Testing

### Test New Table Creation

**Steps:**
1. Go to [Gate 1 - Arrival](https://dev-app.filemonprime.net/importer/arrival.php)
2. Upload an Excel file
3. Set destination table name (e.g., `test_varchar32`)
4. Review SQL Preview (should show `VARCHAR(32)`)
5. Create table
6. Verify table structure

**Expected SQL Preview:**
```sql
CREATE TABLE test_varchar32 (
  test_varchar32_id VARCHAR(32) PRIMARY KEY COMMENT 'UUID primary key',
  -- ... other columns
)
```

**Verify Table Structure:**
```sql
DESCRIBE test_varchar32;

-- Expected:
-- test_varchar32_id | VARCHAR(32) | NO | PRI | NULL |
```

### Test UUID Insertion

**Verify UUIDs fit correctly:**
```sql
SELECT
  test_varchar32_id,
  LENGTH(test_varchar32_id) as uuid_length,
  CHAR_LENGTH(test_varchar32_id) as char_length
FROM test_varchar32
LIMIT 5;

-- Expected:
-- uuid_length: 32
-- char_length: 32
-- (Both should be 32, no trailing spaces)
```

---

## Rollback Plan

If you need to revert to CHAR(36):

### Rollback Code Changes

**1. insert.php (line 204):**
```php
// Revert to:
$createSQL .= "  `{$pkFieldName}` CHAR(36) PRIMARY KEY COMMENT 'UUID primary key',\n";
```

**2. arrival.php (line 820):**
```javascript
// Revert to:
sql += `  \`${pkFieldName}\` CHAR(36) PRIMARY KEY COMMENT 'UUID primary key',\n`;
```

**3. ImportLogger.php (lines 107, 152, 153):**
```sql
-- Revert to:
import_log_id CHAR(36) PRIMARY KEY
import_schema_log_id CHAR(36) PRIMARY KEY
import_log_id CHAR(36) NOT NULL
```

### Rollback Existing Tables (If Migrated)

```sql
-- Only needed if you ran ALTER TABLE migration

ALTER TABLE your_table
MODIFY COLUMN table_id CHAR(36) NOT NULL;
```

---

## Technical Details

### Why VARCHAR vs CHAR?

**CHAR(N):**
- Fixed length: always uses N bytes
- Pads with spaces if value is shorter
- Faster for fixed-width data
- **Best for:** Data that's always exactly N characters

**VARCHAR(N):**
- Variable length: uses actual length + 1 byte for length storage
- No padding (stores exact value)
- More flexible
- **Best for:** Data that varies in length (even if only slightly)

**Our Case:**
- UUIDs are always 32 characters (fixed length)
- However, VARCHAR(32) is still better because:
  - No wasted space from padding
  - Easier to work with (no TRIM needed)
  - Better index performance (no trailing spaces in keys)

### Storage Internals

**CHAR(36) Storage:**
```
Disk: |fc0b89658594910c45b7f430672630f3    | (36 bytes)
                                      ^^^^
                                 4 trailing spaces
```

**VARCHAR(32) Storage:**
```
Length: |20| (1 byte = 32 in hex)
Data:   |fc0b89658594910c45b7f430672630f3| (32 bytes)
Total:  33 bytes (1 + 32)
```

**Comparison:**
- CHAR(36): 36 bytes (4 wasted)
- VARCHAR(32): 33 bytes (0 wasted)
- **Net savings: 3 bytes per UUID**

---

## Related Changes

This migration is part of the ia_guid format implementation:

**Related Documents:**
- [UUID_FORMAT_CHANGE.md](UUID_FORMAT_CHANGE.md) - ia_guid format overview
- [EUROPEAN_NUMBER_FORMAT_FIX.md](EUROPEAN_NUMBER_FORMAT_FIX.md) - European decimal handling
- [ENUM_SQL_PREVIEW_FIX.md](ENUM_SQL_PREVIEW_FIX.md) - ENUM value handling

**Related Files:**
- [lib/DatabaseHelper.php](lib/DatabaseHelper.php#L113-L124) - `generateUUID()` function
- [insert.php](insert.php#L204) - Table creation (backend)
- [arrival.php](arrival.php#L820) - SQL preview (frontend)
- [lib/ImportLogger.php](lib/ImportLogger.php#L107) - Log table creation

---

## FAQ

### Q: Will old UUIDs (with dashes) still work?

**A:** Yes, if they're stored in CHAR(36) columns. However:
- New tables with VARCHAR(32) expect 32-character UUIDs only
- Mixing old (36-char) and new (32-char) UUIDs in VARCHAR(32) will cause errors
- Recommendation: Keep old tables as CHAR(36), new tables as VARCHAR(32)

### Q: What if I have UUIDs longer than 32 characters?

**A:** This shouldn't happen with ia_guid format:
- Standard UUID: 36 characters (with dashes)
- ia_guid format: 32 characters (no dashes)
- If you see 36-char UUIDs in new tables, something is wrong with generateUUID()

### Q: Can I use VARCHAR(36) for backward compatibility?

**A:** Yes, but not recommended:
- ✅ Works with both formats (old 36-char, new 32-char)
- ❌ Wastes 4 bytes on new UUIDs
- ❌ Defeats the purpose of the migration

### Q: Will foreign keys still work?

**A:** Yes, as long as both columns have the same type:
- ✅ VARCHAR(32) → VARCHAR(32) (works)
- ✅ CHAR(36) → CHAR(36) (works)
- ⚠️ CHAR(36) → VARCHAR(32) (may cause issues with trailing spaces)

**Recommendation:** Keep FK types consistent within each database.

---

## Verification Checklist

After deploying this change:

- ✅ Verify arrival.php loads correctly (HTTP 200)
- ✅ Upload a test file and check SQL preview shows `VARCHAR(32)`
- ✅ Create a test table and verify column type with `DESCRIBE`
- ✅ Insert test data and verify UUIDs are 32 characters
- ✅ Check import_logs table uses `VARCHAR(32)` (if recreated)
- ✅ Verify queries still work (SELECT, WHERE, JOIN)

**Quick Test:**
```bash
# 1. Check page loads
curl -s -o /dev/null -w "%{http_code}" https://dev-app.filemonprime.net/importer/arrival.php
# Expected: 200

# 2. Verify code changes
grep "VARCHAR(32)" /lamp/www/importer/insert.php
grep "VARCHAR(32)" /lamp/www/importer/arrival.php
grep "VARCHAR(32)" /lamp/www/importer/lib/ImportLogger.php
# Expected: All show VARCHAR(32) PRIMARY KEY
```

---

**Date Implemented:** 2025-12-30
**Status:** ✅ Production Ready
**Change Type:** Schema optimization (CHAR(36) → VARCHAR(32))
**Savings:** 3 bytes per UUID, ~11% index size reduction
**Backward Compatible:** Yes (existing tables unchanged)
