<?php
/**
 * Exodus Restore API
 *
 * Handles backup restoration and migration execution
 */

header('Content-Type: application/json');

require_once __DIR__ . '/../config.php';
require_once 'lib/ExodusEngine.php';
require_once 'lib/DatabaseMigrator.php';
require_once 'lib/ErrorHandler.php';

try {
    // Check if file was uploaded
    if (!isset($_FILES['backup_file'])) {
        throw new Exception('No backup file uploaded');
    }

    $uploadedFile = $_FILES['backup_file'];

    // Check for upload errors
    if ($uploadedFile['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('File upload error: ' . $uploadedFile['error']);
    }

    // Validate file extension
    $filename = $uploadedFile['name'];
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

    if (!in_array($ext, ['gz', 'zip'])) {
        throw new Exception('Invalid file type. Only .tar.gz and .zip files are supported');
    }

    // Create temporary directory for extraction
    $tempDir = sys_get_temp_dir() . '/exodus_restore_' . uniqid() . '/';
    if (!mkdir($tempDir, 0755, true)) {
        throw new Exception('Failed to create temporary extraction directory');
    }

    $steps = [];

    // Step 1: Extract archive
    $steps[] = ['step' => 'extract', 'status' => 'in_progress', 'message' => 'Extracting archive...'];

    $uploadPath = $uploadedFile['tmp_name'];

    if ($ext === 'gz') {
        // Extract tar.gz
        $cmd = sprintf(
            'tar -xzf %s -C %s 2>&1',
            escapeshellarg($uploadPath),
            escapeshellarg($tempDir)
        );

        exec($cmd, $output, $returnCode);

        if ($returnCode !== 0) {
            throw new Exception('Failed to extract tar.gz: ' . implode("\n", $output));
        }
    } else {
        // Extract ZIP
        $zip = new ZipArchive();
        if ($zip->open($uploadPath) !== true) {
            throw new Exception('Failed to open ZIP archive');
        }

        $zip->extractTo($tempDir);
        $zip->close();
    }

    $steps[0]['status'] = 'completed';
    $steps[0]['message'] = 'Archive extracted successfully';

    // Step 2: Read manifest
    $manifestPath = $tempDir . 'exodus_manifest.json';
    if (!file_exists($manifestPath)) {
        ErrorHandler::logError('Warning: No manifest found in backup');
        $manifest = null;
    } else {
        $manifest = json_decode(file_get_contents($manifestPath), true);
    }

    // Step 3: Restore code files
    $steps[] = ['step' => 'code', 'status' => 'in_progress', 'message' => 'Restoring code files...'];

    $baseDir = __DIR__ . '/../';

    // Copy PHP files, lib/, api/, etc.
    $codeDirs = ['lib', 'api', 'crud', 'forms'];
    $filesRestored = 0;

    foreach ($codeDirs as $dir) {
        $sourceDir = $tempDir . $dir;
        $destDir = $baseDir . $dir;

        if (is_dir($sourceDir)) {
            // Copy recursively
            $filesRestored += copyDirectoryRecursive($sourceDir, $destDir);
        }
    }

    // Copy root files (excluding config.php for safety)
    $rootFiles = glob($tempDir . '*.php');
    foreach ($rootFiles as $file) {
        $basename = basename($file);
        if ($basename !== 'config.php') {  // Don't overwrite config
            copy($file, $baseDir . $basename);
            $filesRestored++;
        }
    }

    $steps[1]['status'] = 'completed';
    $steps[1]['message'] = "Restored {$filesRestored} code files";

    // Step 4: Restore uploads
    $steps[] = ['step' => 'uploads', 'status' => 'in_progress', 'message' => 'Restoring uploads...'];

    $uploadsSource = $tempDir . 'uploads';
    $uploadsDest = $baseDir . 'uploads';

    if (is_dir($uploadsSource)) {
        $uploadFiles = copyDirectoryRecursive($uploadsSource, $uploadsDest);
        $steps[2]['status'] = 'completed';
        $steps[2]['message'] = "Restored {$uploadFiles} uploaded files";
    } else {
        $steps[2]['status'] = 'skipped';
        $steps[2]['message'] = 'No uploads in backup';
    }

    // Step 5: Run database migrations
    $steps[] = ['step' => 'database', 'status' => 'in_progress', 'message' => 'Running database migrations...'];

    // Find SQL migration files in backup
    $migrationFiles = glob($tempDir . 'migrations/*.sql');

    if (!empty($migrationFiles)) {
        foreach ($migrationFiles as $sqlFile) {
            $result = DatabaseMigrator::executeMigration($sqlFile);

            $steps[3]['status'] = 'completed';
            $steps[3]['message'] = sprintf(
                'Executed %d SQL statements (%d succeeded, %d failed)',
                $result['total_statements'],
                $result['executed'],
                $result['failed']
            );
        }
    } else {
        $steps[3]['status'] = 'skipped';
        $steps[3]['message'] = 'No database migrations found';
    }

    // Clean up temp directory
    removeDirectory($tempDir);

    echo json_encode([
        'success' => true,
        'message' => 'Restore completed successfully',
        'steps' => $steps,
        'files_restored' => $filesRestored,
        'manifest' => $manifest
    ]);

} catch (Exception $e) {
    ErrorHandler::logError('Exodus restore error: ' . $e->getMessage());

    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'steps' => $steps ?? []
    ]);
}

/**
 * Copy directory recursively
 *
 * @param string $source Source directory
 * @param string $dest Destination directory
 * @return int Number of files copied
 */
function copyDirectoryRecursive($source, $dest) {
    $count = 0;

    if (!is_dir($dest)) {
        mkdir($dest, 0755, true);
    }

    $dir = opendir($source);

    while (($file = readdir($dir)) !== false) {
        if ($file === '.' || $file === '..') {
            continue;
        }

        $sourcePath = $source . '/' . $file;
        $destPath = $dest . '/' . $file;

        if (is_dir($sourcePath)) {
            $count += copyDirectoryRecursive($sourcePath, $destPath);
        } else {
            copy($sourcePath, $destPath);
            $count++;
        }
    }

    closedir($dir);

    return $count;
}

/**
 * Remove directory recursively
 *
 * @param string $dir Directory to remove
 */
function removeDirectory($dir) {
    if (!is_dir($dir)) {
        return;
    }

    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );

    foreach ($iterator as $file) {
        if ($file->isDir()) {
            rmdir($file->getPathname());
        } else {
            unlink($file->getPathname());
        }
    }

    rmdir($dir);
}
