# Class: DatabaseHelper

**File:** `DatabaseHelper.php`


## Description

DatabaseHelper - The Connection Guardian
Manages MySQL connections, transactions, and UUID generation

## Methods

### Public::connect()

/**
DatabaseHelper - The Connection Guardian
Manages MySQL connections, transactions, and UUID generation
/

```php
/**
 * DatabaseHelper - The Connection Guardian
 * Manages MySQL connections, transactions, and UUID generation
 */

class DatabaseHelper {
    private static $instance = null;
    private $conn = null;
    private $inTransaction = false;

    private function __construct() {
        // Private constructor for singleton pattern
    }

    /**
     * Get singleton instance
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Establish database connection (Linux/Windows compatible)
     */
    public function connect(
```

### Public::createAndSelectDatabase()

/**
Create database if not exists and select it
/

```php
/**
     * Create database if not exists and select it
     */
    public function createAndSelectDatabase(
```

### Public::escapeIdentifier()

/**
Escape identifier (table/column name) with backticks
/

```php
/**
     * Escape identifier (table/column name) with backticks
     */
    public function escapeIdentifier(
```

### Public::generateUUID()

/**
Generate UUID v4 in ia_guid() format (reversed segment order, no dashes)
Standard UUID: aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee
ia_guid format: eeeeeeeeeeeeddddccccbbbbaaaaaaaa
/

```php
/**
     * Generate UUID v4 in ia_guid() format (reversed segment order, no dashes)
     * Standard UUID: aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee
     * ia_guid format: eeeeeeeeeeeeddddccccbbbbaaaaaaaa
     */
    public function generateUUID(
```

### Public::beginTransaction()

/**
Begin transaction
/

```php
/**
     * Begin transaction
     */
    public function beginTransaction(
```

### Public::commit()

/**
Commit transaction
/

```php
/**
     * Commit transaction
     */
    public function commit(
```

### Public::rollback()

/**
Rollback transaction
/

```php
/**
     * Rollback transaction
     */
    public function rollback(
```

### Public::getConnection()

/**
Get connection object
/

```php
/**
     * Get connection object
     */
    public function getConnection(
```

### Public::getError()

/**
Get last error
/

```php
/**
     * Get last error
     */
    public function getError(
```

### Public::close()

/**
Close connection
/

```php
/**
     * Close connection
     */
    public function close(
```

### Public::tableExists()

/**
Check if table exists
/

```php
/**
     * Check if table exists
     */
    public function tableExists(
```

### Public::__wakeup()

/**
Get table columns information
/
    public static function getColumns($tableName, $dbName) {
        $db = self::getInstance();
        $conn = $db->connect();

```php
/**
     * Get table columns information
     */
    public static function getColumns($tableName, $dbName) {
        $db = self::getInstance();
        $conn = $db->connect();
        
        // Validate DB parameter is provided
        if (empty($dbName)) {
            throw new Exception("DB is required for getColumns()");
        }
        $actualDbName = $dbName;
        
        if (!$conn->select_db($actualDbName)) {
            throw new Exception("Could not select database: " . $conn->error);
        }
        
        // Get column information with schema qualification
        $sql = "SHOW COLUMNS FROM `{$actualDbName}`.`{$tableName}`";
        $result = $conn->query($sql);
        
        if (!$result) {
            throw new Exception("Could not get columns: " . $conn->error);
        }
        
        $columns = [];
        while ($row = $result->fetch_assoc()) {
            $columns[] = [
                'name' => $row['Field'],
                'type' => $row['Type'],
                'nullable' => $row['Null'],
                'default' => $row['Default'],
                'key' => $row['Key'],
                'extra' => $row['Extra'],
                'comment' => $row['Comment'] ?? ''
            ];
        }
        
        return $columns;
    }

    /**
     * Get paginated rows from a table
     */
    public static function getRows($tableName, $dbName, $limit = 50, $offset = 0) {
        $db = self::getInstance();
        $conn = $db->connect();
        
        // Validate DB parameter is provided
        if (empty($dbName)) {
            throw new Exception("DB is required for getRows()");
        }
        $actualDbName = $dbName;
        
        if (!$conn->select_db($actualDbName)) {
            throw new Exception("Could not select database: " . $conn->error);
        }
        
        // Escape table name
        $safeTableName = $db->escapeIdentifier($tableName);
        
        // Get rows with pagination
        $sql = "SELECT * FROM {$safeTableName} LIMIT {$limit} OFFSET {$offset}";
        $result = $conn->query($sql);
        
        if (!$result) {
            throw new Exception("Could not get rows: " . $conn->error);
        }
        
        $rows = [];
        while ($row = $result->fetch_assoc()) {
            $rows[] = $row;
        }
        
        return $rows;
    }

    /**
     * Get a single row by ID
     */
    public static function getRow($tableName, $id, $dbName) {
        $db = self::getInstance();
        $conn = $db->connect();
        
        // Validate DB parameter is provided
        if (empty($dbName)) {
            throw new Exception("DB is required for getRow()");
        }
        $actualDbName = $dbName;
        
        if (!$conn->select_db($actualDbName)) {
            throw new Exception("Could not select database: " . $conn->error);
        }
        
        // Escape table name
        $safeTableName = $db->escapeIdentifier($tableName);

        // Get dynamic primary key field name
        $pkField = SchemaConventions::getPrimaryKeyName($tableName);

        // Try Spanish canonical field first, fall back to legacy 'id' if table uses English schema
        $mode = SchemaConventions::detectFieldMode($tableName, $dbName);
        if ($mode === 'english') {
            $pkField = SchemaConventions::getLegacyPrimaryKeyName();
        }

        // Get single row by ID
        $sql = "SELECT * FROM {$safeTableName} WHERE `{$pkField}` = ?";
        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }

        $stmt->bind_param("s", $id);
        $stmt->execute();
        $result = $stmt->get_result();

        if (!$result) {
            throw new Exception("Could not get row: " . $conn->error);
        }

        return $result->fetch_assoc();
    }

    /**
     * Insert a new row into a table
     */
    public static function insertRow($tableName, $data, $dbName) {
        $db = self::getInstance();
        $conn = $db->connect();
        
        // Validate DB parameter is provided
        if (empty($dbName)) {
            throw new Exception("DB is required for insertRow()");
        }
        $actualDbName = $dbName;
        
        if (!$conn->select_db($actualDbName)) {
            throw new Exception("Could not select database: " . $conn->error);
        }
        
        // Escape table name
        $safeTableName = $db->escapeIdentifier($tableName);

        // Detect field mode (Spanish canonical or English legacy)
        $mode = SchemaConventions::detectFieldMode($tableName, $dbName);

        // Get dynamic primary key and audit field names
        $pkField = ($mode === 'english')
            ? SchemaConventions::getLegacyPrimaryKeyName()
            : SchemaConventions::getPrimaryKeyName($tableName);

        $altaDbField = ($mode === 'english') ? 'insert_db' : 'alta_db';
        $altaPorField = ($mode === 'english') ? 'insert_by' : 'alta_por';

        // Prepare columns and values
        $columns = array_keys($data);
        $values = array_values($data);

        // Skip auto-managed fields using SchemaConventions
        $autoManagedFields = SchemaConventions::getAutoManagedFields($tableName, $dbName);
        $columns = array_filter($columns, function($col) use ($autoManagedFields) {
            return !in_array($col, $autoManagedFields);
        });

        // Re-index values after filtering columns
        $filteredData = [];
        foreach ($columns as $col) {
            $filteredData[$col] = $data[$col];
        }
        $values = array_values($filteredData);

        // Add UUID generation for primary key field
        if (!isset($data[$pkField])) {
            $data[$pkField] = $db->generateUUID();
            $columns[] = $pkField;
            $values[] = $data[$pkField];
        }

        // Add timestamp fields (Spanish canonical or English legacy)
        if (!isset($data[$altaDbField])) {
            $data[$altaDbField] = date('Y-m-d H:i:s');
            $columns[] = $altaDbField;
            $values[] = $data[$altaDbField];
        }

        if (!isset($data[$altaPorField])) {
            $data[$altaPorField] = 'system';
            $columns[] = $altaPorField;
            $values[] = $data[$altaPorField];
        }

        // Generate placeholders for prepared statement
        $placeholders = str_repeat('?,', count($columns) - 1) . '?';

        // Build SQL
        $sql = "INSERT INTO {$safeTableName} (`" . implode('`, `', $columns) . "`) VALUES ({$placeholders})";

        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }

        // Bind parameters
        $types = str_repeat('s', count($values)); // assuming all strings for simplicity
        $stmt->bind_param($types, ...$values);

        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }

        return $data[$pkField];
    }

    /**
     * Update an existing row in a table
     */
    public static function updateRow($tableName, $id, $data, $dbName) {
        $db = self::getInstance();
        $conn = $db->connect();
        
        // Validate DB parameter is provided
        if (empty($dbName)) {
            throw new Exception("DB is required for updateRow()");
        }
        $actualDbName = $dbName;
        
        if (!$conn->select_db($actualDbName)) {
            throw new Exception("Could not select database: " . $conn->error);
        }
        
        // Escape table name
        $safeTableName = $db->escapeIdentifier($tableName);

        // Detect field mode (Spanish canonical or English legacy)
        $mode = SchemaConventions::detectFieldMode($tableName, $dbName);

        // Get dynamic primary key and audit field names
        $pkField = ($mode === 'english')
            ? SchemaConventions::getLegacyPrimaryKeyName()
            : SchemaConventions::getPrimaryKeyName($tableName);

        $ultimoCambioField = ($mode === 'english') ? 'last_change' : 'ultimo_cambio';
        $ultimoCambioPorField = ($mode === 'english') ? 'last_change_by' : 'ultimo_cambio_por';

        // Skip auto-managed fields using SchemaConventions
        $autoManagedFields = SchemaConventions::getAutoManagedFields($tableName, $dbName);
        $columns = array_keys($data);
        $columns = array_filter($columns, function($col) use ($autoManagedFields) {
            return !in_array($col, $autoManagedFields);
        });

        if (empty($columns)) {
            throw new Exception("No columns to update");
        }

        // Add timestamp field for last change (Spanish canonical or English legacy)
        if (!isset($data[$ultimoCambioField])) {
            $data[$ultimoCambioField] = date('Y-m-d H:i:s');
            $columns[] = $ultimoCambioField;
        }

        // Add user field for last change by
        if (!isset($data[$ultimoCambioPorField])) {
            $data[$ultimoCambioPorField] = 'system';
            $columns[] = $ultimoCambioPorField;
        }

        // Build SET clause
        $setClause = '';
        $filteredData = [];
        $i = 0;
        foreach ($columns as $column) {
            if ($i > 0) $setClause .= ', ';
            $setClause .= "`{$column}` = ?";
            $filteredData[] = $data[$column];
            $i++;
        }

        // Build SQL with dynamic primary key
        $sql = "UPDATE {$safeTableName} SET {$setClause} WHERE `{$pkField}` = ?";

        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }

        // Prepare values array
        $filteredData[] = $id; // Add ID to end for WHERE clause

        // Bind parameters
        $types = str_repeat('s', count($filteredData)); // assuming all strings for simplicity
        $stmt->bind_param($types, ...$filteredData);

        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }

        return true;
    }

    /**
     * Delete a row from a table
     */
    public static function deleteRow($tableName, $id, $dbName) {
        $db = self::getInstance();
        $conn = $db->connect();
        
        // Validate DB parameter is provided
        if (empty($dbName)) {
            throw new Exception("DB is required for deleteRow()");
        }
        $actualDbName = $dbName;
        
        if (!$conn->select_db($actualDbName)) {
            throw new Exception("Could not select database: " . $conn->error);
        }
        
        // Escape table name
        $safeTableName = $db->escapeIdentifier($tableName);

        // Detect field mode and get dynamic primary key
        $mode = SchemaConventions::detectFieldMode($tableName, $dbName);
        $pkField = ($mode === 'english')
            ? SchemaConventions::getLegacyPrimaryKeyName()
            : SchemaConventions::getPrimaryKeyName($tableName);

        // Build SQL with dynamic primary key
        $sql = "DELETE FROM {$safeTableName} WHERE `{$pkField}` = ?";

        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }

        $stmt->bind_param("s", $id);

        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        return true;
    }

    /**
     * Prevent cloning
     */
    private function __clone() {}

    /**
     * Get row count for a table with optional WHERE clause
     */
    public static function getRowCount($tableName, $dbName, $whereClause = '', $params = []) {
        $db = self::getInstance();
        $conn = $db->connect();
        
        // Validate DB parameter is provided
        if (empty($dbName)) {
            throw new Exception("DB is required for getRowCount()");
        }
        $actualDbName = $dbName;
        
        if (!$conn->select_db($actualDbName)) {
            throw new Exception("Could not select database: " . $conn->error);
        }
        
        // Escape table name
        $safeTableName = $db->escapeIdentifier($tableName);
        
        // Build SQL
        $sql = "SELECT COUNT(*) as total FROM {$safeTableName}";
        
        if (!empty($whereClause)) {
            $sql .= " {$whereClause}";
        }
        
        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }
        
        if (!empty($params)) {
            $types = str_repeat('s', count($params));
            $stmt->bind_param($types, ...$params);
        }
        
        $stmt->execute();
        $result = $stmt->get_result();
        
        if (!$result) {
            throw new Exception("Could not get row count: " . $conn->error);
        }
        
        $row = $result->fetch_assoc();
        return (int)$row['total'];
    }

    /**
     * Prevent unserialization
     */
    public function __wakeup(
```

