<?php
/**
 * 🌌 Gate X: The Exodus — Cross-Platform Migration
 *
 * "The grid remembers. The exodus preserves. The migration endures."
 */

require_once __DIR__ . '/config.php';
require_once 'lib/ExodusEngine.php';
require_once 'lib/DatabaseMigrator.php';

// Get recent backups
$recentBackups = ExodusEngine::listRecentBackups(5);
$platform = ExodusEngine::detectPlatform();
$optimalFormat = ExodusEngine::getOptimalArchiveFormat();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🌌 Gate X: The Exodus — Project Migration</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Courier New', monospace;
            background: linear-gradient(135deg, #0a0a0a 0%, #1a1a2e 100%);
            color: #e0e0e0;
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
        }

        /* Header */
        header {
            text-align: center;
            margin-bottom: 40px;
            padding: 30px;
            background: rgba(0, 234, 255, 0.05);
            border: 2px solid #00eaff;
            border-radius: 10px;
            box-shadow: 0 0 30px rgba(0, 234, 255, 0.3);
            animation: pulse-glow 3s ease-in-out infinite;
        }

        @keyframes pulse-glow {
            0%, 100% { box-shadow: 0 0 20px rgba(0, 234, 255, 0.3); }
            50% { box-shadow: 0 0 40px rgba(0, 234, 255, 0.5); }
        }

        h1 {
            font-size: 2.5em;
            color: #00eaff;
            text-shadow: 0 0 20px rgba(0, 234, 255, 0.8);
            margin-bottom: 10px;
        }

        .subtitle {
            font-size: 1em;
            color: #a0a0a0;
            font-style: italic;
        }

        /* Main card */
        .card {
            background: rgba(26, 26, 46, 0.9);
            border: 2px solid #00eaff;
            border-radius: 10px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 0 20px rgba(0, 234, 255, 0.2);
        }

        .card h2 {
            color: #00eaff;
            margin-bottom: 20px;
            font-size: 1.8em;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        /* Form sections */
        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            color: #00eaff;
            margin-bottom: 8px;
            font-size: 0.9em;
        }

        .checkbox-group {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 12px;
            margin-top: 10px;
        }

        .checkbox-item {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 10px;
            background: rgba(0, 234, 255, 0.05);
            border: 1px solid rgba(0, 234, 255, 0.3);
            border-radius: 5px;
            transition: all 0.3s ease;
        }

        .checkbox-item:hover {
            background: rgba(0, 234, 255, 0.1);
            border-color: #00eaff;
        }

        .checkbox-item input[type="checkbox"] {
            width: 20px;
            height: 20px;
            cursor: pointer;
        }

        .checkbox-item label {
            cursor: pointer;
            margin: 0;
            flex: 1;
        }

        .file-size {
            color: #888;
            font-size: 0.85em;
        }

        /* Select dropdown */
        select {
            width: 100%;
            padding: 12px;
            background: rgba(0, 234, 255, 0.05);
            border: 2px solid rgba(0, 234, 255, 0.3);
            border-radius: 5px;
            color: #e0e0e0;
            font-family: 'Courier New', monospace;
            font-size: 1em;
            cursor: pointer;
        }

        select:focus {
            outline: none;
            border-color: #00eaff;
            box-shadow: 0 0 10px rgba(0, 234, 255, 0.5);
        }

        /* Buttons */
        .btn {
            padding: 15px 30px;
            background: linear-gradient(135deg, #00eaff 0%, #0088ff 100%);
            border: none;
            border-radius: 5px;
            color: #0a0a0a;
            font-family: 'Courier New', monospace;
            font-size: 1.1em;
            font-weight: bold;
            cursor: pointer;
            text-transform: uppercase;
            transition: all 0.3s ease;
            box-shadow: 0 0 20px rgba(0, 234, 255, 0.5);
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 30px rgba(0, 234, 255, 0.7);
        }

        .btn:active {
            transform: translateY(0);
        }

        .btn-secondary {
            background: linear-gradient(135deg, #6c757d 0%, #5a6268 100%);
        }

        /* Platform badge */
        .platform-badge {
            display: inline-block;
            padding: 5px 15px;
            background: rgba(0, 234, 255, 0.2);
            border: 1px solid #00eaff;
            border-radius: 15px;
            font-size: 0.85em;
            color: #00eaff;
        }

        /* Recent backups list */
        .backup-list {
            list-style: none;
        }

        .backup-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px;
            background: rgba(0, 234, 255, 0.05);
            border: 1px solid rgba(0, 234, 255, 0.2);
            border-radius: 5px;
            margin-bottom: 10px;
            transition: all 0.3s ease;
        }

        .backup-item:hover {
            background: rgba(0, 234, 255, 0.1);
            border-color: #00eaff;
        }

        .backup-info {
            flex: 1;
        }

        .backup-name {
            color: #00eaff;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .backup-meta {
            color: #888;
            font-size: 0.85em;
        }

        .backup-actions {
            display: flex;
            gap: 10px;
        }

        .btn-small {
            padding: 8px 15px;
            font-size: 0.85em;
        }

        /* Progress indicator */
        .progress-container {
            display: none;
            margin-top: 20px;
        }

        .progress-container.active {
            display: block;
        }

        .progress-bar {
            width: 100%;
            height: 30px;
            background: rgba(0, 234, 255, 0.1);
            border: 2px solid #00eaff;
            border-radius: 15px;
            overflow: hidden;
            position: relative;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #00eaff 0%, #0088ff 100%);
            width: 0%;
            transition: width 0.3s ease;
            box-shadow: 0 0 20px rgba(0, 234, 255, 0.8);
        }

        .progress-text {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            color: #0a0a0a;
            font-weight: bold;
            text-shadow: 0 0 5px rgba(255, 255, 255, 0.8);
        }

        /* Status messages */
        .status-message {
            padding: 15px;
            border-radius: 5px;
            margin-top: 20px;
            display: none;
        }

        .status-message.show {
            display: block;
        }

        .status-success {
            background: rgba(40, 167, 69, 0.2);
            border: 2px solid #28a745;
            color: #28a745;
        }

        .status-error {
            background: rgba(220, 53, 69, 0.2);
            border: 2px solid #dc3545;
            color: #dc3545;
        }

        .status-info {
            background: rgba(0, 234, 255, 0.2);
            border: 2px solid #00eaff;
            color: #00eaff;
        }

        /* File upload */
        .file-upload {
            position: relative;
            display: inline-block;
            width: 100%;
        }

        .file-upload input[type="file"] {
            position: absolute;
            opacity: 0;
            width: 100%;
            height: 100%;
            cursor: pointer;
        }

        .file-upload-label {
            display: block;
            padding: 40px;
            background: rgba(0, 234, 255, 0.05);
            border: 2px dashed rgba(0, 234, 255, 0.5);
            border-radius: 10px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .file-upload-label:hover {
            background: rgba(0, 234, 255, 0.1);
            border-color: #00eaff;
        }

        .file-upload-text {
            color: #00eaff;
            font-size: 1.1em;
        }

        /* Info box */
        .info-box {
            background: rgba(0, 234, 255, 0.05);
            border-left: 4px solid #00eaff;
            padding: 15px;
            margin: 15px 0;
            border-radius: 5px;
        }

        .info-box p {
            margin: 5px 0;
            color: #a0a0a0;
        }

        .info-box strong {
            color: #00eaff;
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <header>
            <h1>🌌 Gate X: The Exodus</h1>
            <p class="subtitle">"The grid remembers. The exodus preserves. The migration endures."</p>
            <p style="margin-top: 10px;">
                <span class="platform-badge">Platform: <?= ucfirst($platform) ?></span>
                <span class="platform-badge">Format: <?= $optimalFormat ?></span>
            </p>
        </header>

        <!-- Create Backup Section -->
        <div class="card">
            <h2>📦 Create Backup</h2>

            <div class="info-box">
                <p><strong>🌌 Smart Backup System</strong></p>
                <p>Creates portable archives with all dependencies and database schema.</p>
                <p>Safe to restore on Linux or Windows - platform auto-detected.</p>
            </div>

            <form id="backupForm">
                <div class="form-group">
                    <label>Select Components to Backup:</label>
                    <div class="checkbox-group">
                        <div class="checkbox-item">
                            <input type="checkbox" id="include_code" name="include_code" checked>
                            <label for="include_code">
                                PHP Source Code
                                <span class="file-size">(~500KB)</span>
                            </label>
                        </div>
                        <div class="checkbox-item">
                            <input type="checkbox" id="include_vendor" name="include_vendor" checked>
                            <label for="include_vendor">
                                Vendor Dependencies
                                <span class="file-size">(~25MB)</span>
                            </label>
                        </div>
                        <div class="checkbox-item">
                            <input type="checkbox" id="include_uploads" name="include_uploads" checked>
                            <label for="include_uploads">
                                Uploaded Files
                                <span class="file-size">(~3MB)</span>
                            </label>
                        </div>
                        <div class="checkbox-item">
                            <input type="checkbox" id="include_logs" name="include_logs">
                            <label for="include_logs">
                                Log Files
                                <span class="file-size">(~500KB)</span>
                            </label>
                        </div>
                        <div class="checkbox-item">
                            <input type="checkbox" id="include_database" name="include_database" checked>
                            <label for="include_database">
                                Database Schema
                                <span class="file-size">(~100KB)</span>
                            </label>
                        </div>
                        <div class="checkbox-item">
                            <input type="checkbox" id="include_tmp" name="include_tmp">
                            <label for="include_tmp">
                                Temp Directory
                                <span class="file-size">(varies)</span>
                            </label>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <label for="format">Archive Format:</label>
                    <select id="format" name="format">
                        <option value="tar.gz" <?= $optimalFormat === 'tar.gz' ? 'selected' : '' ?>>tar.gz (Recommended for Linux)</option>
                        <option value="zip" <?= $optimalFormat === 'zip' ? 'selected' : '' ?>>ZIP (Recommended for Windows)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="platform">Target Platform:</label>
                    <select id="platform" name="platform">
                        <option value="auto" selected>Auto-detect</option>
                        <option value="linux">Linux</option>
                        <option value="windows">Windows</option>
                    </select>
                </div>

                <button type="submit" class="btn">📥 Create & Download Backup</button>
            </form>

            <!-- Progress -->
            <div id="progressContainer" class="progress-container">
                <div class="progress-bar">
                    <div id="progressFill" class="progress-fill"></div>
                    <div id="progressText" class="progress-text">0%</div>
                </div>
            </div>

            <!-- Status message -->
            <div id="statusMessage" class="status-message"></div>
        </div>

        <!-- Restore from Backup Section -->
        <div class="card">
            <h2>📤 Restore from Backup</h2>

            <div class="info-box">
                <p><strong>🔄 Intelligent Restoration</strong></p>
                <p>Upload a backup archive to restore code, database, and uploads.</p>
                <p>Migrations are applied safely - can be re-run without errors.</p>
            </div>

            <form id="restoreForm">
                <div class="form-group">
                    <div class="file-upload">
                        <input type="file" id="restoreFile" name="backup_file" accept=".tar.gz,.zip">
                        <label for="restoreFile" class="file-upload-label">
                            <div class="file-upload-text">
                                📁 Click to choose backup file
                                <br>
                                <span style="font-size: 0.85em; color: #888;">Supports: .tar.gz, .zip</span>
                            </div>
                        </label>
                    </div>
                </div>

                <button type="submit" class="btn btn-secondary">🚀 Restore & Migrate</button>
            </form>

            <div id="restoreStatus" class="status-message"></div>
        </div>

        <!-- Recent Backups Section -->
        <div class="card">
            <h2>📊 Recent Backups</h2>

            <?php if (empty($recentBackups)): ?>
                <p style="color: #888; text-align: center; padding: 20px;">
                    No backups found. Create your first backup above!
                </p>
            <?php else: ?>
                <ul class="backup-list">
                    <?php foreach ($recentBackups as $backup): ?>
                        <li class="backup-item">
                            <div class="backup-info">
                                <div class="backup-name"><?= htmlspecialchars($backup['filename']) ?></div>
                                <div class="backup-meta">
                                    <?= $backup['size_mb'] ?> MB •
                                    <?= $backup['created_at'] ?> •
                                    <?= $backup['age_hours'] ?> hours ago
                                </div>
                            </div>
                            <div class="backup-actions">
                                <a href="api/exodus_download.php?file=<?= urlencode($backup['filename']) ?>" class="btn btn-small">⬇️ Download</a>
                                <button class="btn btn-small btn-secondary" onclick="deleteBackup('<?= htmlspecialchars($backup['filename']) ?>')">🗑️ Delete</button>
                            </div>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Create backup
        document.getElementById('backupForm').addEventListener('submit', async function(e) {
            e.preventDefault();

            const formData = new FormData(this);
            const options = {
                include_code: formData.get('include_code') === 'on',
                include_vendor: formData.get('include_vendor') === 'on',
                include_uploads: formData.get('include_uploads') === 'on',
                include_logs: formData.get('include_logs') === 'on',
                include_database: formData.get('include_database') === 'on',
                include_tmp: formData.get('include_tmp') === 'on',
                format: formData.get('format'),
                platform: formData.get('platform')
            };

            showProgress(true);
            showStatus('Creating backup...', 'info');

            try {
                const response = await fetch('api/exodus_backup.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'create', options })
                });

                const result = await response.json();

                if (result.success) {
                    showStatus(`✅ Backup created successfully! (${result.size_mb} MB)`, 'success');
                    updateProgress(100);

                    // Trigger download
                    window.location.href = `api/exodus_download.php?file=${encodeURIComponent(result.filename)}`;

                    // Reload page after 2 seconds to show new backup in list
                    setTimeout(() => window.location.reload(), 2000);
                } else {
                    showStatus('❌ Error: ' + result.message, 'error');
                }
            } catch (error) {
                showStatus('❌ Error: ' + error.message, 'error');
            } finally {
                setTimeout(() => showProgress(false), 2000);
            }
        });

        // Restore from backup
        document.getElementById('restoreForm').addEventListener('submit', async function(e) {
            e.preventDefault();

            const formData = new FormData(this);

            if (!formData.get('backup_file').name) {
                showRestoreStatus('⚠️ Please select a backup file', 'error');
                return;
            }

            showRestoreStatus('Uploading and restoring backup...', 'info');

            try {
                const response = await fetch('api/exodus_restore.php', {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();

                if (result.success) {
                    showRestoreStatus('✅ Restore completed successfully!', 'success');
                } else {
                    showRestoreStatus('❌ Error: ' + result.message, 'error');
                }
            } catch (error) {
                showRestoreStatus('❌ Error: ' + error.message, 'error');
            }
        });

        // Delete backup
        async function deleteBackup(filename) {
            if (!confirm(`Delete backup: ${filename}?`)) return;

            try {
                const response = await fetch('api/exodus_backup.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'delete', filename })
                });

                const result = await response.json();

                if (result.success) {
                    window.location.reload();
                } else {
                    alert('Error deleting backup: ' + result.message);
                }
            } catch (error) {
                alert('Error: ' + error.message);
            }
        }

        // Show progress bar
        function showProgress(show) {
            document.getElementById('progressContainer').classList.toggle('active', show);
            if (!show) {
                updateProgress(0);
            }
        }

        // Update progress
        function updateProgress(percent) {
            document.getElementById('progressFill').style.width = percent + '%';
            document.getElementById('progressText').textContent = Math.round(percent) + '%';
        }

        // Show status message
        function showStatus(message, type) {
            const statusEl = document.getElementById('statusMessage');
            statusEl.textContent = message;
            statusEl.className = 'status-message show status-' + type;
        }

        // Show restore status
        function showRestoreStatus(message, type) {
            const statusEl = document.getElementById('restoreStatus');
            statusEl.textContent = message;
            statusEl.className = 'status-message show status-' + type;
        }

        // Update file upload label when file selected
        document.getElementById('restoreFile').addEventListener('change', function() {
            const label = this.nextElementSibling.querySelector('.file-upload-text');
            if (this.files.length > 0) {
                label.innerHTML = `✅ ${this.files[0].name}<br><span style="font-size: 0.85em; color: #888;">${(this.files[0].size / 1024 / 1024).toFixed(2)} MB</span>`;
            }
        });
    </script>
</body>
</html>
