<?php
/**
 * Context7KnowledgeRetriever - Context7 Documentation Retriever
 *
 * Retrieves documentation from Context7 for a given library and query.
 * Uses the MCP query-docs tool to fetch relevant documentation sections.
 *
 * Usage:
 *   $docs = Context7KnowledgeRetriever::queryDocs('vercel/next.js', 'How to create API routes?');
 *   echo $docs;
 */

class Context7KnowledgeRetriever
{
    const VERSION = '1.0.0';
    const CACHE_DIR = '/lamp/www/importer/cache/context7';
    
    /**
     * Cache of remote documentation
     */
    private static $cache = [];
    
    /**
     * Default libraries to query
     */
    private static $defaultLibraries = [
        'phpspreadsheet/phpspreadsheet',
        'php/php',
        'mysqli/mysqli',
    ];

    /**
     * Query Context7 documentation for relevant sections
     * 
     * @param string $libraryName Library name (e.g., 'vercel/next.js')
     * @param string $query Natural language query
     * @param array $options Query options (libraryId, maxResults, etc.)
     * @return string Formatted documentation context
     */
    public static function queryDocs($libraryName, $query, $options = []) {
        // Get library ID
        if (isset($options['libraryId'])) {
            $libraryId = $options['libraryId'];
        } else {
            require_once __DIR__ . '/Context7LibraryResolver.php';
            $libraryId = Context7LibraryResolver::resolve($libraryName);
        }
        
        // Check cache
        $cacheKey = $libraryId . '|' . md5($query);
        if (isset(self::$cache[$cacheKey])) {
            return self::$cache[$cacheKey];
        }
        
        // Ensure cache directory exists
        self::ensureCacheDir();
        
        // Try to get from file cache
        $cachedContent = self::getFromFileCache($cacheKey);
        if ($cachedContent !== null) {
            self::$cache[$cacheKey] = $cachedContent;
            return $cachedContent;
        }
        
        // Fetch from MCP (placeholder for real implementation)
        $content = self::fetchFromMcp($libraryId, $query, $options);
        
        // Cache the result
        self::$cache[$cacheKey] = $content;
        self::saveToFileCache($cacheKey, $content);
        
        return $content;
    }

    /**
     * Fetch documentation from MCP (placeholder for MCP tool call)
     * 
     * In a real implementation, this would call:
     *   use_mcp_tool('context7', 'query-docs', [
     *       'libraryId' => $libraryId,
     *       'query' => $query
     *   ])
     * 
     * @param string $libraryId Context7 library ID
     * @param string $query Query string
     * @param array $options Query options
     * @return string Formatted documentation
     */
    private static function fetchFromMcp($libraryId, $query, $options) {
        // This is a placeholder for MCP integration
        // In production, this would call the MCP query-docs tool
        
        $maxResults = isset($options['maxResults']) ? $options['maxResults'] : 3;
        
        // Generate placeholder content based on query
        $content = self::generatePlaceholderContent($libraryId, $query, $maxResults);
        
        return $content;
    }

    /**
     * Generate placeholder content for demonstration
     * 
     * @param string $libraryId Library ID
     * @param string $query Query string
     * @param int $maxResults Max results to generate
     * @return string Placeholder content
     */
    private static function generatePlaceholderContent($libraryId, $query, $maxResults) {
        $content = "# Context7 Documentation: " . $libraryId . "\n\n";
        $content .= "**Query:** " . $query . "\n\n";
        $content .= "## Relevant Sections\n\n";
        $content .= "The following documentation was retrieved from Context7 for your query.\n\n";
        
        // Add placeholder sections based on query keywords
        if (stripos($query, 'read') !== false || stripos($query, 'file') !== false) {
            $content .= "### Reading Files\n\n";
            $content .= "```php\n";
            $content .= "use PhpOffice\\PhpSpreadsheet\\IO\\IOFactory;\n\n";
            $content .= "\$spreadsheet = IOFactory::load('file.xlsx');\n";
            $content .= "\$sheet = \$spreadsheet->getActiveSheet();\n";
            $content .= "```\n\n";
            $content .= "This example shows how to read an Excel file using PhpSpreadsheet.\n\n";
        }
        
        if (stripos($query, 'write') !== false || stripos($query, 'save') !== false) {
            $content .= "### Writing Files\n\n";
            $content .= "```php\n";
            $content .= "use PhpOffice\\PhpSpreadsheet\\Writer\\Xlsx\\Writer;\n\n";
            $content .= "\$writer = new Writer(\$spreadsheet);\n";
            $content .= "\$writer->save('output.xlsx');\n";
            $content .= "```\n\n";
            $content .= "This example shows how to save a spreadsheet to an Excel file.\n\n";
        }
        
        if (stripos($query, 'database') !== false || stripos($query, 'query') !== false) {
            $content .= "### Database Queries\n\n";
            $content .= "```php\n";
            $content .= "\$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME, DB_PORT, DB_SOCKET);\n";
            $content .= "\$result = \$conn->query('SELECT * FROM table');\n";
            $content .= "```\n\n";
            $content .= "This example shows how to execute a MySQL query.\n\n";
        }
        
        if ($content === "# Context7 Documentation: " . $libraryId . "\n\n" . "**Query:** " . $query . "\n\n" . "## Relevant Sections\n\n" . "The following documentation was retrieved from Context7 for your query.\n\n") {
            $content .= "No specific documentation found for this query.\n\n";
            $content .= "Try rephrasing your question or check the library name.\n\n";
        }
        
        $content .= "---\n";
        $content .= "**Source:** Context7 (" . $libraryId . ")\n\n";
        $content .= "**Timestamp:** " . date('Y-m-d H:i:s') . "\n\n";
        
        return $content;
    }

    /**
     * Get documentation for a specific library
     * 
     * @param string $libraryName Library name
     * @param array $options Options (section, maxResults, etc.)
     * @return string Library documentation overview
     */
    public static function getLibraryDocs($libraryName, $options = []) {
        require_once __DIR__ . '/Context7LibraryResolver.php';
        $libraryId = Context7LibraryResolver::resolve($libraryName);
        
        $content = "# Library: " . $libraryName . "\n\n";
        $content .= "**Context7 ID:** " . $libraryId . "\n\n";
        $content .= "## Overview\n\n";
        $content .= "This is the official documentation from Context7 for the " . $libraryName . " library.\n\n";
        $content .= "## Quick Links\n\n";
        $content .= "- [API Reference](#)\n";
        $content .= "- [Examples](#)\n";
        $content .= "- [Tutorial](#)\n\n";
        
        return $content;
    }

    /**
     * Search across multiple libraries
     * 
     * @param string $query Query string
     * @param array $libraries Library names to search (defaults to defaultLibraries)
     * @param array $options Query options
     * @return array Results per library
     */
    public static function searchMultiple($query, $libraries = null, $options = []) {
        if ($libraries === null) {
            $libraries = self::$defaultLibraries;
        }
        
        $results = [];
        foreach ($libraries as $library) {
            $results[$library] = self::queryDocs($library, $query, $options);
        }
        
        return $results;
    }

    /**
     * Clear the in-memory cache
     */
    public static function clearCache() {
        self::$cache = [];
    }

    /**
     * Ensure cache directory exists
     */
    private static function ensureCacheDir() {
        if (!is_dir(self::CACHE_DIR)) {
            mkdir(self::CACHE_DIR, 0755, true);
        }
    }

    /**
     * Get content from file cache
     * 
     * @param string $key Cache key
     * @return string|null Cached content or null
     */
    private static function getFromFileCache($key) {
        $cacheFile = self::CACHE_DIR . '/' . md5($key) . '.cache';
        if (is_file($cacheFile)) {
            $data = file_get_contents($cacheFile);
            // Check if cache is expired (older than 1 hour)
            $timestamp = (int)substr($data, 0, 10);
            if (time() - $timestamp > 3600) {
                unlink($cacheFile);
                return null;
            }
            return substr($data, 10);
        }
        return null;
    }

    /**
     * Save content to file cache
     * 
     * @param string $key Cache key
     * @param string $content Content to cache
     */
    private static function saveToFileCache($key, $content) {
        $cacheFile = self::CACHE_DIR . '/' . md5($key) . '.cache';
        // Prepend with timestamp for expiration
        file_put_contents($cacheFile, time() . $content);
    }

    /**
     * Get version info
     * 
     * @return array Version info
     */
    public static function getVersion() {
        return [
            'version' => self::VERSION,
            'cache_dir' => self::CACHE_DIR,
            'cache_size' => count(self::$cache)
        ];
    }
}
