<?php
/**
 * 🌌 DatabaseMigrator - Smart Schema Migration System
 *
 * Generates safe, resumable ALTER TABLE statements with per-column checks.
 * Never fails due to existing columns - can be re-run multiple times safely.
 *
 * Features:
 * - Per-column ALTER TABLE (safe to re-run)
 * - Schema comparison (source vs target)
 * - Missing column detection
 * - Automatic migration script generation
 * - Rollback script creation
 * - Migration logging
 *
 * "The schema evolves. The migration endures."
 */

require_once 'DatabaseHelper.php';
require_once 'ErrorHandler.php';

class DatabaseMigrator {

    const MIGRATION_DIR = __DIR__ . '/../migrations/';

    /**
     * Export database schema with CREATE TABLE + per-column ALTERs
     *
     * @param string $databaseName Database to export
     * @param string $outputFile Output SQL file
     * @return array Table information
     */
    public static function exportDatabaseSchema($databaseName, $outputFile) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        // Select database
        if (!$conn->select_db($databaseName)) {
            throw new Exception("Database '$databaseName' not found");
        }

        // Get all tables
        $tablesResult = $conn->query("SHOW TABLES");
        if (!$tablesResult) {
            throw new Exception("Failed to get tables from database '$databaseName'");
        }

        $tables = [];
        while ($row = $tablesResult->fetch_array()) {
            $tables[] = $row[0];
        }

        // Open output file
        $sqlContent = self::generateMigrationHeader($databaseName);

        // Export each table
        foreach ($tables as $tableName) {
            $tableSQL = self::generateTableMigration($tableName, $databaseName, $conn);
            $sqlContent .= $tableSQL;
        }

        // Add footer
        $sqlContent .= self::generateMigrationFooter();

        // Write to file
        file_put_contents($outputFile, $sqlContent);

        return $tables;
    }

    /**
     * Generate migration header with metadata
     *
     * @param string $databaseName Database name
     * @return string SQL header
     */
    private static function generateMigrationHeader($databaseName) {
        $timestamp = date('Y-m-d H:i:s');
        $platform = ExodusEngine::detectPlatform();

        return <<<SQL
-- ============================================
-- 🌌 Exodus Migration Script
-- ============================================
-- Generated: {$timestamp}
-- Database: {$databaseName}
-- Platform: {$platform}
-- PHP Version: {PHP_VERSION}
-- ============================================
--
-- SAFE TO RE-RUN: This script checks for existing
-- tables and columns before making changes.
--
-- Each column is added individually, so partial
-- migrations can be resumed without errors.
--
-- ============================================

-- Ensure database exists (create if missing)
CREATE DATABASE IF NOT EXISTS `{$databaseName}`
    DEFAULT CHARACTER SET utf8mb4
    COLLATE utf8mb4_unicode_ci;

USE `{$databaseName}`;


SQL;
    }

    /**
     * Generate migration footer
     *
     * @return string SQL footer
     */
    private static function generateMigrationFooter() {
        $timestamp = date('Y-m-d H:i:s');

        return <<<SQL

-- ============================================
-- Migration Complete
-- Finished: {$timestamp}
-- ============================================

SELECT 'Migration completed successfully' AS status;

SQL;
    }

    /**
     * Generate migration SQL for a single table
     *
     * @param string $tableName Table name
     * @param string $databaseName Database name
     * @param mysqli $conn Database connection
     * @return string SQL statements
     */
    private static function generateTableMigration($tableName, $databaseName, $conn) {
        // Get CREATE TABLE statement
        $createResult = $conn->query("SHOW CREATE TABLE `{$tableName}`");
        if (!$createResult) {
            throw new Exception("Failed to get CREATE TABLE for '$tableName'");
        }

        $createRow = $createResult->fetch_assoc();
        $createSQL = $createRow['Create Table'];

        // Get column information
        $columnsResult = $conn->query("DESCRIBE `{$tableName}`");
        if (!$columnsResult) {
            throw new Exception("Failed to describe table '$tableName'");
        }

        $columns = [];
        while ($row = $columnsResult->fetch_assoc()) {
            $columns[] = $row;
        }

        // Start SQL output
        $sql = "\n-- ============================================\n";
        $sql .= "-- Table: {$tableName}\n";
        $sql .= "-- Columns: " . count($columns) . "\n";
        $sql .= "-- ============================================\n\n";

        // Create table if not exists (minimal version)
        $sql .= "-- Create table structure (if missing)\n";
        $sql .= "CREATE TABLE IF NOT EXISTS `{$tableName}` (\n";
        $sql .= "    `placeholder_id` INT AUTO_INCREMENT PRIMARY KEY COMMENT 'Temporary placeholder'\n";
        $sql .= ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;\n\n";

        // Add each column individually (safe to re-run)
        $sql .= "-- Add columns one by one (safe to re-run)\n";

        foreach ($columns as $index => $column) {
            $columnName = $column['Field'];
            $columnNumber = $index + 1;

            $sql .= "\n-- Column {$columnNumber}: {$columnName}\n";
            $sql .= self::generatePerColumnAlter($tableName, $column);
        }

        // Remove placeholder if it exists
        $sql .= "\n-- Remove placeholder column (if exists)\n";
        $sql .= self::generateDropColumnSafe($tableName, 'placeholder_id');

        $sql .= "\n";

        return $sql;
    }

    /**
     * Generate safe ALTER TABLE for a single column
     *
     * Uses INFORMATION_SCHEMA to check if column exists
     *
     * @param string $tableName Table name
     * @param array $column Column definition from DESCRIBE
     * @return string SQL ALTER TABLE statement
     */
    private static function generatePerColumnAlter($tableName, $column) {
        $columnName = $column['Field'];
        $columnType = $column['Type'];
        $nullable = $column['Null'] === 'YES' ? 'NULL' : 'NOT NULL';
        $default = $column['Default'];
        $extra = $column['Extra'];  // auto_increment, on update, etc.
        $key = $column['Key'];  // PRI, UNI, MUL

        // Build column definition
        $columnDef = "`{$columnName}` {$columnType}";

        // Add NOT NULL / NULL
        $columnDef .= " {$nullable}";

        // Add DEFAULT (if not null and not auto_increment)
        if ($default !== null && stripos($extra, 'auto_increment') === false) {
            if ($default === 'CURRENT_TIMESTAMP' || $default === 'NULL') {
                $columnDef .= " DEFAULT {$default}";
            } else {
                $columnDef .= " DEFAULT '{$default}'";
            }
        }

        // Add extra attributes (auto_increment, on update, etc.)
        if (!empty($extra)) {
            // Handle ON UPDATE CURRENT_TIMESTAMP
            if (stripos($extra, 'on update') !== false) {
                $columnDef .= " ON UPDATE CURRENT_TIMESTAMP";
            }

            // Handle AUTO_INCREMENT
            if (stripos($extra, 'auto_increment') !== false) {
                $columnDef .= " AUTO_INCREMENT";
            }
        }

        // Add COMMENT if present (from full SHOW CREATE TABLE)
        // Note: DESCRIBE doesn't show comments, so we'll skip for now
        // TODO: Parse SHOW CREATE TABLE for comments

        // Generate safe ALTER TABLE using prepared statement approach
        $sql = <<<SQL
SET @col_exists = (
    SELECT COUNT(*)
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
        AND TABLE_NAME = '{$tableName}'
        AND COLUMN_NAME = '{$columnName}'
);

SET @sql = IF(@col_exists = 0,
    'ALTER TABLE `{$tableName}` ADD COLUMN {$columnDef}',
    'SELECT "Column {$columnName} already exists in {$tableName}" AS message'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;


SQL;

        return $sql;
    }

    /**
     * Generate safe DROP COLUMN statement
     *
     * @param string $tableName Table name
     * @param string $columnName Column to drop
     * @return string SQL DROP COLUMN statement
     */
    private static function generateDropColumnSafe($tableName, $columnName) {
        $sql = <<<SQL
SET @col_exists = (
    SELECT COUNT(*)
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
        AND TABLE_NAME = '{$tableName}'
        AND COLUMN_NAME = '{$columnName}'
);

SET @sql = IF(@col_exists > 0,
    'ALTER TABLE `{$tableName}` DROP COLUMN `{$columnName}`',
    'SELECT "Column {$columnName} does not exist in {$tableName}" AS message'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;


SQL;

        return $sql;
    }

    /**
     * Compare two database schemas
     *
     * @param string $sourceDB Source database name
     * @param string $targetDB Target database name
     * @return array Differences
     */
    public static function compareSchemas($sourceDB, $targetDB) {
        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        // Get tables from source
        $conn->select_db($sourceDB);
        $sourceTablesResult = $conn->query("SHOW TABLES");
        $sourceTables = [];
        while ($row = $sourceTablesResult->fetch_array()) {
            $sourceTables[] = $row[0];
        }

        // Get tables from target
        $conn->select_db($targetDB);
        $targetTablesResult = $conn->query("SHOW TABLES");
        $targetTables = [];
        while ($row = $targetTablesResult->fetch_array()) {
            $targetTables[] = $row[0];
        }

        // Find missing tables
        $missingTables = array_diff($sourceTables, $targetTables);

        // Find extra tables (in target but not in source)
        $extraTables = array_diff($targetTables, $sourceTables);

        // Find common tables and compare columns
        $commonTables = array_intersect($sourceTables, $targetTables);
        $columnDifferences = [];

        foreach ($commonTables as $tableName) {
            $differences = self::compareTableColumns($tableName, $sourceDB, $targetDB, $conn);
            if (!empty($differences)) {
                $columnDifferences[$tableName] = $differences;
            }
        }

        return [
            'missing_tables' => array_values($missingTables),
            'extra_tables' => array_values($extraTables),
            'common_tables' => array_values($commonTables),
            'column_differences' => $columnDifferences
        ];
    }

    /**
     * Compare columns between same table in two databases
     *
     * @param string $tableName Table name
     * @param string $sourceDB Source database
     * @param string $targetDB Target database
     * @param mysqli $conn Database connection
     * @return array Column differences
     */
    private static function compareTableColumns($tableName, $sourceDB, $targetDB, $conn) {
        // Get source columns
        $conn->select_db($sourceDB);
        $sourceResult = $conn->query("DESCRIBE `{$tableName}`");
        $sourceColumns = [];
        while ($row = $sourceResult->fetch_assoc()) {
            $sourceColumns[$row['Field']] = $row;
        }

        // Get target columns
        $conn->select_db($targetDB);
        $targetResult = $conn->query("DESCRIBE `{$tableName}`");
        $targetColumns = [];
        while ($row = $targetResult->fetch_assoc()) {
            $targetColumns[$row['Field']] = $row;
        }

        // Find missing columns
        $missingColumns = array_diff_key($sourceColumns, $targetColumns);

        // Find extra columns
        $extraColumns = array_diff_key($targetColumns, $sourceColumns);

        // Find type mismatches
        $typeMismatches = [];
        foreach ($sourceColumns as $colName => $sourceCol) {
            if (isset($targetColumns[$colName])) {
                $targetCol = $targetColumns[$colName];
                if ($sourceCol['Type'] !== $targetCol['Type']) {
                    $typeMismatches[$colName] = [
                        'source_type' => $sourceCol['Type'],
                        'target_type' => $targetCol['Type']
                    ];
                }
            }
        }

        $differences = [];

        if (!empty($missingColumns)) {
            $differences['missing_columns'] = array_keys($missingColumns);
        }

        if (!empty($extraColumns)) {
            $differences['extra_columns'] = array_keys($extraColumns);
        }

        if (!empty($typeMismatches)) {
            $differences['type_mismatches'] = $typeMismatches;
        }

        return $differences;
    }

    /**
     * Execute a migration SQL file
     *
     * @param string $sqlFile Path to SQL file
     * @return array Execution results
     */
    public static function executeMigration($sqlFile) {
        if (!file_exists($sqlFile)) {
            throw new Exception("Migration file not found: {$sqlFile}");
        }

        $db = DatabaseHelper::getInstance();
        $conn = $db->getConnection();

        // Read SQL file
        $sqlContent = file_get_contents($sqlFile);

        // Split into statements (simple split by semicolon - may need refinement)
        $statements = array_filter(
            array_map('trim', explode(';', $sqlContent)),
            function($stmt) {
                return !empty($stmt) && !preg_match('/^--/', $stmt);
            }
        );

        $results = [
            'total_statements' => count($statements),
            'executed' => 0,
            'failed' => 0,
            'errors' => []
        ];

        // Execute each statement
        foreach ($statements as $index => $statement) {
            // Skip comments and empty statements
            if (empty(trim($statement)) || preg_match('/^--/', trim($statement))) {
                continue;
            }

            $result = $conn->multi_query($statement . ';');

            if ($result) {
                // Clear any results
                while ($conn->more_results()) {
                    $conn->next_result();
                    if ($res = $conn->store_result()) {
                        $res->free();
                    }
                }
                $results['executed']++;
            } else {
                $results['failed']++;
                $results['errors'][] = [
                    'statement' => substr($statement, 0, 100) . '...',
                    'error' => $conn->error
                ];
            }
        }

        // Log migration result
        self::logMigrationResult($sqlFile, $results);

        return $results;
    }

    /**
     * Log migration execution result
     *
     * @param string $sqlFile SQL file path
     * @param array $results Execution results
     */
    private static function logMigrationResult($sqlFile, $results) {
        $logFile = self::MIGRATION_DIR . 'migration_log.txt';
        $timestamp = date('Y-m-d H:i:s');

        $logEntry = sprintf(
            "[%s] Migration: %s | Executed: %d | Failed: %d | Errors: %d\n",
            $timestamp,
            basename($sqlFile),
            $results['executed'],
            $results['failed'],
            count($results['errors'])
        );

        file_put_contents($logFile, $logEntry, FILE_APPEND);

        // Also log via ErrorHandler
        if ($results['failed'] > 0) {
            ErrorHandler::logError('Migration had errors', $results);
        }
    }

    /**
     * Create rollback script for applied migrations
     *
     * @param array $appliedMigrations List of applied migration files
     * @return string Rollback script path
     */
    public static function createRollbackScript($appliedMigrations) {
        $timestamp = date('Y-m-d_H-i-s');
        $rollbackFile = self::MIGRATION_DIR . "rollback_{$timestamp}.sql";

        $sql = "-- ============================================\n";
        $sql .= "-- Rollback Script\n";
        $sql .= "-- Generated: " . date('Y-m-d H:i:s') . "\n";
        $sql .= "-- ============================================\n\n";

        $sql .= "-- WARNING: This will reverse the following migrations:\n";
        foreach ($appliedMigrations as $migration) {
            $sql .= "-- - {$migration}\n";
        }

        $sql .= "\n-- TODO: Implement specific rollback logic\n";
        $sql .= "-- This is a placeholder for now\n";

        file_put_contents($rollbackFile, $sql);

        return $rollbackFile;
    }
}
