<?php

declare(strict_types=1);

require_once __DIR__ . '/FormTemplate.php';
require_once __DIR__ . '/SchemaConventions.php';

/**
 * Class FormBuilder
 *
 * The main engine for PhoenixForms (Gate 7).
 * It generates complete form files based on database table metadata.
 */
class FormBuilder
{
    private string $tableName;
    private array $columns;
    private string $device;
    private string $templatePath;
    private FormTemplate $templateEngine;

    private const TEMPLATE_BASE_PATH = '/importer/forms/templates/';
    private const OUTPUT_PATH = '/importer/forms/generated/';

    /**
     * @param string $tableName The name of the database table.
     * @param array $columns Metadata for the table columns (from DatabaseHelper).
     * @param string $device The target device ('mobile', 'tablet', 'desktop').
     */
    public function __construct(string $tableName, array $columns, string $device = 'desktop')
    {
        $this->tableName = $tableName;
        $this->columns = $columns;
        $this->device = $device;
        $this->templatePath = $_SERVER['DOCUMENT_ROOT'] . self::TEMPLATE_BASE_PATH . $this->device . '/';
        $this->templateEngine = new FormTemplate();

        if (!is_dir($this->templatePath)) {
            throw new Exception("Template directory for device '{$this->device}' not found.");
        }

        $fullOutputPath = $_SERVER['DOCUMENT_ROOT'] . self::OUTPUT_PATH;
        if (!is_dir($fullOutputPath) && !mkdir($fullOutputPath, 0777, true)) {
            throw new Exception("Could not create output directory: " . $fullOutputPath);
        }
    }

    /**
     * Static factory to generate a form.
     *
     * @param string $tableName
     * @param array $columns
     * @param string $device
     * @return string The path to the generated file.
     * @throws Exception
     */
    public static function generate(string $tableName, array $columns, string $device): string
    {
        $builder = new self($tableName, $columns, $device);
        return $builder->build();
    }

    /**
     * Builds the form and writes it to a file.
     * @return string The path to the generated file.
     * @throws Exception
     */
    public function build(): string
    {
        $fieldsHtml = $this->buildFields();
        $actionsHtml = $this->buildActions();

        $formTitle = 'Edit ' . ucwords(str_replace('_', ' ', $this->tableName));

        $formTemplate = (new FormTemplate())
            ->loadTemplate($this->templatePath . 'form.php.tpl')
            ->replace('TABLE', $this->tableName)
            ->replace('FORM_TITLE', $formTitle)
            ->replace('DEVICE_CLASS', 'form-' . $this->device)
            ->replace('THEME', '-cyberpunk') // Use cyberpunk theme instead of basic theme
            ->replace('FIELDS', $fieldsHtml)
            ->replace('ACTIONS', $actionsHtml);

        $finalForm = $formTemplate->render();

        $outputFile = $_SERVER['DOCUMENT_ROOT'] . self::OUTPUT_PATH . "{$this->tableName}_{$this->device}.php";
        file_put_contents($outputFile, $finalForm);

        return $outputFile;
    }

    /**
     * Builds the HTML for all form fields.
     * @return string
     * @throws Exception
     */
    private function buildFields(): string
    {
        $allFieldsHtml = '';
        $fieldTemplate = (new FormTemplate())->loadTemplate($this->templatePath . 'field.php.tpl')->render();

        foreach ($this->columns as $column) {
            // Skip auto-managed fields (both Spanish canonical and English legacy)
            // Use SchemaConventions to properly identify standard audit fields
            if (SchemaConventions::isStandardField($column['name'])) {
                continue;
            }

            $currentField = new FormTemplate();
            $currentField->loadTemplate($this->templatePath . 'field.php.tpl');

            $label = ucwords(str_replace('_', ' ', $column['name']));
            $inputHtml = $this->mapDbToHtml($column);

            $currentField->replace('column_name', $column['name'])
                ->replace('Label', $label)
                ->replace('input', $inputHtml);

            $allFieldsHtml .= $currentField->render();
        }

        return $allFieldsHtml;
    }

    /**
     * Maps a database column type to an HTML input element.
     * @param array $column
     * @return string
     */
    private function mapDbToHtml(array $column): string
    {
        $name = $column['name'];
        $type = strtolower($column['type']);
        $length = $column['length'] ?? 255;
        $required = ($column['nullable'] === 'NO') ? 'required' : '';

        if (str_contains($type, 'tinyint(1)')) {
            return "<input type=\"checkbox\" id=\"{$name}\" name=\"{$name}\" value=\"1\" class=\"phoenix-input\">";
        }
        if (str_contains($type, 'int')) {
            return "<input type=\"number\" id=\"{$name}\" name=\"{$name}\" class=\"phoenix-input\" maxlength=\"{$length}\" {$required}>";
        }
        if (str_contains($type, 'datetime') || str_contains($type, 'timestamp')) {
            return "<input type=\"datetime-local\" id=\"{$name}\" name=\"{$name}\" class=\"phoenix-input\" maxlength=\"255\" {$required}>";
        }
        if (str_contains($type, 'text')) {
            return "<textarea id=\"{$name}\" name=\"{$name}\" class=\"phoenix-input\" rows=\"5\" maxlength=\"{$length}\" {$required}></textarea>";
        }
        if (str_contains($type, 'enum')) {
            preg_match_all("/'([^']+)'/", $column['type'], $matches);
            $options = $matches[1];
            $optionsHtml = '';
            foreach ($options as $option) {
                $optionsHtml .= "<option value=\"{$option}\">" . htmlspecialchars($option) . "</option>";
            }
            return "<select id=\"{$name}\" name=\"{$name}\" class=\"phoenix-input\" maxlength=\"{$length}\" {$required}>{$optionsHtml}</select>";
        }

        // Default to varchar/text
        return "<input type=\"text\" id=\"{$name}\" name=\"{$name}\" class=\"phoenix-input\" maxlength=\"{$length}\" {$required}>";
    }

    private function buildActions(): string
    {
        return (new FormTemplate())
            ->loadTemplate($this->templatePath . 'actions.php.tpl')
            ->render();
    }
}
