<?php
/**
 * RagQuery - The RAG Query Interface (with Context7 Integration)
 * 
 * Queries the RAG index to retrieve relevant context for LLM requests.
 * Combines local documentation with remote Context7 knowledge.
 * 
 * Usage:
 *   $context = RagQuery::getContext("How do I add a new column to the import_logs table?");
 *   echo $context;
 * 
 *   // With Context7 augmentation:
 *   $context = RagQuery::getContext("How to read Excel files in PHP?", ['useContext7' => true]);
 *   echo $context;
 */

require_once __DIR__ . '/../config.php';

class RagQuery
{
    const VERSION = '2.0.0';
    const INDEX_DIR = __DIR__ . '/../docs/rag';
    
    /**
     * Whether to use Context7 integration
     */
    const USE_CONTEXT7 = true;
    
    /**
     * Keywords for each documentation section
     */
    private static $keywords = [
        'codebase' => [
            'class', 'function', 'method', 'object', 'property', 'constant',
            'databasehelper', 'schemadetector', 'importlogger', 'errorhandler',
            'datavalidator', 'datacleaner', 'tableoperations', 'schemaconventions',
            'config', 'arrival', 'insert', 'upload', 'viewer', 'dashboard',
        ],
        'database' => [
            'table', 'column', 'schema', 'database', 'primary', 'foreign', 'key',
            'index', 'constraint', 'import_logs', 'import_schema_logs',
            'audit', 'field', 'type', 'convention', 'naming',
        ],
        'api' => [
            'endpoint', 'request', 'response', 'parameter', 'query',
            'getdatabases', 'gettables', 'gettablerows', 'gettablenames',
            'getimportschema', 'getlastimportschema', 'getprogress',
            'tableoperations', 'copy', 'move', 'rename', 'delete',
        ],
    ];
    
    /**
     * Context7 libraries to query based on query content
     */
    private static $context7LibraryMap = [
        'excel' => 'phpspreadsheet/phpspreadsheet',
        'spreadsheet' => 'phpspreadsheet/phpspreadsheet',
        'xlsx' => 'phpspreadsheet/phpspreadsheet',
        'csv' => 'phpspreadsheet/phpspreadsheet',
        'database' => 'mysqli/mysqli',
        'query' => 'mysqli/mysqli',
        'sql' => 'mysqli/mysqli',
        'php' => 'php/php',
        'http' => 'php/php',
    ];
    
    /**
     * Get context for a natural language query
     * Combines local documentation with Context7 remote knowledge
     * 
     * @param string $query Natural language query
     * @param array $options Query options:
     *   - useContext7 (bool): Whether to use Context7 augmentation (default: true)
     *   - maxResults (int): Max Context7 results (default: 3)
     *   - libraries (array): Libraries to query
     * @return string Formatted context for LLM
     */
    public static function getContext($query, $options = []) {
        // Determine which sections are relevant
        $sections = self::determineRelevantSections($query);
        
        // Gather context from local documentation
        $localContext = self::gatherContext($sections, $options);
        
        // Optionally augment with Context7 remote knowledge
        $useContext7 = isset($options['useContext7']) ? $options['useContext7'] : self::USE_CONTEXT7;
        
        if ($useContext7) {
            $context7Context = self::getContext7Context($query, $options);
            if ($context7Context) {
                return $localContext . "\n\n" . $context7Context;
            }
        }
        
        return $localContext;
    }
    
    /**
     * Get Context7 remote knowledge context
     * 
     * @param string $query Natural language query
     * @param array $options Query options
     * @return string Context7 context or empty string
     */
    private static function getContext7Context($query, $options) {
        // Determine which libraries to query based on query content
        $libraries = isset($options['libraries']) ? $options['libraries'] : self::determineLibraries($query);
        
        if (empty($libraries)) {
            return '';
        }
        
        // Load Context7 retriever
        require_once __DIR__ . '/Context7KnowledgeRetriever.php';
        
        // Query each library
        $context = '';
        foreach ($libraries as $library) {
            $docs = Context7KnowledgeRetriever::queryDocs(
                $library,
                $query,
                $options
            );
            if ($docs) {
                $context .= "\n\n" . $docs;
            }
        }
        
        return $context;
    }
    
    /**
     * Determine which Context7 libraries to query based on query content
     * 
     * @param string $query Query string
     * @return array Library names to query
     */
    private static function determineLibraries($query) {
        $queryLower = strtolower($query);
        $libraries = [];
        
        foreach (self::$context7LibraryMap as $keyword => $library) {
            if (strpos($queryLower, $keyword) !== false) {
                if (!in_array($library, $libraries)) {
                    $libraries[] = $library;
                }
            }
        }
        
        // Always include PHP as base library
        if (empty($libraries)) {
            $libraries = ['php/php'];
        }
        
        return $libraries;
    }
    
    /**
     * Get context as JSON for structured access
     * 
     * @param string $query Natural language query
     * @return array Context data structure
     */
    public static function getContextAsData($query) {
        $sections = self::determineRelevantSections($query);
        $data = [];
        
        foreach ($sections as $section) {
            $data[$section] = self::loadSection($section);
        }
        
        return $data;
    }
    
    /**
     * Determine which sections are relevant for the query
     */
    private static function determineRelevantSections($query) {
        $queryLower = strtolower($query);
        $sections = [];
        
        // Check each section for keyword matches
        foreach (self::$keywords as $section => $keywords) {
            foreach ($keywords as $keyword) {
                if (strpos($queryLower, strtolower($keyword)) !== false) {
                    $sections[] = $section;
                    break;
                }
            }
        }
        
        // If no specific section matched, include all
        if (empty($sections)) {
            $sections = ['codebase', 'database', 'api'];
        }
        
        return $sections;
    }
    
    /**
     * Gather context from relevant sections
     */
    private static function gatherContext($sections, $options) {
        $context = "";
        
        foreach ($sections as $section) {
            $sectionContent = self::loadSection($section);
            if ($sectionContent) {
                $context .= self::formatSection($section, $sectionContent);
            }
        }
        
        return $context;
    }
    
    /**
     * Load all documentation for a section
     */
    private static function loadSection($section) {
        $sectionDir = self::INDEX_DIR . '/' . $section;
        if (!is_dir($sectionDir)) {
            return [];
        }
        
        $files = glob($sectionDir . '/*.md');
        $docs = [];
        
        foreach ($files as $file) {
            $filename = basename($file);
            $content = file_get_contents($file);
            $docs[$filename] = $content;
        }
        
        return $docs;
    }
    
    /**
     * Format a section for LLM context
     */
    private static function formatSection($section, $files) {
        $output = "";
        $output .= str_repeat('#', 2) . " Section: " . ucfirst($section) . "\n\n";
        
        foreach ($files as $filename => $content) {
            $output .= str_repeat('#', 3) . " File: $filename\n\n";
            $output .= $content . "\n";
        }
        
        return $output;
    }
    
    /**
     * Search for specific pattern in documentation
     */
    public static function search($pattern) {
        $results = [];
        
        // Search all markdown files
        $allFiles = glob(self::INDEX_DIR . '/*/*/*.md');
        foreach ($allFiles as $file) {
            $content = file_get_contents($file);
            if (preg_match($pattern, $content, $matches)) {
                $results[basename($file)] = [
                    'path' => $file,
                    'matches' => $matches
                ];
            }
        }
        
        return $results;
    }
    
    /**
     * Get class documentation
     */
    public static function getClassDoc($className) {
        $classFile = self::INDEX_DIR . '/codebase/lib/' . $className . '.php.md';
        if (is_file($classFile)) {
            return file_get_contents($classFile);
        }
        return null;
    }
    
    /**
     * Get API endpoint documentation
     */
    public static function getApiDoc($endpointName) {
        $apiFile = self::INDEX_DIR . '/api/' . strtolower($endpointName) . '.md';
        if (is_file($apiFile)) {
            return file_get_contents($apiFile);
        }
        return null;
    }
    
    /**
     * Get database conventions documentation
     */
    public static function getDatabaseConventions() {
        $convFile = self::INDEX_DIR . '/database/conventions.md';
        if (is_file($convFile)) {
            return file_get_contents($convFile);
        }
        return null;
    }
    
    /**
     * Get import logger schema documentation
     */
    public static function getImportLoggerSchema() {
        $schemaFile = self::INDEX_DIR . '/database/import_logger.md';
        if (is_file($schemaFile)) {
            return file_get_contents($schemaFile);
        }
        return null;
    }
    
    /**
     * Get full index structure
     */
    public static function getIndexStructure() {
        $readmeFile = self::INDEX_DIR . '/README.md';
        if (is_file($readmeFile)) {
            return file_get_contents($readmeFile);
        }
        return null;
    }
    
    /**
     * Rebuild the index (run the indexer)
     */
    public static function rebuildIndex() {
        require_once __DIR__ . '/RagIndexer.php';
        RagIndexer::run();
    }
    
    /**
     * Get version info
     */
    public static function getVersion() {
        return [
            'version' => self::VERSION,
            'index_dir' => self::INDEX_DIR,
            'context7_enabled' => self::USE_CONTEXT7
        ];
    }
    
    /**
     * Get Context7 status
     */
    public static function getContext7Status() {
        return [
            'enabled' => self::USE_CONTEXT7,
            'library_map' => self::$context7LibraryMap
        ];
    }
}
