/**
 * TableNameMemory - Client-side table name memory with frequency tracking
 * Persists to localStorage for offline access and quick loading
 */
(function(global) {
    'use strict';
    
    var STORAGE_KEY = 'importer_table_names';
    var MAX_ENTRIES = 50;
    
    // Load from localStorage
    function load() {
        try {
            var stored = localStorage.getItem(STORAGE_KEY);
            return stored ? JSON.parse(stored) : {};
        } catch (e) {
            console.warn('TableNameMemory: Failed to load from localStorage', e);
            return {};
        }
    }
    
    // Save to localStorage
    function save(data) {
        try {
            localStorage.setItem(STORAGE_KEY, JSON.stringify(data));
        } catch (e) {
            console.warn('TableNameMemory: Failed to save to localStorage', e);
        }
    }
    
    // Record usage of a table name
    function recordUsage(tableName) {
        if (!tableName || typeof tableName !== 'string' || tableName.trim() === '') return;
        
        var data = load();
        var normalizedName = tableName.toLowerCase().trim();
        
        if (!data[normalizedName]) {
            data[normalizedName] = {
                name: tableName,
                count: 0,
                firstUsed: Date.now(),
                lastUsed: Date.now()
            };
        }
        
        data[normalizedName].count++;
        data[normalizedName].lastUsed = Date.now();
        
        // Keep only most used entries
        var sorted = Object.values(data)
            .sort(function(a, b) { 
                return b.count - a.count || b.lastUsed - a.lastUsed; 
            })
            .slice(0, MAX_ENTRIES);
        
        // Rebuild object maintaining order
        var newData = {};
        sorted.forEach(function(item) {
            newData[item.name.toLowerCase()] = item;
        });
        
        save(newData);
    }
    
    // Get suggestions sorted by frequency
    function getSuggestions(limit) {
        limit = limit || 10;
        var data = load();
        return Object.values(data)
            .sort(function(a, b) { 
                return b.count - a.count || b.lastUsed - a.lastUsed; 
            })
            .slice(0, limit)
            .map(function(item) { return item.name; });
    }
    
    // Filter suggestions by search term
    function search(query, limit) {
        if (!query) return getSuggestions(limit);
        
        var suggestions = getSuggestions(50);
        var lowerQuery = query.toLowerCase();
        
        return suggestions
            .filter(function(name) { return name.toLowerCase().includes(lowerQuery); })
            .slice(0, limit || 10);
    }
    
    // Get most recently used
    function getRecent(limit) {
        limit = limit || 5;
        var data = load();
        return Object.values(data)
            .sort(function(a, b) { return b.lastUsed - a.lastUsed; })
            .slice(0, limit)
            .map(function(item) { return item.name; });
    }
    
    // Get most frequently used
    function getFrequent(limit) {
        return getSuggestions(limit);
    }
    
    // Clear all memory
    function clear() {
        localStorage.removeItem(STORAGE_KEY);
    }
    
    // Export for testing/debugging
    function getAll() {
        return load();
    }
    
    // TableNameMemory API
    global.TableNameMemory = {
        recordUsage: recordUsage,
        getSuggestions: getSuggestions,
        search: search,
        getRecent: getRecent,
        getFrequent: getFrequent,
        clear: clear,
        load: load,
        getAll: getAll
    };
    
})(typeof window !== 'undefined' ? window : this);
